package com.xdja.base.ucm.systemconfig.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.stereotype.Repository;

import com.xdja.base.common.dao.BaseDao;
import com.xdja.base.ucm.systemconfig.bean.QueryConfigBean;
import com.xdja.base.ucm.systemconfig.entity.Client;
import com.xdja.base.util.DateUtils;
import com.xdja.base.util.page.SQLUtil;

/**
 * @ProjectName：ucm
 * @ClassName：SystemDataDao
 * @Description：子系统配置数据管理DAO层实现
 * @author: mayanpei
 * @date: 2014-8-20 上午11:33:41
 */
@Repository
public class ConfigDao extends BaseDao {

	private Logger logger = LoggerFactory.getLogger(ConfigDao.class);
	
	/**
	 * @Description：查询子系统配置数据总数
	 * @author: mayanpei
	 * @date: 2014-8-20 上午11:33:34
	 * @param queryBean
	 *            查询bean
	 * @return
	 */
	public Integer queryConfigCount(QueryConfigBean queryBean) {
		StringBuffer sql = new StringBuffer("SELECT n_id,c_client_name,c_short_name, ");
		sql.append(" c_client_ip,c_client_port,n_user_id,c_note,c_add_time,c_update_time  ");
		sql.append(" FROM t_ucm_client WHERE 1=1 ");

		MapSqlParameterSource parameterSource = new MapSqlParameterSource();

		if (StringUtils.isNotBlank(queryBean.getSystemName())) {
			sql.append(" AND c_client_name LIKE :name ");
			parameterSource.addValue("name", "%" + queryBean.getSystemName() + "%");
		}
		if (StringUtils.isNotBlank(queryBean.getShortName())) {
			sql.append(" AND c_short_name LIKE :shortName ");
			parameterSource.addValue("shortName", "%" + queryBean.getShortName() + "%");
		}
		// 开始时间
		if (StringUtils.isNotBlank(queryBean.getStartDate())) {
			sql.append(" AND c_add_time >= :start");
			parameterSource.addValue("start", queryBean.getStartDate() + DateUtils.START);
		}

		// 结束时间
		if (StringUtils.isNotBlank(queryBean.getEndDate())) {
			sql.append(" AND c_add_time <= :end");
			parameterSource.addValue("end", queryBean.getEndDate() + DateUtils.END);
		}
		
		//非管理员用户
		if(1 != queryBean.getUserType()) {
			sql.append(" AND n_id in (SELECT n_client_id FROM t_ucm_user_client WHERE n_user_id = :userId) "); 
			parameterSource.addValue("userId", queryBean.getUserId());
		}
		
		//sql.append(" ORDER BY n_id DESC");
		SQLUtil util = new SQLUtil(sql.toString());
		return queryForInt(util.getCountSQL(), parameterSource);
	}

	/**
	 * @Description：查询子系统配置数据
	 * @author: mayanpei
	 * @date: 2014-8-20 上午11:34:14
	 * @param queryBean
	 *            查询bean
	 * @return
	 */
	public List<Map<String, Object>> queryConfigList(QueryConfigBean queryBean) {
		StringBuffer sql = new StringBuffer("SELECT n_id,c_client_name,c_short_name, ");
		sql.append(" c_client_ip,c_client_port,n_user_id,c_note,c_add_time,c_update_time  ");
		sql.append(" FROM t_ucm_client WHERE 1=1 ");
		MapSqlParameterSource parameterSource = new MapSqlParameterSource();

		if (StringUtils.isNotBlank(queryBean.getSystemName())) {
			sql.append(" AND c_client_name LIKE :name ");
			parameterSource.addValue("name", "%" + queryBean.getSystemName() + "%");
		}

		if (StringUtils.isNotBlank(queryBean.getShortName())) {
			sql.append(" AND c_short_name LIKE :shortName ");
			parameterSource.addValue("shortName", "%" + queryBean.getShortName() + "%");
		}

		// 开始时间
		if (StringUtils.isNotBlank(queryBean.getStartDate())) {
			sql.append(" AND c_add_time >= :start");
			parameterSource.addValue("start", queryBean.getStartDate() + DateUtils.START);
		}

		// 结束时间
		if (StringUtils.isNotBlank(queryBean.getEndDate())) {
			sql.append(" AND c_add_time <= :end");
			parameterSource.addValue("end", queryBean.getEndDate() + DateUtils.END);
		}
		
		//非管理员用户
		if(1 != queryBean.getUserType()) {
			sql.append(" AND n_id in (SELECT n_client_id FROM t_ucm_user_client WHERE n_user_id = :userId) "); 
			parameterSource.addValue("userId", queryBean.getUserId());
		}
		sql.append(" ORDER BY n_id DESC");

		SQLUtil util = new SQLUtil(sql.toString(), queryBean.getPageNo(), queryBean.getPageSize());
		return queryForList(util.getPageSql(), parameterSource);
	}

	/**
	 * @Description：查询所有子系统配置数据
	 * @author: mayanpei
	 * @date: 2014-8-20 上午11:34:14
	 * @param userId
	 *            用户ID
	 * @return List<Map<String, Object>>
	 */
	public List<Map<String, Object>> queryConfigList(Long userId) {
		StringBuffer sql = new StringBuffer();
		MapSqlParameterSource parameter = new MapSqlParameterSource();

		if (userId != null && userId > 0) {
			sql.append("SELECT c.n_id,c.c_client_name,c.c_short_name,c.c_client_ip, ");
			sql.append(" c.c_client_port,c.n_user_id,c.c_note,c.c_add_time,c.c_update_time  ");
			sql.append(" FROM t_ucm_user_client uc");
			sql.append(" LEFT JOIN t_ucm_user u ON uc.n_user_id = u.n_id");
			sql.append(" LEFT JOIN t_ucm_client c ON uc.n_client_id = c.n_id ");
			sql.append(" WHERE u.n_id =:userId");
			parameter.addValue("userId", userId);
		} else {
			sql.append("SELECT c.n_id,c.c_client_name,c.c_short_name,c.c_client_ip, ");
			sql.append(" c.c_client_port,c.n_user_id,c.c_note,c.c_add_time,c.c_update_time  ");
			sql.append(" FROM t_ucm_client c ORDER BY n_id asc");
		}

		logger.debug(sql.toString());
		
		return queryForList(sql.toString(), parameter);
	}

	/**
	 * @Description：根据ID查询子系统配置数据详情
	 * @author: mayanpei
	 * @date: 2014-8-20 上午11:34:14
	 * @param id
	 *            子系统配置id
	 * @return Map<String, Object>
	 */
	public Map<String, Object> queryConfigById(Long id) {
		StringBuffer sql = new StringBuffer("SELECT n_id,c_client_name,c_short_name, ");
		sql.append(" c_client_ip,c_client_port,n_user_id,c_note,c_add_time,c_update_time  ");
		sql.append(" FROM t_ucm_client  WHERE n_id=:id ");
		return queryForMap(sql.toString(), new MapSqlParameterSource().addValue("id", id));
	}

	/**
	 * @Description：添加子系统配置数据
	 * @author: mayanpei
	 * @date: 2014-8-20 下午3:35:24
	 * @param client
	 *            子系统
	 * @param userId
	 *            操作人ID
	 */
	public void addConfig(Client client, Long userId) {
		StringBuilder sql = new StringBuilder("INSERT INTO t_ucm_client(c_client_name,c_short_name,");
		sql.append("c_client_ip,c_client_port,n_user_id,c_note,c_add_time,c_update_time)");
		sql.append("VALUES(:clientName,:shortName,:ip,:port,:userId,:note,:time,:updatetime)");

		// 构造参数
		MapSqlParameterSource parameterSource = new MapSqlParameterSource();
		parameterSource.addValue("clientName", client.getClientName());
		parameterSource.addValue("shortName", client.getShortName().toLowerCase());
		parameterSource.addValue("ip", client.getClientIp());
		parameterSource.addValue("port", client.getClientPort());
		parameterSource.addValue("userId", userId);
		parameterSource.addValue("note", client.getNote());
		parameterSource.addValue("time", DateUtils.getToday());
		parameterSource.addValue("updatetime", DateUtils.getToday());

		executeSql(sql.toString(), parameterSource);
	}

	/**
	 * @Description：修改子系统配置数据
	 * @author: mayanpei
	 * @date: 2014-8-20 下午3:35:31
	 * @param client
	 *            子系统bean
	 */
	public void updateConfig(Client client) {
		StringBuilder sql = new StringBuilder();
		sql.append("UPDATE t_ucm_client SET c_client_name=:clientName,c_client_ip=:ip,");
		sql.append(" c_client_port=:port,c_note=:note,c_update_time=:time WHERE n_id=:id");

		// 构造参数
		MapSqlParameterSource parameterSource = new MapSqlParameterSource();
		parameterSource.addValue("clientName", client.getClientName());
		parameterSource.addValue("ip", client.getClientIp());
		parameterSource.addValue("port", client.getClientPort());
		parameterSource.addValue("note", client.getNote());
		parameterSource.addValue("time", DateUtils.getToday());
		parameterSource.addValue("id", client.getId());

		executeSql(sql.toString(), parameterSource);
	}

	/**
	 * @Description：删除子系统配置数据
	 * @author: mayanpei
	 * @date: 2014-8-20 下午3:35:07
	 * @param id
	 *            子系统参数配置ID
	 */
	public void deleteConfig(Long id) {
		Map<String, Object> map = this.queryConfigById(id);

		String dataSql = "DELETE FROM t_ucm_client_data WHERE c_data_client=:client";
		update(dataSql, new MapSqlParameterSource().addValue("client", map.get("c_short_name")));

		String delUserClientSql = "DELETE FROM t_ucm_user_client WHERE n_client_id=:id";
		executeSql(delUserClientSql, new MapSqlParameterSource().addValue("id", id));

		String deleteClientSql = "DELETE FROM t_ucm_client WHERE n_id=:id";
		executeSql(deleteClientSql, new MapSqlParameterSource().addValue("id", id));

	}

	/**
	 * @Description： 根据子系统名和简称查询数据
	 * @author: mayanpei
	 * @date: 2014-8-25 上午9:15:22
	 * @param name
	 *            子系统名称
	 * @param shortName
	 *            子系统简称
	 * @return
	 */
	public List<Map<String, Object>> queryConfigByName(String name, String shortName) {
		StringBuffer sql = new StringBuffer("SELECT n_id,c_client_name,c_short_name, ");
		sql.append(" c_client_ip,c_client_port,n_user_id,c_note,c_add_time,c_update_time  ");
		sql.append(" FROM t_ucm_client WHERE 1=1 ");
		MapSqlParameterSource param = new MapSqlParameterSource();

		// 系统名称
		if (StringUtils.isNotBlank(name)) {
			sql.append(" AND c_client_name = :name");
			param.addValue("name", name);
		}

		// 系统名称简称
		if (StringUtils.isNotBlank(shortName)) {
			sql.append(" AND c_short_name = :shortName");
			param.addValue("shortName", shortName.toLowerCase());
		}

		return queryForList(sql.toString(), param);
	}

	/**
	 * @Description： 根据子系统简称查询数据子系统详情
	 * @author: mayanpei
	 * @date: 2014-8-25 上午9:15:22
	 * @param shortName
	 *            系统简称
	 * @return Client
	 */
	public Client queryConfigByName(String shortName) {
		StringBuffer sql = new StringBuffer("SELECT n_id,c_client_name,c_short_name, ");
		sql.append(" c_client_ip,c_client_port,n_user_id,c_note,c_add_time,c_update_time  ");
		sql.append(" FROM t_ucm_client WHERE 1=1 ");
		MapSqlParameterSource param = new MapSqlParameterSource();

		// 系统名称简称
		if (StringUtils.isNotBlank(shortName)) {
			sql.append(" AND c_short_name = :shortName");
			param.addValue("shortName", shortName.toLowerCase());
		}

		RowMapper<Client> mapper = new RowMapper<Client>() {
			public Client mapRow(ResultSet rs, int rowNum) throws SQLException {
				Client client = new Client();
				client.setId(rs.getLong("n_id"));
				client.setClientName(rs.getString("c_client_name"));
				client.setShortName(rs.getString("c_short_name"));
				client.setClientIp(rs.getString("c_client_ip"));
				client.setClientPort(rs.getString("c_client_port"));
				client.setUserId(rs.getLong("n_user_id"));
				client.setAddTime(rs.getString("c_add_time"));
				client.setUpdateTime(rs.getString("c_update_time"));
				return client;
			}
		};
		return queryForObject(sql.toString(), param, mapper);
	}
	
	/**
	 * @Description： 根据子系统简称查询系统列表
	 * @author: mayanpei
	 * @date: 2014-8-25 上午9:15:22
	 * @param shortName
	 *            系统简称
	 * @return Client
	 */
	public List<Map<String, Object>> queryClientList() {
		StringBuilder sql = new StringBuilder("SELECT n_id,c_client_name,c_short_name,");
		sql.append(" c_client_ip,c_client_port,n_user_id,c_note,c_add_time,c_update_time ");
		sql.append(" FROM t_ucm_client ");
		return queryForList(sql.toString(), new MapSqlParameterSource());
	}

	/**
	 * 根据系统简称，查询所有系统静态配置参数
	 * 
	 * @author: mayanpei
	 * @date: 2014-9-4 上午11:32:44
	 * @param shortName
	 *            系统简称
	 * @return List<Map<String, Object>>
	 */
	public List<Map<String, Object>> queryConfigStaticData(String shortName) {
		StringBuilder sql = new StringBuilder();
		sql.append(" SELECT n_id,c_data_client,c_data_key,c_data_value,n_data_type,c_data_file");
		sql.append(" FROM t_ucm_client_data t ");
		sql.append(" WHERE c_data_client = :shortName");
		return queryForList(sql.toString(), new MapSqlParameterSource().addValue("shortName", shortName));
	}
}


