package com.xdja.base.ucm.user.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.stereotype.Repository;

import com.xdja.base.common.dao.BaseDao;
import com.xdja.base.system.entity.User;
import com.xdja.base.util.Constants;
import com.xdja.base.util.PasswordUtils;
import com.xdja.base.util.page.SQLUtil;

/**
 * @ProjectName：ucm
 * @ClassName：UserDao
 * @Description：用户管理DAO层实现
 * @author: mayanpei
 * @date: 2014-8-14 下午7:41:13
 */
@Repository
public class UserDao extends BaseDao {

	/**
	 * @Description：查询用户
	 * @author: mayanpei
	 * @date: 2014-8-14 下午7:25:42
	 * @param adminId
	 *            管理员ID
	 * @param loginName
	 *            登录名
	 * @param name
	 *            管理员名称
	 * @param pageSize
	 *            一页显示条数
	 * @param pageNo
	 *            当前页
	 * @return Pagination
	 */
	public List<Map<String, Object>> queryUserList(Long adminId, String loginName, String name, Integer pageSize,
			Integer pageNo) {
		StringBuilder sql = new StringBuilder();
		sql.append(" SELECT n_id,c_login_name,c_name,c_mobile,c_email,c_note ");
		sql.append(" FROM t_ucm_user WHERE n_id<>:id AND n_type<>1");

		MapSqlParameterSource parameterSource = new MapSqlParameterSource();
		parameterSource.addValue("id", adminId);

		// 登录用户名
		if (StringUtils.isNotBlank(loginName)) {
			sql.append(" AND c_login_name LIKE :loginName");
			parameterSource.addValue("loginName", "%" + loginName + "%");
		}

		// 姓名
		if (StringUtils.isNotBlank(name)) {
			sql.append(" AND c_name LIKE :name");
			parameterSource.addValue("name", "%" + name + "%");
		}

		sql.append(" ORDER BY n_id DESC");
		SQLUtil util = new SQLUtil(sql.toString(), pageNo, pageSize);
		return queryForList(util.getPageSql(), parameterSource);
	}

	/**
	 * @Description：查询用户总数
	 * @author: mayanpei
	 * @date: 2014-8-14 下午7:25:42
	 * @param userId
	 *            管理员ID
	 * @param loginName
	 *            登录名
	 * @param name
	 *            管理员名称
	 * @return Pagination
	 */
	public int queryUserCount(Long userId, String loginName, String name) {
		StringBuilder sql = new StringBuilder();
		sql.append(" SELECT n_id,c_login_name,c_name,c_note ");
		sql.append(" FROM t_ucm_user WHERE n_id <> :userId  AND n_type<>1");
		MapSqlParameterSource parameterSource = new MapSqlParameterSource();
		parameterSource.addValue("userId", userId);

		// 登录帐户名
		if (StringUtils.isNotBlank(loginName)) {
			sql.append(" AND c_login_name LIKE :loginName");
			parameterSource.addValue("loginName", "%" + loginName + "%");
		}

		// 用户姓名
		if (StringUtils.isNotBlank(name)) {
			sql.append(" AND c_name LIKE :name");
			parameterSource.addValue("name", "%" + name + " % ");

		}
		sql.append(" ORDER BY n_id DESC");
		SQLUtil util = new SQLUtil(sql.toString());
		return queryForInt(util.getCountSQL(), parameterSource);
	}

	/**
	 * @Description：查询用户信息
	 * @author: mayanpei
	 * @date: 2014-8-18 下午3:18:32
	 * @param loginName
	 *            用户名
	 * @return
	 */
	public List<Map<String, Object>> queryUserExsit(String loginName) {
		String sql = "SELECT n_id,c_login_name FROM t_ucm_user WHERE c_login_name = :name";
		return queryForList(sql, new MapSqlParameterSource().addValue("name", loginName));
	}

	/**
	 * @Description：保存用户
	 * @author: mayanpei
	 * @date: 2014-8-14 下午7:26:35
	 * @param user
	 *            用户
	 * @return :User
	 */
	public User saveUser(User user) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT MAX(n_id)+1 FROM t_ucm_user");
		Long userId = queryForLong(sql.toString(), new MapSqlParameterSource());

		sql = new StringBuilder();
		sql.append("INSERT INTO t_ucm_user (n_id,c_login_name,c_name,n_type,c_password,");
		sql.append(" 	c_mobile,c_email,n_status,c_note)");
		sql.append(" VALUES(:id,:loginName,:name,:type,:pwd,:mobile,:email,:status,:note)");

		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("id", userId);
		sqlParam.addValue("loginName", user.getLoginName());
		sqlParam.addValue("name", user.getName());
		sqlParam.addValue("type", Constants.USUAL_ADMINISTRATOR);
		sqlParam.addValue("pwd", PasswordUtils.encodePassword(Constants.USER_PASSWORD, user.getLoginName()));
		sqlParam.addValue("mobile", user.getMobile());
		sqlParam.addValue("email", user.getEmail());
		sqlParam.addValue("status", Constants.USER_STAUT_NORMAL);
		sqlParam.addValue("note", user.getNote());

		user.setId(userId);
		update(sql.toString(), sqlParam);
		sql = null;
		return user;
	}

	/**
	 * @Description：根据用户ID查询用户信息
	 * @author: mayanpei
	 * @date: 2014-8-13 下午4:41:09
	 * @param userId
	 *            用户ID
	 * @return
	 */
	public List<Map<String, Object>> queryUserByUserId(Long userId) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT n_id,c_login_name,c_name,c_password,c_mobile,c_email,n_status,c_note ");
		sql.append("FROM t_ucm_user WHERE n_id=:id");
		return queryForList(sql.toString(), new MapSqlParameterSource().addValue("id", userId));
	}

	/**
	 * @Description：更新用户
	 * @author: mayanpei
	 * @date: 2014-8-14 下午7:26:35
	 * @param user
	 *            用户
	 * @param roleIds
	 *            角色ID
	 */
	public void updateUser(User user, String roleIds) {
		StringBuilder sql = new StringBuilder();
		sql.append(" UPDATE t_ucm_user SET c_name =:name ,c_mobile=:mobile,c_email=:email,c_note=:note");
		sql.append(" WHERE n_id=:id");
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("name", user.getName());
		sqlParam.addValue("mobile", user.getMobile());
		sqlParam.addValue("email", user.getEmail());
		sqlParam.addValue("note", user.getNote());
		sqlParam.addValue("id", user.getId());
		update(sql.toString(), sqlParam);
		saveUserRole(user.getId(), Arrays.asList(StringUtils.split(roleIds, ",")));
		sql = null;
	}

	/**
	 * @Description：删除用户
	 * @author: mayanpei
	 * @date: 2014-8-14 下午7:26:58
	 * @param userId
	 *            用户ID
	 */
	public void deleteUser(Long userId) {
		String sql = "DELETE FROM t_ucm_user WHERE n_id=:id";
		executeSql(sql, new MapSqlParameterSource().addValue("id", userId));
		sql = null;
	}

	/**
	 * @Description：初始化用户密码
	 * @author: mayanpei
	 * @date: 2014-8-14 下午7:27:10
	 * @param userId
	 *            用户ID
	 */
	public void resetPassword(Long userId) {
		String sql = "UPDATE t_ucm_user set c_password=:pwd WHERE n_id=id";
		MapSqlParameterSource parameterSource = new MapSqlParameterSource();
		parameterSource.addValue("id", userId);
		parameterSource.addValue("pwd", Constants.USER_PASSWORD);
		executeSql(sql, parameterSource);
		sql = null;
	}

	/**
	 * @Description：查询管理员拥有的角色
	 * @author: mayanpei
	 * @date: 2014-7-18 上午9:51:14
	 * @param userId
	 *            管理员ID
	 * @return List<Map<String, Object>>
	 */
	public List<Map<String, Object>> queryUserRole(Long userId) {
		StringBuilder querySql = new StringBuilder();
		querySql.append(" SELECT n_id AS id,n_role_id roleId,n_user_id AS userId ");
		querySql.append(" FROM t_ucm_user_role");
		querySql.append(" WHERE n_user_id = :adminId ");
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("adminId", userId);
		return queryForList(querySql.toString(), sqlParam);
	}

	/**
	 * @Description：保存管理员角色
	 * @author: mayanpei
	 * @date: 2014-7-18 上午9:51:03
	 * @param userId
	 *            管理员ID
	 * @param roleIds
	 *            角色ID
	 */
	public void saveUserRole(Long userId, List<String> roleIds) {
		if (roleIds != null && roleIds.size() > 0) {
			String delSQL = "DELETE FROM t_ucm_user_role WHERE n_user_id = :userId";
			MapSqlParameterSource mapSqlParameterSource = new MapSqlParameterSource();
			mapSqlParameterSource.addValue("userId", userId);
			update(delSQL, mapSqlParameterSource);

			StringBuilder saveSQL = new StringBuilder();
			saveSQL.append("INSERT INTO t_ucm_user_role(n_user_id,n_role_id) ");
			saveSQL.append("VALUES(:userId,:roleId)");

			MapSqlParameterSource[] parameter = new MapSqlParameterSource[roleIds.size()];
			for (int i = 0; i < roleIds.size(); i++) {

				MapSqlParameterSource parameterSource = new MapSqlParameterSource();
				parameterSource.addValue("userId", userId);
				parameterSource.addValue("roleId", roleIds.get(i));
				parameter[i] = parameterSource;

			}

			// 批量保存用户角色信息
			addOrUpdate(saveSQL.toString(), parameter);

			delSQL = null;
			saveSQL = null;
		}
	}

	/**
	 * @Description：根据用户ID查询用户信息
	 * @author: mayanpei
	 * @date: 2014-8-13 下午4:41:09
	 * @param userId
	 *            用户ID
	 * @return
	 */
	public User queryUserById(Long userId) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT n_id,c_login_name,c_name,c_password,c_mobile,c_email,n_status,c_note ");
		sql.append("FROM t_ucm_user where n_id=:id");
		RowMapper<User> mapper = new RowMapper<User>() {
			public User mapRow(ResultSet rs, int rowNum) throws SQLException {
				User user = new User();
				user.setId(rs.getLong("n_id"));
				user.setLoginName(rs.getString("c_login_name"));
				user.setName(rs.getString("c_name"));
				user.setMobile(rs.getString("c_mobile"));
				user.setEmail(rs.getString("c_email"));
				user.setStatus(rs.getInt("n_status"));
				user.setPassword(rs.getString("c_password"));
				user.setStatus(rs.getInt("n_status"));
				user.setNote(rs.getString("c_note"));
				return user;
			}
		};
		User user = queryForObject(sql.toString(), new MapSqlParameterSource().addValue("id", userId), mapper);
		return user;
	}

	/**
	 * 修改密码
	 * 
	 * @author: mayanpei
	 * @date: 2014-8-19 下午3:11:20
	 * @param userId
	 *            用户ID
	 * @param password
	 *            密码
	 */
	public void updatePassword(Long userId, String password) {
		String sql = "UPDATE t_ucm_user SET c_password=:pwd WHERE n_id=:id";
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("pwd", password);
		sqlParam.addValue("id", userId);
		executeSql(sql, sqlParam);
		sql = null;
	}

	/**
	 * @Description：根据用户ID，查询用户所管理的子系统
	 * @author: mayanpei
	 * @date: 2014-9-1 上午9:06:48
	 * @param userId
	 * @return List<Map<String, Object>>
	 */
	public List<Map<String, Object>> queryManageChildSystem(Long userId) {
		StringBuilder sql = new StringBuilder();
//		sql.append(" SELECT u.n_id AS userId,c.n_id AS clientId");
//		sql.append(" FROM t_ucm_user_client uc ");
//		sql.append(" LEFT JOIN t_ucm_user u ON uc.n_user_id = u.n_id");
//		sql.append(" LEFT JOIN t_ucm_client c ON uc.n_client_id = c.n_id");
		
		sql.append(" SELECT n_user_id AS userId, n_client_id AS clientId");
		sql.append(" FROM t_ucm_user_client  ");
		sql.append(" WHERE n_user_id = :userId  ");
		logger.debug(sql.toString());
		
		return queryForList(sql.toString(), new MapSqlParameterSource().addValue("userId", userId));
	}

	/**
	 * @Description：保存用户可管理的子系统
	 * @author: mayanpei
	 * @date: 2014-9-1 上午9:27:38
	 * @param userId
	 * @param childSystemId
	 */
	public void addManageChildSystem(Long userId, String childSystemId) {
		String delSql = "DELETE FROM t_ucm_user_client WHERE n_user_id = :id";
		executeSql(delSql, new MapSqlParameterSource().addValue("id", userId));

		StringBuilder sqlBuilder = new StringBuilder();
		sqlBuilder.append("INSERT INTO t_ucm_user_client(n_user_id,n_client_id) VALUES(:userId,:childSystemId)");

		String[] childId = StringUtils.split(childSystemId, ",");
		MapSqlParameterSource[] parameter = new MapSqlParameterSource[childId.length];
		if (childId.length > 0) {

			for (int i = 0; i < childId.length; i++) {
				MapSqlParameterSource sqlParam = new MapSqlParameterSource();
				sqlParam.addValue("userId", userId);
				sqlParam.addValue("childSystemId", childId[i]);
				parameter[i] = sqlParam;
			}

			addOrUpdate(sqlBuilder.toString(), parameter);

		}
		delSql = null;
		sqlBuilder = null;
	}
}
