package com.xdja.update;

import android.content.Context;
import android.os.AsyncTask;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;

import com.xdja.update.bean.CheckResult;
import com.xdja.update.utils.EncryptUtils;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.net.HttpURLConnection;
import java.net.URL;

/**
 * <b>Description: 文件下载任务</b>
 * Created by <a href="mailto:fjd@xdja.com">fanjiandong</a> on 2017/11/1 16:33.
 */
public class DownloadTask extends AsyncTask<String, Long, Boolean> {

    /**
     * The constant TAG.
     */
    public static final String TAG = DownloadTask.class.getSimpleName();
    /**
     * The constant BUFFER_UNIT.
     */
    public static final int BUFFER_UNIT = 1024 * 512;
    /**
     * The constant DEFAULT_BOUNDARY.
     */
    public static final String DEFAULT_BOUNDARY = "----7d4261b4f3";

    private Context context;
    private CheckResult.Update.File file;
    private DownloadCallback callback;

    private Handler mainHandler;

    private boolean isStoped = false;

    /**
     * Instantiates a new Download task.
     *
     * @param context  the context
     * @param file     the file
     * @param callback the callback
     */
    public DownloadTask(@NonNull Context context,
                        @NonNull CheckResult.Update.File file,
                        @NonNull DownloadCallback callback) {
        this.context = context.getApplicationContext();
        this.file = file;
        this.callback = callback;
        this.mainHandler = new Handler(Looper.getMainLooper());
    }

    /**
     * Stop.
     */
    public synchronized void stop() {
        isStoped = true;
    }

    @Override
    protected void onPreExecute() {
        super.onPreExecute();
        this.callback.onStart(this.file.getFileId());
    }

    @Override
    protected Boolean doInBackground(String... params) {
        //params数组包含baseUrl和票据
        final String fileId = file.getFileId();
        String billInfo = params[1];

        if (TextUtils.isEmpty(fileId)) {
            Log.d(TAG, "============fileId为空================");
            runOnMainThread(new Runnable() {
                @Override
                public void run() {
                    callback.onError(fileId, "fileId为空", null);
                }
            });
            return false;
        }

        RandomAccessFile randomAccessFile = null;
        InputStream is;
        File targetFile = new File(context.getFilesDir() + "/" + fileId + ".apk");

        try {
            long offset = getOffset(targetFile);
            if (offset >= file.getfSize() - 1) {
                Log.d(TAG, "=======文件已下载完成======");
                if (verifyFile(targetFile, file.getCheckCode())) {
                    return true;
                }
                runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        callback.onError(fileId, "文件校验失败", null);
                    }
                });
                return false;
            }
            long endIndex = getEndIndex(offset);

            randomAccessFile = new RandomAccessFile(targetFile, "rws");
            randomAccessFile.seek(offset);
            StringBuffer sb = getMultipartBody(fileId, billInfo);

            URL url = new URL(params[0] + "/httpgetfile.do?");
            //while (endIndex <= file.getfSize() - 1 && !isStoped) {
            while (!isStoped) {
                HttpURLConnection conn = getHttpURLConnection(offset, endIndex, sb, url);
                final int responseCode = conn.getResponseCode();
                if (responseCode == HttpURLConnection.HTTP_OK
                        || responseCode == HttpURLConnection.HTTP_PARTIAL) {
                    is = conn.getInputStream();
                    if (is != null) {
                        appendStream(randomAccessFile, is);
                        publishProgress(endIndex + 1);

                        offset = endIndex + 1;
                        if (offset >= file.getfSize() - 1) {
                            Log.d(TAG, "=======文件已下载完成======");
                            if (verifyFile(targetFile, file.getCheckCode())) {
                                return true;
                            }
                            runOnMainThread(new Runnable() {
                                @Override
                                public void run() {
                                    callback.onError(fileId, "文件校验失败", null);
                                }
                            });
                            return false;
                        }
                        endIndex = getEndIndex(offset);
                        randomAccessFile.seek(offset);
                    } else {
                        Log.d(TAG, "============文件下载出错================");
                        runOnMainThread(new Runnable() {
                            @Override
                            public void run() {
                                callback.onError(fileId, "文件下载出错", null);
                            }
                        });
                        return false;
                    }
                } else {
                    Log.d(TAG, "============服务器响应错误，响应码：" + responseCode + "================");
                    runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            callback.onError(fileId, "服务器响应错误，响应码：" + responseCode, null);
                        }
                    });
                    return false;
                }
            }
            runOnMainThread(new Runnable() {
                @Override
                public void run() {
                    callback.onStop(file.getfName());
                }
            });
            return false;
        } catch (final IOException e) {
            Log.d(TAG, "=============\r\n文件下载出错 : " + e.getMessage() + "\r\n==============");
            runOnMainThread(new Runnable() {
                @Override
                public void run() {
                    callback.onError(fileId, "文件下载出错 : " + e.getMessage(), e);
                }
            });
            return false;
        } finally {
            try {
                if (randomAccessFile != null) {
                    randomAccessFile.close();
                }
            } catch (IOException e) {

            }
        }
    }

    @Override
    protected void onProgressUpdate(Long... values) {
        super.onProgressUpdate(values);
        callback.onProgress(file.getFileId(), file.getfSize(), values[0]);
    }

    @Override
    protected void onPostExecute(Boolean result) {
        super.onPostExecute(result);
        if (result) {
            callback.onComplete(file.getFileId());
        }
    }

    private void appendStream(RandomAccessFile randomAccessFile, InputStream is) throws IOException {

        byte[] buffer = new byte[BUFFER_UNIT];
        int count = -1;
        while ((count = is.read(buffer, 0, BUFFER_UNIT)) != -1) {
            randomAccessFile.write(buffer, 0, count);
        }
        is.close();
    }

    @NonNull
    private HttpURLConnection getHttpURLConnection(long offset, long endIndex, StringBuffer sb, URL url) throws IOException {
        HttpURLConnection conn = (HttpURLConnection) url.openConnection();
        conn.setDoInput(true);
        conn.setDoOutput(true);
        conn.setConnectTimeout(15 * 1000);
        conn.setReadTimeout(15 * 1000);
        conn.setRequestProperty("Content-Type", "multipart/form-data;boundary=" + DEFAULT_BOUNDARY);
        conn.setRequestProperty("Connection", "Keep-Alive");
        conn.setRequestMethod("POST");
        conn.setRequestProperty("Range", "bytes=" + offset + "-" + endIndex);
        OutputStream outputStream = conn.getOutputStream();
        outputStream.write(sb.toString().getBytes("UTF-8"));
        outputStream.flush();
        return conn;
    }

    @NonNull
    private StringBuffer getMultipartBody(String fileId, String billInfo) {
        return new StringBuffer().append("--").append(DEFAULT_BOUNDARY).append("\r\n")
                .append("Content-Disposition:form-data;name=\"fileId\"")
                .append("\r\n\r\n")
                .append(fileId)
                .append("\r\n")
                .append("--").append(DEFAULT_BOUNDARY).append("\r\n")
                .append("Content-Disposition:form-data;name=\"billInfo\"")
                .append("\r\n\r\n")
                .append(TextUtils.isEmpty(billInfo) ? "" : billInfo)
                .append("\r\n")
                .append("--").append(DEFAULT_BOUNDARY).append("--").append("\r\n");
    }

    private long getEndIndex(long offset) {
        long endIndex = 0;
        long l = file.getfSize() - 1 - offset;
        if (l >= callback.unit()) {
            endIndex = offset + callback.unit();
        } else {
            endIndex = file.getfSize() - 1;
        }
        return endIndex;
    }

    private long getOffset(File of) throws IOException {
        long offset = 0;
        if (!of.exists()) {
            if (!of.createNewFile()) {
                Log.d(TAG, "============文件创建失败================");
            }
        } else {
            offset = of.length() - 1;
        }
        return offset;
    }

    private boolean verifyFile(File file, String except) {
        String actual = EncryptUtils.encryptMD5File2String(file).toLowerCase();
        return actual.equals(except.toLowerCase());
    }

    private void runOnMainThread(Runnable runnable) {
        this.mainHandler.post(runnable);
    }
}
