package com.xdja.update;

import android.content.Context;
import android.os.AsyncTask;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;

import com.xdja.update.bean.CheckResult;
import com.xdja.update.enums.DownLoadFileError;

import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.net.HttpURLConnection;
import java.util.Arrays;

/**
 * <P>description : 文件下载任务 </P>
 * <P>className : DownloadTask </P>
 * <P>package : com.xdja.update </P>
 * <P>author : <a href="mailto:fjd@xdja.com">fanjiandong</a> </P>
 * <P>date-time : 2019/9/19 11:40 </P>
 *
 * @author fjd
 */
@SuppressWarnings("FieldCanBeLocal")
class DownloadTask extends AsyncTask<String, Long, Boolean> {

    private final String TAG = DownloadTask.class.getSimpleName();
    private final int BUFFER_UNIT = 1024 * 512;
    private final String DOWNLOAD_PROJECT_NAME = "/Update/api/upgrade/downloadApk.do?fileId=";

    private final CheckResult.Update.File file;
    private final DownloadCallback callback;
    private final String baseUrl;


    private volatile boolean isStop = false;
    private Context context;
    private String certName;
    private String pass;

    /**
     * Instantiates a new DownloadTask.
     *
     * @param file     需要下载的目标文件
     * @param callback 文件下载回调
     * @param baseUrl  升级服务baseUrl
     */
    DownloadTask(@NonNull CheckResult.Update.File file,
                 @NonNull DownloadCallback callback,
                 @NonNull String baseUrl,
                 Context context,
                 String certName,
                 String pass) {
        this.context = context;
        this.file = file;
        this.callback = new DelegateDownloadCallback(callback);
        this.baseUrl = baseUrl;
        this.certName = certName;
        this.pass = pass;
    }

    /**
     * 停止下载任务
     */
    synchronized void stop() {
        this.isStop = true;
    }

    @Override
    protected void onPreExecute() {
        super.onPreExecute();
        this.callback.onStart(this.file.getFileId());
    }

    @Override
    protected Boolean doInBackground(String... params) {
        final String fileId = file.getFileId();
        RandomAccessFile randomAccessFile = null;

        //检验文件ID合法性
        if (checkFileId(file.getFileId())) {
            return false;
        }

        try {
            //检验目标文件夹合法性
            if (checkDir(fileId)) {
                return false;
            }

            //检验目标文件合法性
            File targetFile = new File(callback.downloadDir() + "/" + fileId + ".apk");
            if (checkFile(fileId, targetFile)) {
                return false;
            }

            long endIndex;
            //初始化挪移游标
            long offset = targetFile.length();
            randomAccessFile = new RandomAccessFile(targetFile, "rws");
            Boolean result = seekFile(fileId, offset, randomAccessFile, targetFile);
            if (result != null) {
                return result;
            }
            endIndex = getEndIndex(offset);

            //获取文件下载地址
            String fileUrl = getFileUrl(this.baseUrl, fileId);

            while (!isStop) {
                //创建下载连接
                HttpURLConnection conn = U.HttpUtils.getDownloadFileConn(fileUrl, offset, endIndex, U.MySSLSocketFactory.getSocketFactory(context, certName, pass));
                //获取响应码
                final int responseCode = conn.getResponseCode();
                //响应正常
                if (responseCode == HttpURLConnection.HTTP_OK
                        || responseCode == HttpURLConnection.HTTP_PARTIAL) {
                    //获取输入流
                    InputStream is = conn.getInputStream();
                    if (is != null) {
                        //写入请求数据到文件中
                        appendStream(randomAccessFile, is);
                        is.close();
                        //重设下次请求开始游标
                        offset = endIndex + 1;
                        result = seekFile(fileId, offset, randomAccessFile, targetFile);
                        if (result != null) {
                            return result;
                        }
                        endIndex = getEndIndex(offset);
                    } else {
                        Log.d(TAG, "====文件下载出错====");
                        callback.onError(fileId, DownLoadFileError.DOWNLOAD_DEFAULT, null);
                        return false;
                    }
                } else {
                    Log.d(TAG, "====服务器响应错误，响应码：" + responseCode + "====");
                    callback.onError(fileId, DownLoadFileError.DOWNLOAD_SERVER_RESPONSE_ERROR, null);
                    return false;
                }
            }
            U.HandlerUtils.runOnMainThread(new Runnable() {
                @Override
                public void run() {
                    callback.onStop(fileId);
                }
            });
            return false;
        } catch (Exception e) {
            Log.d(TAG, "====\r\n文件下载出错 : " + e.getMessage() + "\r\n====");
            Log.d(TAG, "downloadFileError: " + Arrays.toString(e.getStackTrace()));
            callback.onError(fileId, DownLoadFileError.DOWNLOAD_DEFAULT, e);
            return false;
        } finally {
            try {
                if (randomAccessFile != null) {
                    randomAccessFile.close();
                }
            } catch (IOException e) {
                callback.onError(fileId, DownLoadFileError.DOWNLOAD_DEFAULT, e);
            }
        }

    }

    @Override
    protected void onProgressUpdate(Long... values) {
        super.onProgressUpdate(values);
        callback.onProgress(file.getFileId(), file.getfSize(), values[0]);
    }

    @Override
    protected void onPostExecute(Boolean result) {
        super.onPostExecute(result);
        if (result) {
            callback.onComplete(file.getFileId(), callback.downloadDir() + "/" + file.getFileId() + ".apk");
        }
    }

    private void appendStream(RandomAccessFile randomAccessFile, InputStream is) throws IOException {
        byte[] buffer = new byte[BUFFER_UNIT];
        int count;
        while ((count = is.read(buffer, 0, BUFFER_UNIT)) != -1) {
            randomAccessFile.write(buffer, 0, count);
        }
    }

    private String getFileUrl(String baseUrl, String fileId) throws Exception {
        String fileUrlResult = U.HttpUtils.postString(baseUrl + DOWNLOAD_PROJECT_NAME + fileId, null, U.MySSLSocketFactory.getSocketFactory(context, certName, pass));
        JSONObject jsonObject = new JSONObject(fileUrlResult);
        int code = jsonObject.getInt("code");
        String message = jsonObject.getString("message");
        String data = jsonObject.getString("data");
        if (code == 0) {
            throw new Exception(message);
        }
        return data;
    }

    private long getEndIndex(long offset) {
        long endIndex;
        long l = file.getfSize() - offset;
        if (l >= callback.unit()) {
            endIndex = offset + callback.unit() - 1;
        } else {
            endIndex = file.getfSize() - 1;
        }
        return endIndex;
    }

    private boolean verifyFile(File file, String except) {
        String actual = U.EncryptUtils.encryptMD5File2String(file).toLowerCase();
        return actual.equals(except.toLowerCase());
    }

    private Boolean seekFile(String fileId,
                             long offset,
                             RandomAccessFile randomAccessFile,
                             File targetFile) throws IOException {
        //文件已经下载完成
        if (offset >= file.getfSize()) {
            Log.d(TAG, "====文件已下载完成====");
            //验证文件签名
            if (verifyFile(targetFile, file.getCheckCode())) {
                return true;
            }
            //验证失败，删除文件
            if (!targetFile.delete()) {
                callback.onError(fileId, DownLoadFileError.DOWNLOAD_STORGE_PERMISSION_ERROR, null);
                return false;
            }
            //通知上层文件校验失败
            callback.onError(fileId, DownLoadFileError.DOWNLOAD_HASHCODE_ERROR, null);
            return false;
        }
        //发布已经下载的文件进度
        publishProgress(offset);
        //挪移游标
        randomAccessFile.seek(offset);
        return null;
    }

    private boolean checkDir(String fileId) {
        File targetDir = new File(callback.downloadDir());
        if (!targetDir.exists()) {
            boolean mkdirs = targetDir.mkdirs();
            if (!mkdirs) {
                Log.d(TAG, "====创建目标文件夹出错，请检查文件下载路径是否合法====");
                callback.onError(fileId, DownLoadFileError.DOWNLOAD_STORGE_PERMISSION_ERROR, null);
                return true;
            }
        }
        return false;
    }

    private boolean checkFileId(String fileId) {
        if (TextUtils.isEmpty(fileId)) {
            Log.d(TAG, "====fileId为空====");
            callback.onError(fileId, DownLoadFileError.DOWNLOAD_NULL_FIELD_ID, null);
            return true;
        }
        return false;
    }

    private boolean checkFile(String fileId, File targetFile) throws IOException {
        if (!targetFile.exists()) {
            if (!targetFile.createNewFile()) {
                Log.d(TAG, "====文件创建失败====");
                callback.onError(fileId, DownLoadFileError.DOWNLOAD_DEFAULT, null);
                return true;
            }
        }
        return false;
    }
}
