package com.xdja.update;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.AsyncTask;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.xdja.update.bean.CheckProtocol;
import com.xdja.update.bean.CheckResult;
import com.xdja.update.enums.CheckUpdateError;
import com.xdja.update.enums.DownLoadFileError;

import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;


/**
 * <P>description : 升级操作统一管理类 </P>
 * <P>className : UpdateManager </P>
 * <P>package : com.xdja.update </P>
 * <P>author : <a href="mailto:fjd@xdja.com">fanjiandong</a> </P>
 * <P>date-time : 2019/9/19 10:56 </P>
 *
 * @author fjd
 */
public class UpdateManager {
    private static final String TAG = UpdateManager.class.getSimpleName();
    private static final int CORE_POOL_SIZE = 2;
    private static final int MAXIMUM_POOL_SIZE = 5;
    private static final long KEEP_ALIVE_TIME = 10 * 1000;
    private static final int QUEUE_CAPACITY = 2;
    private static final String THREAD_GROUP_NAME = "UpdateThreadGroup";
    private static final String THREAD_NAME = "UpdateThread";
    private static final String CLIENT_VERSION_FILE = "ClientVer.xml";

    @SuppressLint("StaticFieldLeak")
    private static volatile UpdateManager INSTANCE;

    /**
     * 获取升级组件单例对象
     *
     * @return 升级组件单例对象
     */
    public static UpdateManager getInstance() {
        if (INSTANCE == null) {
            synchronized (UpdateManager.class) {
                if (INSTANCE == null) {
                    INSTANCE = new UpdateManager();
                }
            }
        }
        return INSTANCE;
    }

    private UpdateManager() {
        this.accessConfig = new AccessConfig();
    }

    private AccessConfig accessConfig;

    /**
     * 线程调度器
     */
    private Executor exec;
    /**
     * application级别的Context上下文
     */
    private Context context;

    /**
     * 检测升级流程
     */
    private AsyncTask checkTask;
    /**
     * 文件下载流程
     */
    private DownloadTask downloadTask;

    /**
     * 初始化升级组件
     *
     * @param context 上下文
     */
    public void install(@NonNull Context context) {
        this.context = context.getApplicationContext();
        this.exec = new ThreadPoolExecutor(
                CORE_POOL_SIZE,
                MAXIMUM_POOL_SIZE,
                KEEP_ALIVE_TIME,
                TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<Runnable>(QUEUE_CAPACITY),
                new ThreadFactory() {
                    int count = 0;

                    @Override
                    public Thread newThread(@NonNull Runnable r) {
                        return new Thread(
                                new ThreadGroup(THREAD_GROUP_NAME),
                                r,
                                THREAD_NAME + count++
                        );
                    }
                }
        );
    }

    /**
     * 初始化升级组件
     *
     * @param context      上下文
     * @param ip           ip
     * @param port         port
     * @param cardNo       设备标识
     * @param isAppSelfish 部分配置信息（Soft/VerType/Version）是否从APK安装包中读取(默认为false)
     * @param isSubstitute 是否替换ClientVer.xml文件中配置的ip和端口(默认为false)
     */
    public void install(
            @NonNull Context context,
            @Nullable String ip,
            @Nullable String port,
            @Nullable String cardNo,
            boolean isAppSelfish,
            boolean isSubstitute) {
        this.install(context, null, ip, port, cardNo, isAppSelfish, isSubstitute);
    }

    /**
     * 初始化升级组件
     *
     * @param context      上下文
     * @param scheme       http协议还是https协议
     * @param ip           ip
     * @param port         port
     * @param cardNo       设备标识
     * @param isAppSelfish 部分配置信息（Soft/VerType/Version）是否从APK安装包中读取(默认为false)
     * @param isSubstitute 是否替换ClientVer.xml文件中配置的ip和端口(默认为false)
     */
    public void install(
            @NonNull Context context,
            @Nullable HTTP_SCHEME scheme,
            @Nullable String ip,
            @Nullable String port,
            @Nullable String cardNo,
            boolean isAppSelfish,
            boolean isSubstitute) {
        this.install(context, scheme, ip, port, cardNo, isAppSelfish, isSubstitute, null, null);
    }

    /**
     * 初始化升级组件
     *
     * @param context      上下文
     * @param scheme       http协议还是https协议
     * @param ip           ip
     * @param port         port
     * @param cardNo       设备标识
     * @param isAppSelfish 部分配置信息（Soft/VerType/Version）是否从APK安装包中读取(默认为false)
     * @param isSubstitute 是否替换ClientVer.xml文件中配置的ip和端口(默认为false)
     * @param certName     assets目录下服务器证书，使用bks格式的证书
     * @param pass         证书库密码
     */
    public void install(
            @NonNull Context context,
            @Nullable HTTP_SCHEME scheme,
            @Nullable String ip,
            @Nullable String port,
            @Nullable String cardNo,
            boolean isAppSelfish,
            boolean isSubstitute,
            @Nullable String certName,
            @Nullable String pass) {
        install(context);
        if (scheme != null) {
            this.accessConfig.setSchemeStr(scheme.value);
        }
        this.accessConfig.setIp(ip);
        this.accessConfig.setPort(port);
        this.accessConfig.setCardNo(cardNo);
        this.accessConfig.setAppSelfish(isAppSelfish);
        this.accessConfig.setSubstitute(isSubstitute);
        this.accessConfig.setCertName(certName);
        this.accessConfig.setPass(pass);
    }

    /**
     * 释放组件相关资源
     */
    public void uninstall() {
        this.context = null;
        this.exec = null;
        this.checkTask = null;
        this.downloadTask = null;
        this.accessConfig = null;
    }

    /**
     * 检测升级
     *
     * @param checkCallback 检测升级回调
     */
    public void checkUpdate(@NonNull CheckCallback checkCallback) {
        if (isBusy()) {
            checkCallback.onCheckResult(buildBusyError());
            Log.d(TAG, "======有未执行完成的任务,本次检测任务取消=======");
            return;
        }
        Log.d(TAG, "======无待执行的任务,本次检测即将开始=======");
        this.checkTask = new CheckTask<String>(context, checkCallback, this.accessConfig)
                .executeOnExecutor(this.exec, CLIENT_VERSION_FILE);
    }

    /**
     * 检测升级
     *
     * @param checkCallback 升级回调
     * @param checkProtocol 升级检测协议对象
     */
    public void checkUpdate(
            @NonNull CheckCallback checkCallback,
            @NonNull CheckProtocol checkProtocol) {
        if (isBusy()) {
            checkCallback.onCheckResult(buildBusyError());
            Log.d(TAG, "======有未执行完成的任务,本次检测任务取消=======");
            return;
        }
        Log.d(TAG, "======无待执行的任务,本次检测即将开始=======");
        this.checkTask = new CheckTask<CheckProtocol>(context, checkCallback, this.accessConfig)
                .executeOnExecutor(this.exec, checkProtocol);
    }

    /**
     * 取消升级检测
     */
    public void cancelCheckUpdate() {
        if (this.checkTask != null) {
            this.checkTask.cancel(true);
        }
    }

    /**
     * 下载文件
     *
     * @param file     待下载文件
     * @param callback 文件下载回调
     */
    public void downloadFile(@NonNull CheckResult.Update.File file, @NonNull DownloadCallback callback) {
        if (isBusy()) {
            callback.onError(file.getFileId(), DownLoadFileError.DOWNLOAD_BUSY_ERROR, null);
            Log.d(TAG, "======有未执行完成的任务,本次下载取消=======");
            return;
        }
        Log.d(TAG, "======无待执行的任务,本次下载即将开始=======");
        this.downloadTask = new DownloadTask(file, callback, accessConfig.getBaseUrl(), context, this.accessConfig.getCertName(), this.accessConfig.getPass());
        this.downloadTask.executeOnExecutor(exec);
    }

    /**
     * 停止文件下载流程
     */
    public void stopDownload() {
        if (this.downloadTask != null
                && this.downloadTask.getStatus() != AsyncTask.Status.FINISHED) {
            this.downloadTask.stop();
        }
    }

    private CheckResult buildBusyError() {
        CheckResult checkResult = new CheckResult();
        checkResult.setResult(CheckResult.RESULT_UPDATE_ERROR);
        checkResult.setCheckUpdateError(CheckUpdateError.CHECK_BUSY_ERROR);
        return checkResult;
    }

    /**
     * 检测当前是否有正在执行的任务
     *
     * @return 检测结果
     */
    private boolean isBusy() {
        return (this.checkTask != null
                && this.checkTask.getStatus() != AsyncTask.Status.FINISHED)
                || (this.downloadTask != null
                && this.downloadTask.getStatus() != AsyncTask.Status.FINISHED);
    }

    public enum HTTP_SCHEME {
        HTTP("http"), HTTPS("https");
        private String value;
        HTTP_SCHEME(String value) {
            this.value = value;
        }
    }

    static final class AccessConfig {

        /**
         * assets目录服务器证书名称
         */
        private String certName;

        /**
         * http协议还是https协议，默认http
         */
        private String schemeStr = HTTP_SCHEME.HTTP.value;

        /**
         * 升级服务IP
         */
        private String ip;

        /**
         * 升级服务端口
         */
        private String port;

        /**
         * 作为灰度升级标识
         */
        private String cardNo;
        /**
         * 部分配置信息（Soft/VerType/Version）是否从APK安装包中读取（默认为false）
         */
        private boolean isAppSelfish = false;
        /**
         * 是否替换ClientVer.xml文件中配置的ip和端口（默认为false）
         */
        private boolean isSubstitute = false;

        /**
         * 升级服务baseUrl
         */
        private String baseUrl;

        /**
         * 证书密码
         */
        private String pass;


        /**
         * @value #baseUrl
         */
        String getBaseUrl() {
            return baseUrl;
        }

        /**
         * @param baseUrl {@link #baseUrl}
         */
        void setBaseUrl(String baseUrl) {
            this.baseUrl = baseUrl;
        }

        /**
         * @return {@link #ip}
         */
        String getIp() {
            return ip;
        }

        /**
         * @param ip {@link #ip}
         */
        private void setIp(String ip) {
            this.ip = ip;
        }

        /**
         * @return {@link #port}
         */
        String getPort() {
            return port;
        }

        /**
         * @param port {@link #port}
         */
        private void setPort(String port) {
            this.port = port;
        }

        /**
         * @return {@link #cardNo}
         */
        String getCardNo() {
            return cardNo;
        }

        /**
         * @param cardNo {@link #cardNo}
         */
        private void setCardNo(String cardNo) {
            this.cardNo = cardNo;
        }

        /**
         * @return {@link #isSubstitute}
         */
        boolean isSubstitute() {
            return isSubstitute;
        }

        /**
         * @param substitute {@link #isSubstitute}
         */
        private void setSubstitute(boolean substitute) {
            isSubstitute = substitute;
        }

        /**
         * @return {@link #isAppSelfish}
         */
        public boolean isAppSelfish() {
            return isAppSelfish;
        }

        /**
         * @param isAppSelfish {@link #isAppSelfish}
         */
        public void setAppSelfish(boolean isAppSelfish) {
            this.isAppSelfish = isAppSelfish;
        }

        /**
         *
         * @return {@link #schemeStr}
         */
        String getSchemeStr() {
            return schemeStr;
        }

        /**
         *
         * @param schemeStr {@link #schemeStr}
         */
        void setSchemeStr(String schemeStr) {
            this.schemeStr = schemeStr;
        }

        /**
         *
         * @return {@link #certName}
         */
        String getCertName() {
            return certName;
        }

        /**
         *
         * @param certName {@link #certName}
         */
        void setCertName(String certName) {
            this.certName = certName;
        }

        /**
         *
         * @return {@link #pass}
         */
        String getPass() {
            return pass;
        }

        /**
         *
         * @param pass {@link #pass}
         */
        void setPass(String pass) {
            this.pass = pass;
        }
    }
}
