package com.xdja.upgrade.task;

import com.xdja.upgrade.NetWorkUtil;
import com.xdja.upgrade.UpdateLog;
import com.xdja.upgrade.bean.ClientVersion;
import com.xdja.upgrade.util.UpdateConfigConst;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;

/**
 * Update模块下载通用Socket
 */
public abstract class SocketService {

    public static final String TAG = SocketService.class.getName();

    protected ClientVersion clientVersion;

    private Socket socket;

    private InputStream inputStream; // 网络输入流

    private OutputStream outputStream;// 网络输出流

    public SocketService(ClientVersion clientVersion) {
        UpdateLog.d(TAG, "SocketService构造函数中的version");
        UpdateLog.d(TAG, clientVersion.toString());
        this.clientVersion = clientVersion;
    }

    /**
     * 1 执行连接服务器
     * 2 发送数据给服务器
     * 3 接收服务器返回数据
     *
     * @return SocketResult
     * @see SocketResult
     */
    public SocketResult executeSocket() {

        UpdateLog.d(TAG, "IP合法性判断");
        if (clientVersion.getIp().equals("")) {
            UpdateLog.d(TAG, "IP未设置");
            return new SocketResult(UpdateConfigConst.CODE_IP_NOT_SET);
        }

        UpdateLog.d(TAG, "端口合法性判断");
        if (clientVersion.getPort().equals("")) {
            UpdateLog.d(TAG, "端口未设置");
            return new SocketResult(UpdateConfigConst.CODE_PORT_NOT_SET);
        }

        UpdateLog.d(TAG, "域名解析");
        String hostIP = NetWorkUtil.getHostAddress(clientVersion.getIp());
        if (hostIP == null) {
            UpdateLog.d(TAG, "IP解析错误");
            return new SocketResult(UpdateConfigConst.CODE_IP_PARSE);
        }

        UpdateLog.d(TAG, "连接服务器");
        int connectServerResult = connectServer();

        UpdateLog.d(TAG, "连接异常判断");
        if (connectServerResult == UpdateConfigConst.ERROR) {
            UpdateLog.d(TAG, "服务器连接异常");
            // TODO: 2017/6/15 0015 1，连接升级服务器失败,重写update_s.xml
            return new SocketResult(UpdateConfigConst.CODE_CON_SERVER);
        }

        //组装请求头，发送请求头
        int requestHeadResult = sendRequestHead(getHeader());
        //升级模块：启动Socket流程 是否请求异常
        if (requestHeadResult == UpdateConfigConst.ERROR) {
            // TODO: 2017/6/15 0015 2，发送下载升级文件请求失败，重写update_s.xml
            return new SocketResult(UpdateConfigConst.CODE_SEND_DATA);
        }

        UpdateLog.d(TAG, "接收返回数据");
        byte[] dataArray = recvData();
        UpdateLog.d(TAG, "读取返回数据异常判断");
        if (dataArray == null) {
            UpdateLog.d(TAG, "读取返回数据异常");
            // TODO: 2017/6/15 0015 3，接收升级文件配置参数超时，重写update_s.xml
            return new SocketResult(UpdateConfigConst.CODE_ACCEPT_DATA_TIMEOUT);
        } else {
            UpdateLog.d(TAG, "读取返回数据正常");
            //升级模块：启动Socket流程 解析返回数据头
            return parseResponseHeader(dataArray);
        }
    }

    protected abstract String getHeader();


    protected abstract SocketResult parseResponseHeader(byte[] rxd);


    protected int read(byte[] buffer, int byteOffset, int byteCount) throws IOException {
        return inputStream.read(buffer, byteOffset, byteCount);
    }

    /**
     * 连接到更新检测服务器
     * 并设置超时时间(超时5s断开连接)
     *
     * @return
     */
    int connectServer() {
        try {
            socket = new Socket(clientVersion.getIp(), Integer.parseInt(clientVersion.getPort()));
            socket.setSoTimeout(clientVersion.getTimeOut());
            inputStream = socket.getInputStream();
            outputStream = socket.getOutputStream();
        } catch (Exception e) {
            e.getMessage();
            return UpdateConfigConst.ERROR;
        }
        return UpdateConfigConst.SUCCESS;
    }

    /**
     * 向服务器发送请求头
     *
     * @param header
     * @return
     */
    int sendRequestHead(String header) {
        UpdateLog.d(TAG, "发送请求头" + header.toString());
        try {
            UpdateLog.d(TAG, "请求异常判断");
            int dataLen = header.getBytes().length;
            byte[] data = new byte[dataLen + 2];
            data[0] = (byte) (dataLen >> 8 & 0xff);
            data[1] = (byte) (dataLen & 0xff);// 两个字节数据长度
            System.arraycopy(header.getBytes(), 0, data, 2, dataLen);// 数据内容
            outputStream.write(data);
            outputStream.flush();
            UpdateLog.d(TAG, "请求正常");
            return UpdateConfigConst.SUCCESS;
        } catch (Exception e) {
            UpdateLog.d(TAG, "请求异常");
            e.getStackTrace();
            return UpdateConfigConst.ERROR;
        }
    }

    /**
     * 接收检测升级服务器返回的数据
     *
     * @return
     */
    //wanghao 这里的逻辑暂时不修改
    byte[] recvData() {
        try {
            byte[] data = null;
            int dataLen = 0;
            int currentLen = 0;
            int tempLen = 0;
            int headLen = 0;

            byte[] len = new byte[2];// 头2个字节是数据长度+ 后面数据内容
            while (tempLen != -1) {//获取数据长度缓冲区
                headLen += tempLen;
                if (headLen < 2) {
                    tempLen = inputStream.read(len, headLen, 2 - headLen);
                } else {
                    break;
                }
            }
            if (tempLen == -1) {
                return null;
            }

            dataLen = bytes2Int(len);// (len[0]&0xff)<<8 + len[1]&0xff;
            data = new byte[dataLen];
            tempLen = 0;
            while (tempLen != -1) {//获取数据内容
                currentLen += tempLen;
                if (currentLen < dataLen) {
                    tempLen = inputStream.read(data, currentLen, dataLen - currentLen);
                } else {
                    break;
                }
            }
            if (tempLen == -1) {
                return null;
            }
            return data;
        } catch (IOException e) {
            e.getStackTrace();
            return null;
        }
    }


    int bytes2Int(byte[] b) {
        int mask = 0xff;
        int temp = 0;
        int res = 0;
        for (int i = 0; i < 2; i++) {
            res <<= 8;
            temp = b[i] & mask;
            res |= temp;
        }
        return res;
    }

    public void close() {
        try {
            if (inputStream != null) {
                inputStream.close();
            }
            if (outputStream != null) {
                outputStream.close();
            }
            if (socket != null) {
                socket.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 封装socket连接服务器之后返回的数据
     */
    public class SocketResult {

        /**
         * 服务器返回错误码
         */
        private int resultCode;

        /**
         * 服务器返回的数据
         */
        private byte[] resultBytes;

//        这个地方原本是打算添加一个方法，然后子类继承重写的
//        因为版本检测时，请求成功的状态码是CODE_DEFAULT_SUCCESS
//        而下载时，请求成功的状态码是SUCCESS，两个略有不同
//        public boolean isSuccess() {
//
//        }

        public SocketResult(int resultCode) {
            this(resultCode, null);
        }

        public SocketResult(int resultCode, byte[] bytes) {
            this.resultCode = resultCode;
            this.resultBytes = bytes;
        }

        public int getResultCode() {
            return resultCode;
        }

        public byte[] getResultBytes() {
            return resultBytes;
        }

    }

    public static String getResultMessage(int resultCode) {
        return UpdateConfigConst.ConfigWrapper.getErrorDescription(resultCode);
    }

}
