/*
 *   Copyright 2013 INFTEL - Proveedor de Servicios de Aplicación
 *   (www.inftel.com.mx). All Rights Reserved (Todos los derechos reservados).
 * 
 *   Copyright 2013 Santos Zatarain Vera (santoszv _at_ inftel.com.mx).
 *   All Rights Reserved (Todos los derechos reservados).
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */
package mx.com.inftel.shiro.oauth2;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authc.AuthenticationToken;
import org.apache.shiro.authc.UsernamePasswordToken;
import org.apache.shiro.codec.Base64;
import org.apache.shiro.subject.Subject;
import org.apache.shiro.web.filter.authc.AuthenticatingFilter;
import org.json.JSONObject;

/**
 * Base class for filters authenticating with OAuth 2.0. This class perform
 * almost all processing for code authorization gran type, subclasses only need
 * to fill some customizations.
 *
 * If current request is made by an authenticated subject, the filters' chain
 * will continue inmediately.
 *
 * If filter is not {@code permissive}, current request's path matchs property
 * {@code loginUrl} and the subject is unauthenticated, login will be attemped.
 * On successful login, the filters' chain will continue, else will not.
 *
 * If filter is {@code permissive}, current request's query parameter
 * {@code oauth2} equals to filter's name and the subject is unauthenticated,
 * login will be attemped. On successful login, the filters' chain will
 * continue, else will not.
 *
 * @author Santos Zatarain Vera <santoszv@inftel.com.mx>
 */
public abstract class AbstractOAuth2AuthenticatingFilter extends AuthenticatingFilter {

    public static final int MIN_LEN = 32;
    private SecureRandom secureRandom;
    private AccessDeniedHandler accessDeniedHandler;
    private LoginSuccessHandler loginSuccessHandler;
    private LoginFailureHandler loginFailureHandler;
    private int stateLength;
    private String stateAttribute;
    private boolean usernamePasswordToken;
    private String redirectUri;
    private String clientId;
    private String clientSecret;

    /**
     * Default constructor.
     */
    public AbstractOAuth2AuthenticatingFilter() {
        this.secureRandom = new SecureRandom();
        this.stateLength = MIN_LEN;
        this.stateAttribute = "STATE_" + random(MIN_LEN);
        this.usernamePasswordToken = false;
        this.redirectUri = "";
        this.clientId = "";
        this.clientSecret = "";
    }

    /**
     * Returns the random bytes generator used internally.
     *
     * @return Random bytes generator.
     */
    public SecureRandom getSecureRandom() {
        return secureRandom;
    }

    /**
     * Modify the random bytes generator used internally.
     *
     * Setting to {@code null} will create a new instance of
     * {@code SecureRandom} internally.
     *
     * @param secureRandom Random bytes generator.
     */
    public void setSecureRandom(SecureRandom secureRandom) {
        this.secureRandom = (secureRandom == null ? new SecureRandom() : secureRandom);
    }

    /**
     * Returns current handler used when access is denied.
     *
     * @return Current handler or {@code null}.
     */
    public AccessDeniedHandler getAccessDeniedHandler() {
        return accessDeniedHandler;
    }

    /**
     * Adjust current handler used when access is denied.
     *
     * @param accessDeniedHandler Handler.
     */
    public void setAccessDeniedHandler(AccessDeniedHandler accessDeniedHandler) {
        this.accessDeniedHandler = accessDeniedHandler;
    }

    /**
     * Returns current handler used when login success.
     *
     * @return Current handler or {@code null}.
     */
    public LoginSuccessHandler getLoginSuccessHandler() {
        return loginSuccessHandler;
    }

    /**
     * Adjust current handler used when login success.
     *
     * @param loginSuccessHandler Handler.
     */
    public void setLoginSuccessHandler(LoginSuccessHandler loginSuccessHandler) {
        this.loginSuccessHandler = loginSuccessHandler;
    }

    /**
     * Returns current handler used when login fails.
     *
     * @return Current handler or {@code null}.
     */
    public LoginFailureHandler getLoginFailureHandler() {
        return loginFailureHandler;
    }

    /**
     * Adjust current handler used when login fails.
     *
     * @param loginFailureHandler Handler.
     */
    public void setLoginFailureHandler(LoginFailureHandler loginFailureHandler) {
        this.loginFailureHandler = loginFailureHandler;
    }

    /**
     * Returns the length of state value used internally.
     *
     * @return Length of state value.
     */
    public int getStateLength() {
        return stateLength;
    }

    /**
     * Adjust the length of state value used internally.
     *
     * Setting to a value less than {@code MIN_LEN} will set {@code MIN_LEN}
     * internally.
     *
     * @param stateLength Length of state value.
     */
    public void setStateLength(int stateLength) {
        this.stateLength = (stateLength < MIN_LEN ? MIN_LEN : stateLength);
    }

    /**
     * Returns attribute's name used for store state value in session.
     *
     * @return Attribute's name for state value.
     */
    public String getStateAttribute() {
        return stateAttribute;
    }

    /**
     * Modify attribute's name used for store state value in session.
     *
     * Setting to {@code null}, empty string or space-like only string will
     * generate a random alpha-numeric string of length {@code MIN_LEN}, then
     * prefix with {@code "STATE_"} and finally set internally.
     *
     * @param stateAttribute Attribute's name for state value.
     */
    public void setStateAttribute(String stateAttribute) {
        this.stateAttribute = (stateAttribute == null || stateAttribute.trim().isEmpty() ? "STATE_" + random(MIN_LEN) : stateAttribute);
    }

    /**
     * Returna a indication for using {@code UsernamePasswordToken} instead of
     * {@code OAuth2AuthenticationToken}. THIS IS INTENDED FOR DEBUG PURPOSE ONLY.
     *
     * @return Value {@code true} if {@code UsernamePasswordToken} should be
     * used.
     */
    public boolean isUsernamePasswordToken() {
        return usernamePasswordToken;
    }

    /**
     * Adjust the indication for using {@code UsernamePasswordToken} instead of
     * {@code OAuth2AuthenticationToken}. THIS IS INTENDED FOR DEBUG PURPOSE ONLY.
     *
     * @param usernamePasswordToken Use {@code true} if
     * {@code UsernamePasswordToken} should be used.
     */
    public void setUsernamePasswordToken(boolean usernamePasswordToken) {
        this.usernamePasswordToken = usernamePasswordToken;
    }

    /**
     * Returns the value of property {@code redirectURI}.
     *
     * @return Value of property {@code redirectURI}.
     */
    public String getRedirectUri() {
        return redirectUri;
    }

    /**
     * Modify the value of property {@code redirectURI}.
     *
     * @param redirectURI New value for property {@code redirectURI}.
     */
    public void setRedirectUri(String redirectURI) {
        this.redirectUri = (redirectURI == null || redirectURI.trim().isEmpty() ? "" : redirectURI);
    }

    /**
     * Returns the value of property {@code clientId}.
     *
     * @return Value of property {@code clientId}.
     */
    public String getClientId() {
        return clientId;
    }

    /**
     * Modify the value of property {@code clientId}.
     *
     * @param clientId New value for property {@code clientId}.
     */
    public void setClientId(String clientId) {
        this.clientId = (clientId == null || clientId.trim().isEmpty() ? "" : clientId);
    }

    /**
     * Returns the value of property {@code clientSecret}.
     *
     * @return Value of property {@code clientSecret}.
     */
    public String getClientSecret() {
        return clientSecret;
    }

    /**
     * Modify the value of property {@code clientId}.
     *
     * @param clientSecret New value for property {@code clientSecret}.
     */
    public void setClientSecret(String clientSecret) {
        this.clientSecret = (clientSecret == null || clientSecret.trim().isEmpty() ? "" : clientSecret);
    }

    private String random(int len) {
        byte[] randomBytes = new byte[MIN_LEN * 3 / 4];
        StringBuilder sb = new StringBuilder(len + MIN_LEN);

        while (sb.length() < len) {
            secureRandom.nextBytes(randomBytes);
            sb.append(Base64.encodeToString(randomBytes).replace("+", "").replace("/", ""));
        }
        sb.setLength(len);

        return sb.toString();
    }

    /**
     * Encode a string calling directly {@code URLEncoder.encode(unencoded, "UTF-8")}.
     *
     * @param unencoded String to encode.
     * @return Encoded string.
     * @throws UnsupportedEncodingException If the runtime environment doesn't
     * support UTF-8 character set.
     */
    protected String encodeURL(String unencoded) throws UnsupportedEncodingException {
        return URLEncoder.encode(unencoded, "UTF-8");
    }

    /**
     * Decode a string calling directly {@code URLDecoder.decode(encoded, "UTF-8")}.
     *
     * @param encoded String to decode.
     * @return Dedecoded string.
     * @throws UnsupportedEncodingException If the runtime environment doesn't
     * support UTF-8 character set.
     */
    protected String decodeURL(String encoded) throws UnsupportedEncodingException {
        return URLDecoder.decode(encoded, "UTF-8");
    }

    /**
     * Returns the current request URL without query parameters.
     *
     * This method has the same purpose of {@code HttpServletRequest.getRequestURL()},
     * but returns a {@code StringBuilder}.
     *
     * @param request Current request.
     * @param response Current repsonse.
     * @return Request URL without query string.
     */
    protected StringBuilder getRequestURL(ServletRequest request, ServletResponse response) {
        HttpServletRequest httpRequest = (HttpServletRequest) request;

        StringBuilder sb = new StringBuilder(1024);

        return sb.append(httpRequest.getRequestURL());
    }

    /**
     * Appends the current request's query string. If there isn't query string,
     * this method will returns {@code false}.
     *
     * @param request Current request.
     * @param response Current response.
     * @param sb {@code StringBuilder} where append current query string.
     * @return Value {@code true} if there is query string, else {@code false}.
     * @throws UnsupportedEncodingException If the runtime environment doesn't
     * support UTF-8 character set.
     */
    protected boolean getRequestQUERY(ServletRequest request, ServletResponse response, StringBuilder sb) throws UnsupportedEncodingException {
        HttpServletRequest httpRequest = (HttpServletRequest) request;

        boolean apendded = false;

        Iterator<Map.Entry<String, String[]>> iterator = httpRequest.getParameterMap().entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, String[]> entry = iterator.next();
            String key = entry.getKey();
            String[] values = entry.getValue();
            for (String value : values) {
                if (!apendded) {
                    sb.append("?");
                    apendded = true;
                } else {
                    sb.append("&");
                }
                sb.append(encodeURL(key));
                sb.append("=");
                sb.append(encodeURL(value));
            }
        }

        return apendded;
    }

    /**
     * Returns a standard authorize URL for redirecting resource owner.
     *
     * This method is considering that five parameters must send to the
     * authrozation web endpoint: {@code client_id}, {@code scope},
     * {@code response_type}, {@code redirect_uri} and {@code state}.
     *
     * The returned string has the form
     * {@code <baseURL>?<client_id>&<scope>&<response_type>&<redirect_uri>&<state>}
     *
     * Parameter {@code baseURL} must be an URL with scheme, server and path,
     * without query string neither question mark. This URL should be the web
     * endpoint where resource owner grants authorization.
     *
     * Parameter {@code scope} should not be encoded. Multiple scopes should be
     * separated by space.
     *
     * Before calling this method, state should be stored in session.
     *
     * @param request Current request.
     * @param response Current reposnse
     * @param baseURL Web endpoint where resource owner grant authorization.
     * @param scope Scope(s) of authorization.
     * @return URL where to redirect resource owner.
     * @throws Exception If some exception is thrown while calling this method.
     */
    protected String makeStandardAuthorizeURL(ServletRequest request, ServletResponse response, String baseURL, String scope) throws Exception {
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        HttpSession session = httpRequest.getSession();

        String state = (String) session.getAttribute(stateAttribute);

        StringBuilder sb = new StringBuilder(1024);
        sb.append(baseURL);

        sb.append("?");
        sb.append(encodeURL("client_id"));
        sb.append("=");
        sb.append(encodeURL(getClientId()));

        sb.append("&");
        sb.append(encodeURL("scope"));
        sb.append("=");
        sb.append(encodeURL(scope));

        sb.append("&");
        sb.append(encodeURL("response_type"));
        sb.append("=");
        sb.append(encodeURL("code"));

        sb.append("&");
        sb.append(encodeURL("redirect_uri"));
        sb.append("=");
        sb.append(encodeURL(redirectUri));

        sb.append("&");
        sb.append(encodeURL("state"));
        sb.append("=");
        sb.append(encodeURL(state));

        return sb.toString();
    }

    /**
     * Determines if the current requeste is a login request.
     *
     * @param request Current request.
     * @param response Current response.
     * @param mappedValue Filter's mapped value.
     * @return Value {@code true} if the current request is a login request,
     * else {@code false}.
     */
    protected boolean isLoginRequest(ServletRequest request, ServletResponse response, Object mappedValue) {
        HttpServletRequest httpRequest = (HttpServletRequest) request;

        if (mappedValue != null) {
            String[] values = (String[]) mappedValue;
            if (Arrays.binarySearch(values, PERMISSIVE) >= 0) {
                return getName().equals(httpRequest.getParameter("oauth2"));
            }
        }

        return isLoginRequest(request, response);
    }

    /**
     * Determines if the current requeste is an error request.
     *
     * @param request Current request.
     * @param response Current response.
     * @param mappedValue Filter's mapped value.
     * @return Value {@code true} if the current request is an error request,
     * else {@code false}.
     */
    protected boolean isErrorRequest(ServletRequest request, ServletResponse response, Object mappedValue) {
        String paramError = request.getParameter("error");

        if (paramError != null && !paramError.trim().isEmpty()) {
            return true;
        }

        return false;
    }

    /**
     * Determines if the current requeste is a token request.
     *
     * @param request Current request.
     * @param response Current response.
     * @param mappedValue Filter's mapped value.
     * @return Value {@code true} if the current request is an token request,
     * else {@code false}.
     */
    protected boolean isTokenRequest(ServletRequest request, ServletResponse response, Object mappedValue) {
        String paramCode = request.getParameter("code");
        String paramState = request.getParameter("state");

        if (paramCode != null && !paramCode.trim().isEmpty()
                && paramState != null && !paramState.trim().isEmpty()) {
            return true;
        }

        return false;
    }

    /**
     * Clear state if is not necesary to retain anymore.
     *
     * The state is stored in sesion, is not necesary anymore when the subject
     * is authenticated, current request is an error request or the current
     * request is not a login request neither token request.
     *
     * @param request Current request.
     * @param response Current response.
     * @param mappedValue Filter's mapped value.
     */
    protected void clearStateIfNeccesary(ServletRequest request, ServletResponse response, Object mappedValue) {
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        HttpSession session = httpRequest.getSession();

        if (getSubject(request, response).isAuthenticated()
                || !isLoginRequest(request, response, mappedValue)
                || isErrorRequest(request, response, mappedValue)
                || !isTokenRequest(request, response, mappedValue)) {
            session.removeAttribute(stateAttribute);
        }
    }

    /**
     * Clear the state stored in sesion and returns the value.
     *
     * @param request Current request.
     * @param response Current response.
     * @param mappedValue Filter's mapped value.
     * @return Value of state.
     */
    protected String clearAndReturnState(ServletRequest request, ServletResponse response, Object mappedValue) {
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        HttpSession session = httpRequest.getSession();

        String state = (String) session.getAttribute(stateAttribute);
        session.removeAttribute(stateAttribute);

        return state;
    }

    /**
     * Returns the web endpoint where resource owner grants authorization.
     *
     * @param request Current request.
     * @param response Current response.
     * @return Authrization web endpoint.
     * @throws Exception If some exception is thrown while calling this method.
     */
    protected abstract String getAuthorizeURL(ServletRequest request, ServletResponse response) throws Exception;

    /**
     * Returns the JSON encoded object used as principal.
     *
     * The principal should be a JSON object returned by some web endpoint for
     * querying the profile of authenticated resource owner (not the authorized
     * client).
     *
     * @param request Current request.
     * @param response Current response.
     * @return Principal.
     * @throws Exception If some exception is thrown while calling this method.
     */
    protected abstract JSONObject getOAuth2Principal(ServletRequest request, ServletResponse response) throws Exception;

    /**
     * Returns ths credentials inside the principal.
     *
     * The credentials should be some information well-known by the resource
     * owner and the authorized client like the account email.
     *
     * @param principal Principal.
     * @return Credentials.
     * @throws Exception If some exception is thrown while calling this method.
     */
    protected abstract String getOAuth2Credentials(JSONObject principal) throws Exception;

    /**
     * Create a new authentication token.
     *
     * @param request Current request.
     * @param response Current response.
     * @return Authentication token.
     * @throws Exception If some exception is thrown while calling this method.
     */
    @Override
    protected AuthenticationToken createToken(ServletRequest request, ServletResponse response) throws Exception {
        JSONObject principal = getOAuth2Principal(request, response);
        String credentials = getOAuth2Credentials(principal);
        return (usernamePasswordToken ? new UsernamePasswordToken(credentials, getName()) : new OAuth2AuthenticationToken(principal, credentials, getName()));
    }

    /**
     * Determine if access is allowed.
     *
     * @param request Current request.
     * @param response Current response.
     * @param mappedValue Filter's mapped value.
     * @return Value {@code true} if filters' chain should continue.
     * @throws Exception If some exception is thrown while calling this method.
     */
    @Override
    protected boolean isAccessAllowed(ServletRequest request, ServletResponse response, Object mappedValue) {
        return getSubject(request, response).isAuthenticated() || (!isLoginRequest(request, response, mappedValue) && isPermissive(mappedValue));
    }

    /**
     * Checks if access is allowed or denied.
     *
     * @param request Current request.
     * @param response Current response.
     * @param mappedValue Filter's mapped value.
     * @return Value {@code true} if filters' chain should continue.
     * @throws Exception If some exception is thrown while calling this method.
     */
    @Override
    public boolean onPreHandle(ServletRequest request, ServletResponse response, Object mappedValue) throws Exception {
        clearStateIfNeccesary(request, response, mappedValue);
        return isAccessAllowed(request, response, mappedValue) || onAccessDenied(request, response, mappedValue);
    }

    /**
     * Redirection to authorization web endpoint when is necessary and/or
     * attemps to login if is neccesary.
     *
     * @param request Current request.
     * @param response Current response.
     * @param mappedValue Filter's mapped value.
     * @return Value {@code true} if filters' chain should continue.
     * @throws Exception If some exception is thrown while calling this method.
     */
    @Override
    protected boolean onAccessDenied(ServletRequest request, ServletResponse response, Object mappedValue) throws Exception {
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        HttpServletResponse httpResponse = (HttpServletResponse) response;
        HttpSession session = httpRequest.getSession();

        String paramState = request.getParameter("state");

        if (isLoginRequest(request, response, mappedValue)) {
            if (!isErrorRequest(request, response, mappedValue)) {
                if (isTokenRequest(request, response, mappedValue)) {
                    if (paramState.equals(clearAndReturnState(request, response, mappedValue))) {
                        return executeLogin(request, response) || onAccessDenied(request, response);
                    }
                } else {
                    session.setAttribute(stateAttribute, random(stateLength));
                    httpResponse.sendRedirect(getAuthorizeURL(request, response));
                    return false;
                }
            }
        }

        return onAccessDenied(request, response);
    }

    /**
     * Called when the acces is denied. This method is called also when
     * execution of login failed.
     *
     * @param request Current request.
     * @param response Current response.
     * @return Value {@code true} if filters' chain should continue.
     * @throws Exception If some exception is thrown while calling this method.
     */
    @Override
    protected boolean onAccessDenied(ServletRequest request, ServletResponse response) throws Exception {
        return (accessDeniedHandler != null
                ? accessDeniedHandler.onAccessDenied(this, request, response)
                : false);
    }

    @Override
    protected boolean onLoginSuccess(AuthenticationToken token, Subject subject, ServletRequest request, ServletResponse response) throws Exception {
        return (loginSuccessHandler != null
                ? loginSuccessHandler.onLoginSuccess(this, token, subject, request, response)
                : true);
    }

    @Override
    protected boolean onLoginFailure(AuthenticationToken token, AuthenticationException e, ServletRequest request, ServletResponse response) {
        return (loginFailureHandler != null
                ? loginFailureHandler.onLoginFailure(this, token, e, request, response)
                : false);
    }

    /**
     * Interface for acces denied handler.
     */
    public interface AccessDeniedHandler {

        boolean onAccessDenied(AbstractOAuth2AuthenticatingFilter filter, ServletRequest request, ServletResponse response) throws Exception;
    }

    /**
     * Interface for login sucess handler.
     */
    public interface LoginSuccessHandler {

        boolean onLoginSuccess(AbstractOAuth2AuthenticatingFilter filter, AuthenticationToken token, Subject subject, ServletRequest request, ServletResponse response) throws Exception;
    }

    /**
     * Interface for login failure handler.
     */
    public interface LoginFailureHandler {

        boolean onLoginFailure(AbstractOAuth2AuthenticatingFilter filter, AuthenticationToken token, AuthenticationException e, ServletRequest request, ServletResponse response);
    }
}
