/*
 *   Copyright 2013 INFTEL - Proveedor de Servicios de Aplicación
 *   (www.inftel.com.mx). All Rights Reserved (Todos los derechos reservados).
 * 
 *   Copyright 2013 Santos Zatarain Vera (santoszv _at_ inftel.com.mx).
 *   All Rights Reserved (Todos los derechos reservados).
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */
package mx.com.inftel.shiro.oauth2;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import org.apache.shiro.codec.Base64;
import org.json.JSONObject;

/**
 * Google Accounts authenticating filter.
 *
 * @author Santos Zatarain Vera <santoszv@inftel.com.mx>
 */
public class GoogleAccountsAuthenticatingFilter extends AbstractOAuth2AuthenticatingFilter {

    @Override
    protected String getAuthorizeURL(ServletRequest request, ServletResponse response) throws Exception {
        return makeStandardAuthorizeURL(request, response, "https://accounts.google.com/o/oauth2/auth", "openid email");
    }

    @Override
    protected JSONObject getOAuth2Principal(ServletRequest request, ServletResponse response) throws Exception {
        StringBuilder sbToken = new StringBuilder(1024);
        sbToken.append(encodeURL("code"));
        sbToken.append("=");
        sbToken.append(encodeURL(request.getParameter("code")));
        sbToken.append("&");
        sbToken.append(encodeURL("client_id"));
        sbToken.append("=");
        sbToken.append(encodeURL(getClientId()));
        sbToken.append("&");
        sbToken.append(encodeURL("client_secret"));
        sbToken.append("=");
        sbToken.append(encodeURL(getClientSecret()));
        sbToken.append("&");
        sbToken.append(encodeURL("redirect_uri"));
        sbToken.append("=");
        sbToken.append(encodeURL(getRedirectUri()));
        sbToken.append("&");
        sbToken.append(encodeURL("grant_type"));
        sbToken.append("=");
        sbToken.append(encodeURL("authorization_code"));

        byte[] bodyToken = sbToken.toString().getBytes("UTF-8");
        JSONObject joToken;
        URL urlToken = new URL("https://accounts.google.com/o/oauth2/token");

        HttpURLConnection connToken = (HttpURLConnection) urlToken.openConnection();
        try {
            connToken.setUseCaches(false);
            connToken.setInstanceFollowRedirects(false);
            connToken.setDoInput(true);
            connToken.setDoOutput(true);
            connToken.setRequestMethod("POST");
            connToken.setRequestProperty("Content-Type", "application/x-www-form-urlencoded; charset=UTF-8");
            connToken.setRequestProperty("Content-Length", Integer.toString(bodyToken.length));
            connToken.setRequestProperty("Accept-Charset", "UTF-8");

            OutputStream outputStream = connToken.getOutputStream();
            outputStream.write(bodyToken);
            outputStream.flush();

            byte[] buffer = new byte[1024];
            int read;
            ByteArrayOutputStream baos = new ByteArrayOutputStream(1024);
            InputStream inputStream = connToken.getInputStream();
            while ((read = inputStream.read(buffer)) != -1) {
                baos.write(buffer, 0, read);
            }

            joToken = new JSONObject(baos.toString("UTF-8"));
        } catch (IOException ex) {
            throw new OAuth2AuthenticationException(ex.getMessage(), ex);
        } finally {
            connToken.disconnect();
        }

        String[] jwt = joToken.getString("id_token").split("\\.");
        StringBuilder idToken = new StringBuilder(jwt[1].length() + 4);
        idToken.append(jwt[1].replace("-", "+").replace("_", "/"));
        for (int i = 0, n = 4 - (idToken.length() % 4); i < n; i++) {
            idToken.append("=");
        }

        return new JSONObject(Base64.decodeToString(idToken.toString()));
    }

    @Override
    protected String getOAuth2Credentials(JSONObject principal) throws Exception {
        return principal.getString("email");
    }
}
