/*
 *   Copyright 2013 INFTEL - Proveedor de Servicios de Aplicación
 *   (www.inftel.com.mx). All Rights Reserved (Todos los derechos reservados).
 * 
 *   Copyright 2013 Santos Zatarain Vera (santoszv _at_ inftel.com.mx).
 *   All Rights Reserved (Todos los derechos reservados).
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */
package mx.com.inftel.shiro.oauth2;

import org.apache.shiro.authc.AuthenticationToken;
import org.json.JSONObject;

/**
 * OAuth 2.0 token. This token class should be used during login process by
 * filters subclassing {@code AbstractOAuth2AuthenticatingFilter}.
 *
 * The principal should be a JSON object returned by some web endpoint for
 * querying the profile of authenticated resource owner (not the authorized
 * client).
 *
 * The credentials should be some information well-known by the resource owner
 * and the authorized client like the account email.
 *
 * This token class should include the name of filter where login process is
 * taking place.
 *
 * @author Santos Zatarain Vera <santoszv@inftel.com.mx>
 */
public class OAuth2AuthenticationToken implements AuthenticationToken {

    private JSONObject principal;
    private String credentials;
    private String filter;

    /**
     * Default constructor.
     */
    public OAuth2AuthenticationToken() {
    }

    /**
     * Parametrized constructor.
     *
     * @param principal The profile information.
     * @param credentials The well-known credentials.
     * @param filter The filter used to login.
     */
    public OAuth2AuthenticationToken(JSONObject principal, String credentials, String filter) {
        this.principal = principal;
        this.credentials = credentials;
        this.filter = filter;
    }

    /**
     * Returns the principal.
     *
     * @return A JSON object with profile information.
     */
    @Override
    public JSONObject getPrincipal() {
        return principal;
    }

    /**
     * Adjust the principal.
     *
     * @param principal A JSON object with profile information.
     */
    public void setPrincipal(JSONObject principal) {
        this.principal = principal;
    }

    /**
     * Return the well-known credentals by resource owner and authorized client.
     *
     * @return The well-kown credentials.
     */
    @Override
    public String getCredentials() {
        return credentials;
    }

    /**
     * Adjust the well-known credentials by resource owner and authorized client.
     *
     * @param credentials The well-known credentials.
     */
    public void setCredentials(String credentials) {
        this.credentials = credentials;
    }

    /**
     * Return filter's name where login process is taking place.
     *
     * @return Filter's name.
     */
    public String getFilter() {
        return filter;
    }

    /**
     * Adjust filter's name where login is taling place.
     *
     * @param filter Filter's name.
     */
    public void setFilter(String filter) {
        this.filter = filter;
    }
}
