/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.streams.processor.internals.ChangelogReader;
import org.apache.kafka.streams.processor.internals.StateRestorer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StoreChangelogReader
implements ChangelogReader {
    private static final Logger log = LoggerFactory.getLogger(StoreChangelogReader.class);
    private final String logPrefix;
    private final Consumer<byte[], byte[]> consumer;
    private final Map<TopicPartition, Long> endOffsets = new HashMap<TopicPartition, Long>();
    private final Map<String, List<PartitionInfo>> partitionInfo = new HashMap<String, List<PartitionInfo>>();
    private final Map<TopicPartition, StateRestorer> stateRestorers = new HashMap<TopicPartition, StateRestorer>();
    private final Map<TopicPartition, StateRestorer> needsRestoring = new HashMap<TopicPartition, StateRestorer>();
    private final Map<TopicPartition, StateRestorer> needsInitializing = new HashMap<TopicPartition, StateRestorer>();

    public StoreChangelogReader(String threadId, Consumer<byte[], byte[]> consumer) {
        this.consumer = consumer;
        this.logPrefix = String.format("stream-thread [%s]", threadId);
    }

    public StoreChangelogReader(Consumer<byte[], byte[]> consumer) {
        this("", consumer);
    }

    @Override
    public void register(StateRestorer restorer) {
        this.stateRestorers.put(restorer.partition(), restorer);
        this.needsInitializing.put(restorer.partition(), restorer);
    }

    @Override
    public Collection<TopicPartition> restore() {
        if (!this.needsInitializing.isEmpty()) {
            this.initialize();
        }
        if (this.needsRestoring.isEmpty()) {
            this.consumer.assign(Collections.emptyList());
            return this.completed();
        }
        HashSet<TopicPartition> partitions = new HashSet<TopicPartition>(this.needsRestoring.keySet());
        ConsumerRecords allRecords = this.consumer.poll(10L);
        for (TopicPartition partition : partitions) {
            this.restorePartition((ConsumerRecords<byte[], byte[]>)allRecords, partition);
        }
        if (this.needsRestoring.isEmpty()) {
            this.consumer.assign(Collections.emptyList());
        }
        return this.completed();
    }

    private void initialize() {
        if (!this.consumer.subscription().isEmpty()) {
            throw new IllegalStateException("Restore consumer should not be subscribed to any topics (" + this.consumer.subscription() + ")");
        }
        this.refreshChangelogInfo();
        HashMap<TopicPartition, StateRestorer> initializable = new HashMap<TopicPartition, StateRestorer>();
        for (Map.Entry<TopicPartition, StateRestorer> entry : this.needsInitializing.entrySet()) {
            TopicPartition topicPartition = entry.getKey();
            if (!this.hasPartition(topicPartition)) continue;
            initializable.put(entry.getKey(), entry.getValue());
        }
        try {
            this.endOffsets.putAll(this.consumer.endOffsets(initializable.keySet()));
        }
        catch (TimeoutException e) {
            log.debug("{} Could not fetch end offset for {}; will fall back to partition by partition fetching", (Object)this.logPrefix, initializable);
            return;
        }
        Iterator iter = initializable.keySet().iterator();
        while (iter.hasNext()) {
            TopicPartition topicPartition = (TopicPartition)iter.next();
            Long endOffset = this.endOffsets.get(topicPartition);
            if (endOffset != null) {
                StateRestorer restorer = this.needsInitializing.get(topicPartition);
                if (restorer.checkpoint() >= endOffset) {
                    restorer.setRestoredOffset(restorer.checkpoint());
                    iter.remove();
                } else if (restorer.offsetLimit() == 0L || endOffset == 0L) {
                    restorer.setRestoredOffset(0L);
                    iter.remove();
                }
                this.needsInitializing.remove(topicPartition);
                continue;
            }
            log.info("{} End offset cannot be found form the returned metadata; removing this partition from the current run loop", (Object)this.logPrefix);
            iter.remove();
        }
        if (!initializable.isEmpty()) {
            this.startRestoration(initializable);
        }
    }

    private void startRestoration(Map<TopicPartition, StateRestorer> initialized) {
        log.debug("{} Start restoring state stores from changelog topics {}", (Object)this.logPrefix, initialized.keySet());
        HashSet<TopicPartition> assignment = new HashSet<TopicPartition>(this.consumer.assignment());
        assignment.addAll(initialized.keySet());
        this.consumer.assign(assignment);
        ArrayList<StateRestorer> needsPositionUpdate = new ArrayList<StateRestorer>();
        for (StateRestorer restorer : initialized.values()) {
            if (restorer.checkpoint() != -1L) {
                this.consumer.seek(restorer.partition(), restorer.checkpoint());
                this.logRestoreOffsets(restorer.partition(), restorer.checkpoint(), this.endOffsets.get(restorer.partition()));
                restorer.setStartingOffset(this.consumer.position(restorer.partition()));
                continue;
            }
            this.consumer.seekToBeginning(Collections.singletonList(restorer.partition()));
            needsPositionUpdate.add(restorer);
        }
        for (StateRestorer restorer : needsPositionUpdate) {
            long position = this.consumer.position(restorer.partition());
            this.logRestoreOffsets(restorer.partition(), position, this.endOffsets.get(restorer.partition()));
            restorer.setStartingOffset(position);
        }
        this.needsRestoring.putAll(initialized);
    }

    private void logRestoreOffsets(TopicPartition partition, long startingOffset, Long endOffset) {
        log.debug("{} Restoring partition {} from offset {} to endOffset {}", new Object[]{this.logPrefix, partition, startingOffset, endOffset});
    }

    private Collection<TopicPartition> completed() {
        HashSet<TopicPartition> completed = new HashSet<TopicPartition>(this.stateRestorers.keySet());
        completed.removeAll(this.needsRestoring.keySet());
        log.trace("{} completed partitions {}", (Object)this.logPrefix, completed);
        return completed;
    }

    private void refreshChangelogInfo() {
        try {
            this.partitionInfo.putAll(this.consumer.listTopics());
        }
        catch (TimeoutException e) {
            log.debug("{} Could not fetch topic metadata within the timeout, will retry in the next run loop", (Object)this.logPrefix);
        }
    }

    @Override
    public Map<TopicPartition, Long> restoredOffsets() {
        HashMap<TopicPartition, Long> restoredOffsets = new HashMap<TopicPartition, Long>();
        for (Map.Entry<TopicPartition, StateRestorer> entry : this.stateRestorers.entrySet()) {
            StateRestorer restorer = entry.getValue();
            if (!restorer.isPersistent()) continue;
            restoredOffsets.put(entry.getKey(), restorer.restoredOffset());
        }
        return restoredOffsets;
    }

    @Override
    public void reset() {
        this.partitionInfo.clear();
        this.stateRestorers.clear();
        this.needsRestoring.clear();
        this.endOffsets.clear();
        this.needsInitializing.clear();
    }

    private void restorePartition(ConsumerRecords<byte[], byte[]> allRecords, TopicPartition topicPartition) {
        StateRestorer restorer = this.stateRestorers.get(topicPartition);
        Long endOffset = this.endOffsets.get(topicPartition);
        long pos = this.processNext(allRecords.records(topicPartition), restorer, endOffset);
        restorer.setRestoredOffset(pos);
        if (restorer.hasCompleted(pos, endOffset)) {
            if (pos > endOffset + 1L) {
                throw new IllegalStateException(String.format("Log end offset of %s should not change while restoring: old end offset %d, current offset %d", topicPartition, endOffset, pos));
            }
            log.debug("{} Completed restoring state from changelog {} with {} records ranging from offset {} to {}", new Object[]{this.logPrefix, topicPartition, restorer.restoredNumRecords(), restorer.startingOffset(), restorer.restoredOffset()});
            this.needsRestoring.remove(topicPartition);
        }
    }

    private long processNext(List<ConsumerRecord<byte[], byte[]>> records, StateRestorer restorer, Long endOffset) {
        for (ConsumerRecord<byte[], byte[]> record : records) {
            long offset = record.offset();
            if (restorer.hasCompleted(offset, endOffset)) {
                return offset;
            }
            if (record.key() == null) continue;
            restorer.restore((byte[])record.key(), (byte[])record.value());
        }
        return this.consumer.position(restorer.partition());
    }

    private boolean hasPartition(TopicPartition topicPartition) {
        List<PartitionInfo> partitions = this.partitionInfo.get(topicPartition.topic());
        if (partitions == null) {
            return false;
        }
        for (PartitionInfo partition : partitions) {
            if (partition.partition() != topicPartition.partition()) continue;
            return true;
        }
        return false;
    }
}

