package com.xdja.poc.common.view.bezier;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PointF;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;

import com.xdja.poc.common.utils.LogUtils;

/**
 * Created by gouhao on 12/21/2017.
 */

public class BezierLine extends View {
    private static final String TAG = BezierLine.class.getSimpleName();

    private static final int LINE_COUNT = 4;
    private Paint paint;
    private CustomValueAnimator[] customValueAnimators;
    private int[] durationArray = new int[]{1500, 3000, 5000, 2000};
    private PointF startPoint, controlPoint1, endPoint1, controlPoint2, endPoint2;
    private Path path;
    private int[] offset;
    private boolean isStart;
    private WorkerThread workerThread;

    public BezierLine(Context context) {
        this(context, null);
    }

    public BezierLine(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        paint = new Paint(Paint.ANTI_ALIAS_FLAG);
        paint.setStyle(Paint.Style.STROKE);
        paint.setColor(Color.WHITE);

        startPoint = new PointF();
        controlPoint1 = new PointF();
        controlPoint2 = new PointF();
        endPoint1 = new PointF();
        endPoint2 = new PointF();
        path = new Path();
        offset = new int[LINE_COUNT];
        customValueAnimators = new CustomValueAnimator[LINE_COUNT];
        for(int i = 0; i < LINE_COUNT; i++) {
            customValueAnimators[i] = new CustomValueAnimator();
            customValueAnimators[i].setDuration(durationArray[i]);
            customValueAnimators[i].setUpdateTime(UPDATE_TIME);
        }
        workerThread = new WorkerThread();
    }

    @Override
    public void postInvalidate() {
        synchronized (this) {
            if(isStart) {
                for(int i = 0; i < LINE_COUNT; i++) {
                    offset[i] = customValueAnimators[i].getValue();
                }
            }
        }

        super.postInvalidate();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        synchronized (this) {
            if(getVisibility() == VISIBLE && !isStart) {
                isStart = true;
                for(int i = 0; i < LINE_COUNT; i++) {
                    customValueAnimators[i].setEndValue(getMeasuredWidth());
                    customValueAnimators[i].start();
                }
                workerThread.start();
            }
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        canvas.drawColor(Color.TRANSPARENT);
        int height = getHeight();
        int width = getWidth();
        int deltaX = width / 4;
        int deltaY = height / 8;
        int lineHeight = height / 2;
        int curDeltaY = height / 2;
        path.reset();
        for(int i = 0; i < 4; i++) {

            if(i == 0) {
                paint.setAlpha(229);
            } else if(i == 1) {
                paint.setAlpha(153);
            } else if (i == 2){
                paint.setAlpha(76);
            } else {
                paint.setAlpha(50);
            }

            if(i == 0) {
                lineHeight += deltaY;
            } else if(i == 1) {
                lineHeight -= deltaY;
            } else {
                lineHeight += deltaY;
            }
            startPoint.set(offset[i], lineHeight);
            controlPoint1.set(startPoint.x - deltaX, startPoint.y + curDeltaY);
            endPoint1.set(controlPoint1.x - deltaX, startPoint.y);
            controlPoint2.set(endPoint1.x - deltaX, startPoint.y - curDeltaY);
            endPoint2.set(controlPoint2.x - deltaX, startPoint.y);

            path.moveTo(startPoint.x, startPoint.y);
            path.quadTo(controlPoint1.x, controlPoint1.y, endPoint1.x, endPoint1.y);
            path.moveTo(endPoint1.x, endPoint1.y);
            path.quadTo(controlPoint2.x, controlPoint2.y, endPoint2.x, endPoint2.y);
            canvas.drawPath(path, paint);

            controlPoint1.set(startPoint.x + deltaX, startPoint.y - curDeltaY);
            endPoint1.set(controlPoint1.x + deltaX, startPoint.y);
            controlPoint2.set(endPoint1.x + deltaX, startPoint.y + curDeltaY);
            endPoint2.set(controlPoint2.x + deltaX, startPoint.y);

            path.moveTo(startPoint.x, startPoint.y);
            path.quadTo(controlPoint1.x, controlPoint1.y, endPoint1.x, endPoint1.y);
            path.moveTo(endPoint1.x, endPoint1.y);
            path.quadTo(controlPoint2.x, controlPoint2.y, endPoint2.x, endPoint2.y);
            canvas.drawPath(path, paint);
        }
    }

    @Override
    protected void onDetachedFromWindow() {
        synchronized (this) {
            isStart = false;
        }
        super.onDetachedFromWindow();
    }

    private static final int UPDATE_TIME = 30;

    private class WorkerThread extends Thread{
        @Override
        public void run() {
                while (true){
                    try {
                        Thread.sleep(UPDATE_TIME);
                        synchronized (BezierLine.this) {
                            if(!isStart) {
                                LogUtils.ILog(TAG, "thread stop");
                                break;
                            }
                        }
                        postInvalidate();
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                }
        }
    }
}
