package com.xdja.poc.sdk.record.http;

import android.support.annotation.Nullable;

import com.xdja.poc.common.utils.LogUtils;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import okhttp3.MediaType;
import okhttp3.RequestBody;
import okio.BufferedSink;

/**
 * Created by gouhao on 10/25/2018.
 */

@SuppressWarnings("ALL")
public class UploadFileRequestBody extends RequestBody {
    private static final String TAG = UploadFileRequestBody.class.getSimpleName();

    //在内网测试，52K的buffer在相同网络情况下上传较快
    private static final int DEFAULT_READ_LENGTH = 52 * 1024;

    private IHttpClient.UploadFileCallback callback;
    private File file;
    private long fileSize;
    private long endIndex;
    private volatile boolean canceled;
    private long writeLength;
    private int requestId;
    private long trunkLength;

    public UploadFileRequestBody(int requestId, File file, long startIndex, long blockSize, IHttpClient.UploadFileCallback callback) {
        this.requestId = requestId;
        this.callback = callback;
        this.file = file;
        fileSize = file.length();
        this.writeLength = startIndex;
        long blockIndex = startIndex + blockSize;
        endIndex = Math.min(blockIndex, fileSize);
        trunkLength = getTrunkLength(startIndex,blockSize,fileSize);
        LogUtils.DLog(TAG, "UploadFileRequestBody: startIndex: " + startIndex + ", blockSize: " + blockSize + ", endIndex: " + endIndex);
    }

    @Nullable
    @Override
    public MediaType contentType() {
        return MediaType.parse(BaseHttpClient.TYPE_UPLOAD_FILE);
    }

    @Override
    public long contentLength() throws IOException {
        //如果没传完，采用chunked上传，否则contentLength为0
//        return trunkLength;
        return -1;
    }

    @Override
    public void writeTo(BufferedSink sink) throws IOException {
        BufferedInputStream bis = null;
        try {
            bis = new BufferedInputStream(new FileInputStream(file));
            bis.skip(writeLength);

            byte[] bytes = new byte[DEFAULT_READ_LENGTH];
            int readLength;
            int shouldReadLength = DEFAULT_READ_LENGTH;
            while (writeLength < endIndex && (readLength = bis.read(bytes,0, shouldReadLength)) > 0) {
                sink.buffer().write(bytes, 0, readLength);
                sink.flush();
                writeLength += readLength;
                if(callback != null){
                    callback.onProgress(requestId, writeLength, fileSize);
                }
                if(canceled){
                    LogUtils.DLog(TAG, "writeTo: canceled upload");
                    break;
                }
                shouldReadLength = (int)Math.min(endIndex - writeLength, DEFAULT_READ_LENGTH);
            }
        } catch (Exception e){
            LogUtils.ELog(TAG, "writeTo: error: " + e);
            e.printStackTrace();
        }finally {
            if(bis != null){
                try {
                    bis.close();
                }catch (Exception e){
                    LogUtils.ELog(TAG, "writeTo: closeFileError: " + e);
                    e.printStackTrace();
                }

            }
        }
    }

    public void setCanceled(boolean canceled) {
        this.canceled = canceled;
    }

    /**
     *  * 获取每次分片的大小
     *  * @param startIndex 起始读取位置
     *  * @param blockSize 块大小
     *  * @param fileSize 文件大小
     *  * @return
     *  
     */
    private long getTrunkLength(long startIndex, long blockSize, long fileSize) {
        if (fileSize < blockSize) {
            //文件大小没到blockSize
            return fileSize;
        }
        long difference = fileSize - startIndex;
        if (difference >= blockSize) {
            //分片
            return blockSize;
        }
        //最后一片
        return difference;
    }
}
