package com.xdja.poc.sdk.record.fdfs;

import android.os.Binder;
import android.os.IBinder;
import android.support.annotation.Keep;
import android.text.TextUtils;
import android.util.ArrayMap;
import android.util.SparseArray;

import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;
import com.xdja.poc.common.utils.LogUtils;
import com.xdja.poc.sdk.config.Constants;
import com.xdja.poc.sdk.record.http.HttpClientFactory;
import com.xdja.poc.sdk.record.http.HttpRequest;
import com.xdja.poc.sdk.record.http.HttpUploadFileRequest;
import com.xdja.poc.sdk.record.http.HttpsUtils;
import com.xdja.poc.sdk.record.http.IHttpClient;
import com.xdja.poc.sdk.utils.IPHost;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Calendar;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.Semaphore;

/**
 * Created by gouhao on 10/25/2018.
 */
@SuppressWarnings("ALL")
@Keep
public class FdfsService extends BaseCoreService {
    private static final String TAG = FdfsService.class.getSimpleName();


    public static final String SERVERTIMESTAMP = "server_timestamp";
    public static final String BOOTTIMESTAMP = "boot_timestamp";
    /**
     * FDFS规定一个连接最大上传10M数据。我们每个块最大上传8M
     */
    public static final long DEFAULT_BLOCK_SIZE = 8 * 1024 * 1024;

    private static final String HEADER_FILENAME = "filename";
    private static final String HEADER_PERM = "perm";
    private static final String HEADER_FILESIZE = "filesize";
    private static final String HEADER_FILEID = "fileid";
    public static final String PARAM_USERID = "userid=";
    public static final String PARAM_TS = "ts=";
    public static final String PARAM_SIGN = "sign=";
    public static final String PARAM_TYPE = "mvdp=";
    public static final String PARAM_MARK = "&";

    /**
     * 查询info错误
     */
    public static final int E_GET_INFO_ERROR = 3;

    private FdfsUploadBinder binder;
    private IHttpClient httpClient;
    private Map<Long, UploadData> uploadDataMap;
    private SparseArray<UploadData> uploadingCacheDataMap;
    private IHttpClient.UploadFileCallback uploadFileCallback;
    private Gson gson;

    //最大同时上传三个
    private static final int MAX_UPLOAD_THREAD = 3;
    private Semaphore semaphore;

    @Override
    public void onCreate() {
        super.onCreate();
        binder = new FdfsUploadBinder();
        httpClient = HttpClientFactory.createHttpClient();
        uploadDataMap = new ConcurrentHashMap<>();
        uploadFileCallback = new UploadFileCallback();
        uploadingCacheDataMap = new SparseArray<>();
        semaphore = new Semaphore(MAX_UPLOAD_THREAD);
        gson = new Gson();
    }

    private class FdfsUploadBinder extends Binder implements IFdfsService {
        @Override
        public int upload(long uploadId, String filePath, UploadCallback callback) {
            return uploadInner(uploadId, filePath, null, callback);
        }

        @Override
        public int upload(long uploadId, String filePath, String fileId, UploadCallback callback) {
            return uploadInner(uploadId, filePath, fileId, callback);
        }

        @Override
        public void stopUpload(long uploadId) {
            stopInner(uploadId);
        }
    }

    @SuppressWarnings("CanBeFinal")
    private class UploadTask implements Runnable {
        private long uploadId;
        private String filePath;
        private String fileId;
        private UploadCallback callback;

        public UploadTask(long uploadId, String filePath, String fileId, UploadCallback callback) {
            this.uploadId = uploadId;
            this.filePath = filePath;
            this.fileId = fileId;
            this.callback = callback;
        }

        @Override
        public void run() {
            try {
                if (TextUtils.isEmpty(filePath)) {
                    LogUtils.DLog(TAG, "UploadTask: filePath is empty");
                    sendErrorMessage(UPLOAD_E_PATH_EMPTY);
                    return;
                }
                File file = new File(filePath);
                if (!file.exists()) {
                    LogUtils.DLog(TAG, "UploadTask: file not found");
                    sendErrorMessage(UPLOAD_E_FILE_NOT_EXISTS);
                    return;
                }

                //开始上传
                semaphore.acquire();

                UploadItem uploadItem = new UploadItem(filePath, file.length());
                uploadItem.setId(uploadId);
                UploadData uploadData = new UploadData(uploadItem, callback);
                uploadDataMap.put(uploadId, uploadData);
                LogUtils.DLog(TAG, "UploadTask: uploadId: " + uploadItem.getId() + ", filePath: " + filePath);
                if (TextUtils.isEmpty(fileId)) {
                    realUpload(uploadItem, 0);
                } else {
                    getLastUploadSize(uploadData);
                }
            } catch (InterruptedException e) {
                e.printStackTrace();
                sendErrorMessage(UPLOAD_E_INTERRUPTED);
            }

        }

        private void sendErrorMessage(int code) {
            if (callback != null) {
                callback.onError(uploadId, code);
            }
        }
    }

    public static final int UPLOAD_SUCCESS = 0;
    public static final int UPLOAD_E_PATH_EMPTY = -1;
    public static final int UPLOAD_E_FILE_NOT_EXISTS = -2;
    public static final int UPLOAD_E_INTERRUPTED = -3;

    private int uploadInner(long uploadId, String filePath, String fileId, UploadCallback callback) {
        try {
            ThreadPoolManager.getInstance().execute(new UploadTask(uploadId, filePath, fileId, callback));
        } catch (RejectedExecutionException e) {
            e.printStackTrace();
            LogUtils.DLog(TAG, "uploadInner: error: " + e);
        }
        return 0;
    }

    private void getLastUploadSize(final UploadData data) {
        UploadItem uploadItem = data.uploadItem;
        HttpRequest request = new HttpRequest(IPHost.getFastDfsUrl() + "/info/" + uploadItem.getFileId());
        request.setMethod(IHttpClient.METHOD_TYPE_GET);
        httpClient.request(request, new IHttpClient.Callback() {
            @Override
            public void onSuccess(int requestId, String response) {
                LogUtils.DLog(TAG, "getLastUploadSize: onSuccess:" + response);
                FdfsResponse fdfsResponse = gson.fromJson(response, FdfsResponse.class);
                if (fdfsResponse.size == uploadItem.getSize()) {
                    uploadItem.setStatus(UploadItem.STATUS_SUCCESS);
                    semaphore.release();
                    if (data.uploadCallback != null) {
                        data.uploadCallback.onSuccess(uploadItem.getId(), fdfsResponse.getFileId());
                    }
                } else {
                    realUpload(uploadItem, fdfsResponse.size);
                }
            }

            @Override
            public void onFailed(int requestId, int code, String msg) {
                LogUtils.ELog(TAG, "getLastUploadSize: onFailed: requestId: " + requestId + ", code: " + code + ", msg: " + msg);
                UploadData data = uploadDataMap.get(uploadItem.getId());
                semaphore.release();
                if (data != null && data.uploadCallback != null) {
                    data.uploadCallback.onError(uploadItem.getId(), E_GET_INFO_ERROR);
                }
            }
        });
    }

    private void realUpload(UploadItem uploadItem, long lastUploadSize) {
        StringBuilder sb = new StringBuilder(IPHost.getFastDfsUrl() + "/trunkupload");

        long timeStamp;
//        long serverTimestamp = PreferenceUtils.getPrefLong(MyApp.getAppContext(),
//                SERVERTIMESTAMP, 0);
//        if (serverTimestamp == 0){
        //构造时间戳
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(System.currentTimeMillis());
        // TODO: 2018/5/21 根据需求修改过期时间戳
        calendar.add(Calendar.MINUTE, 10);
        timeStamp = calendar.getTimeInMillis();
//        }else {
//            long bootTimestamp = PreferenceUtils.getPrefLong(MyApp.getAppContext(),
//                    BOOTTIMESTAMP, 0);
//            long curBootTime = SystemClock.elapsedRealtime();
//            long timeGap = curBootTime - bootTimestamp;
//            timeStamp = serverTimestamp + timeGap + 10 * 60 * 1000;
//        }


        //生成签名字符串
        String originalStr = IPHost.getFastDfsAppId() + timeStamp;
        String sign = HttpsUtils.getSignature(IPHost.getFastDfsSecret(), originalStr);
        sb.append("?" + PARAM_USERID).append(IPHost.getFastDfsAppId()).append("&" + PARAM_TS).append(timeStamp).append("&" + PARAM_SIGN).append(sign);
        String url = sb.toString();
        String filePath = uploadItem.getFilePath();
        File file = new File(filePath);
        HttpUploadFileRequest request = new HttpUploadFileRequest(url, filePath);
        request.setLastUploadSize(lastUploadSize);
        request.setBlockSize(DEFAULT_BLOCK_SIZE);
        Map<String, String> headers = new ArrayMap<>();
        try {
            //fdfs规定：必须使用URLEncoder.encode进行utf8编码，
            // 防止中文乱码,编码前的长度不能超过256
            String fileName = file.getName();
            if (fileName.length() > 256) {
                fileName = fileName.substring(0, 256);
            }
            headers.put(HEADER_FILENAME, URLEncoder.encode(fileName, "UTF-8"));
            headers.put(HEADER_PERM, "1");
            headers.put(HEADER_FILESIZE, String.valueOf(file.length()));
            if (!TextUtils.isEmpty(uploadItem.getFileId())) {
                headers.put(HEADER_FILEID, uploadItem.getFileId());
            }
            request.setHeaders(headers);
            LogUtils.DLog(TAG, "realUpload: uploadId: " + uploadItem.getId());
            UploadData data = uploadDataMap.get(uploadItem.getId());
            uploadItem.setStatus(UploadItem.STATUS_UPLOADING);
            final int requestId = httpClient.uploadFile(request, uploadFileCallback);
            if (data == null) {
                //bug:45259:data可能等于NULL，有可能是并发引起的
                LogUtils.ELog(TAG, "realUpload: data is null");
            } else {
                data.requestId = requestId;
            }

            LogUtils.DLog(TAG, "requestId: " + requestId);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            UploadData data = uploadDataMap.get(uploadItem.getId());
            semaphore.release();
            if (data.uploadCallback != null) {
                data.uploadCallback.onError(uploadItem.getId(), -1);
            }
        }

    }

    private class UploadFileCallback implements IHttpClient.UploadFileCallback {
        @Override
        public void onProgress(int requestId, long uploadSize, long totalSize) {
            handleUploadFileCallback(OP_PROGRESS, requestId, String.valueOf(uploadSize), String.valueOf(totalSize));
        }

        @Override
        public void onSuccess(int requestId, String response) {
            handleUploadFileCallback(OP_SUCCESS, requestId, response);
        }

        @Override
        public void onFailed(int requestId, int code, String msg) {
            handleUploadFileCallback(OP_FAILED, requestId, String.valueOf(code), msg);
        }
    }

    private static final int OP_PROGRESS = 1;
    private static final int OP_SUCCESS = 2;
    private static final int OP_FAILED = 3;

    private void handleUploadFileCallback(int op, int requestId, String... params) {
        UploadData data = uploadingCacheDataMap.get(requestId);
        if (data == null) {
            data = findUploadDataByRequestId(requestId);
            uploadingCacheDataMap.put(requestId, data);
        }
        if (data == null) {
            return;
        }
        if (data.uploadItem.getStatus() == UploadItem.STATUS_CANCELING && op == OP_PROGRESS) {
            uploadingCacheDataMap.delete(requestId);
            LogUtils.DLog(TAG, "handleUploadFileCallback: requestId: " + requestId + " canceled");
            return;
        }
        if (op == OP_PROGRESS) {
            handleUploadProgress(data.uploadItem, data.uploadCallback, Long.parseLong(params[0]), Long.parseLong(params[1]));
        } else if (op == OP_SUCCESS) {
            uploadingCacheDataMap.delete(requestId);
            handleUploadSuccess(data.uploadItem, data.uploadCallback, params[0]);
        } else if (op == OP_FAILED) {
            uploadingCacheDataMap.delete(requestId);
            handleUploadFailed(data.uploadItem, data.uploadCallback, Integer.parseInt(params[0]), params[1]);
        }
    }

    private void handleUploadFailed(UploadItem uploadItem, UploadCallback callback, int errorCode, String msg) {
        LogUtils.ELog(TAG, "onFailed: code: " + errorCode + ", msg: " + msg);
        uploadItem.setStatus(UploadItem.STATUS_FAILED);
        semaphore.release();
        if (callback != null) {
            if (msg != null && msg.contains(Constants.URL_TIMESTAMP_OVER_DEADLINE)) {
                errorCode = Constants.CODE_UPLOAD_FILE_TIME_EXCEPTION;
            }
            callback.onError(uploadItem.getId(), errorCode);
        }
    }

    private void handleUploadSuccess(UploadItem uploadItem, UploadCallback callback, String response) {
        LogUtils.DLog(TAG, "handleUploadSuccess: " + response);
        FdfsResponse fdfsResponse = gson.fromJson(response, FdfsResponse.class);

        //先保存数据，再执行之后的操作
        uploadItem.setFileId(fdfsResponse.getFileId());
        uploadItem.setUploadSize(fdfsResponse.size);
        int lastStatus = uploadItem.getStatus();
        boolean uploadFinish = uploadItem.getSize() == uploadItem.getUploadSize();
        if (lastStatus == UploadItem.STATUS_CANCELING) {
            uploadItem.setStatus(UploadItem.STATUS_CANCELED);
        } else if (uploadFinish) {
            uploadItem.setStatus(UploadItem.STATUS_SUCCESS);
        }

        if (lastStatus == UploadItem.STATUS_UPLOADING) {
            if (uploadFinish) {
                semaphore.release();
                if (callback != null) {
                    callback.onSuccess(uploadItem.getId(), uploadItem.getFileId());
                }
            } else {
                realUpload(uploadItem, uploadItem.getUploadSize());
            }
        }
    }

    private void handleUploadProgress(UploadItem uploadItem, UploadCallback callback, long uploadSize, long totalSize) {
        LogUtils.DLog(TAG, "handleUploadProgress: uploadSize: " + uploadSize + ", totalSize: " + totalSize);
        uploadItem.setUploadSize(uploadSize);
        if (callback != null) {
            callback.onProgress(uploadItem.getId(), uploadSize, totalSize);
        }
    }

    private UploadData findUploadDataByRequestId(int requestId) {
        for (Map.Entry<Long, UploadData> longUploadDataEntry : uploadDataMap.entrySet()) {
            UploadData data = longUploadDataEntry.getValue();
            if (data.requestId == requestId) {
                return data;
            }
        }
        return null;
    }

    private void stopInner(long uploadId) {
        LogUtils.DLog(TAG, "stopInner: uploadId: " + uploadId);
        UploadData data = uploadDataMap.get(uploadId);
        if (data == null) {
            LogUtils.ELog(TAG, "stopInner: uploadId: " + uploadId + " data is null");
            return;
        }
        //这里不保存数据库,会在服务器响应之后保存到数据库
        data.uploadItem.setStatus(UploadItem.STATUS_CANCELING);
        httpClient.cancelRequest(data.requestId);
    }

    @Override
    public IBinder getBinder() {
        return binder;
    }

    public interface UploadCallback {
        void onProgress(long uploadId, long uploadSize, long totalSize);

        void onSuccess(long uploadId, String fileId);

        void onError(long uploadId, int errorCode);
    }

    @SuppressWarnings("CanBeFinal")
    private class UploadData {
        private UploadItem uploadItem;
        private UploadCallback uploadCallback;
        private int requestId;

        public UploadData(UploadItem uploadItem, UploadCallback uploadCallback) {
            this.uploadItem = uploadItem;
            this.uploadCallback = uploadCallback;
        }
    }

    @Keep
    private class FdfsResponse {
        private long size;
        @SerializedName(HEADER_FILEID)
        private String fileId;

        public String getFileId() {
            if (!TextUtils.isEmpty(fileId)) {
                return fileId.replace("\\", "").replace("}", "")
                        .replace("\"", "");
            }
            return fileId;
        }
    }
}
