/*
 * Decompiled with CFR 0.152.
 */
package com.xdja.poc.sdk.business.webrtc.apprtc;

import android.annotation.TargetApi;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Build;
import android.os.SystemClock;
import com.xdja.poc.common.utils.LogUtils;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Scanner;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

@TargetApi(value=19)
class CpuMonitor {
    private static final String TAG = "CpuMonitor";
    private static final int MOVING_AVERAGE_SAMPLES = 5;
    private static final int CPU_STAT_SAMPLE_PERIOD_MS = 2000;
    private static final int CPU_STAT_LOG_PERIOD_MS = 6000;
    private final Context appContext;
    private final MovingAverage userCpuUsage;
    private final MovingAverage systemCpuUsage;
    private final MovingAverage totalCpuUsage;
    private final MovingAverage frequencyScale;
    private ScheduledExecutorService executor;
    private long lastStatLogTimeMs;
    private long[] cpuFreqMax;
    private int cpusPresent;
    private int actualCpusPresent;
    private boolean initialized;
    private boolean cpuOveruse;
    private String[] maxPath;
    private String[] curPath;
    private double[] curFreqScales;
    private ProcStat lastProcStat;

    public static boolean isSupported() {
        return Build.VERSION.SDK_INT >= 19 && Build.VERSION.SDK_INT < 24;
    }

    public CpuMonitor(Context context) {
        if (!CpuMonitor.isSupported()) {
            throw new RuntimeException("CpuMonitor is not supported on this Android version.");
        }
        LogUtils.DLog((String)TAG, (String)"CpuMonitor ctor.");
        this.appContext = context.getApplicationContext();
        this.userCpuUsage = new MovingAverage(5);
        this.systemCpuUsage = new MovingAverage(5);
        this.totalCpuUsage = new MovingAverage(5);
        this.frequencyScale = new MovingAverage(5);
        this.lastStatLogTimeMs = SystemClock.elapsedRealtime();
        this.scheduleCpuUtilizationTask();
    }

    public void pause() {
        if (this.executor != null) {
            LogUtils.DLog((String)TAG, (String)"pause");
            this.executor.shutdownNow();
            this.executor = null;
        }
    }

    public void resume() {
        LogUtils.DLog((String)TAG, (String)"resume");
        this.resetStat();
        this.scheduleCpuUtilizationTask();
    }

    public synchronized void reset() {
        if (this.executor != null) {
            LogUtils.DLog((String)TAG, (String)"reset");
            this.resetStat();
            this.cpuOveruse = false;
        }
    }

    public synchronized int getCpuUsageCurrent() {
        return this.doubleToPercent(this.userCpuUsage.getCurrent() + this.systemCpuUsage.getCurrent());
    }

    public synchronized int getCpuUsageAverage() {
        return this.doubleToPercent(this.userCpuUsage.getAverage() + this.systemCpuUsage.getAverage());
    }

    public synchronized int getFrequencyScaleAverage() {
        return this.doubleToPercent(this.frequencyScale.getAverage());
    }

    private void scheduleCpuUtilizationTask() {
        if (this.executor != null) {
            this.executor.shutdownNow();
            this.executor = null;
        }
        this.executor = Executors.newSingleThreadScheduledExecutor();
        ScheduledFuture<?> possiblyIgnoredError = this.executor.scheduleAtFixedRate(new Runnable(){

            @Override
            public void run() {
                CpuMonitor.this.cpuUtilizationTask();
            }
        }, 0L, 2000L, TimeUnit.MILLISECONDS);
    }

    private void cpuUtilizationTask() {
        boolean cpuMonitorAvailable = this.sampleCpuUtilization();
        if (cpuMonitorAvailable && SystemClock.elapsedRealtime() - this.lastStatLogTimeMs >= 6000L) {
            this.lastStatLogTimeMs = SystemClock.elapsedRealtime();
            String statString = this.getStatString();
            LogUtils.DLog((String)TAG, (String)statString);
        }
    }

    private void init() {
        try (FileInputStream fin = new FileInputStream("/sys/devices/system/cpu/present");
             InputStreamReader streamReader = new InputStreamReader((InputStream)fin, Charset.forName("UTF-8"));
             BufferedReader reader = new BufferedReader(streamReader);
             Scanner scanner = new Scanner(reader).useDelimiter("[-\n]");){
            scanner.nextInt();
            this.cpusPresent = 1 + scanner.nextInt();
            scanner.close();
        }
        catch (FileNotFoundException e) {
            LogUtils.ELog((String)TAG, (String)"Cannot do CPU stats since /sys/devices/system/cpu/present is missing");
        }
        catch (IOException e) {
            LogUtils.ELog((String)TAG, (String)"Error closing file");
        }
        catch (Exception e) {
            LogUtils.ELog((String)TAG, (String)"Cannot do CPU stats due to /sys/devices/system/cpu/present parsing problem");
        }
        this.cpuFreqMax = new long[this.cpusPresent];
        this.maxPath = new String[this.cpusPresent];
        this.curPath = new String[this.cpusPresent];
        this.curFreqScales = new double[this.cpusPresent];
        for (int i = 0; i < this.cpusPresent; ++i) {
            this.cpuFreqMax[i] = 0L;
            this.curFreqScales[i] = 0.0;
            this.maxPath[i] = "/sys/devices/system/cpu/cpu" + i + "/cpufreq/cpuinfo_max_freq";
            this.curPath[i] = "/sys/devices/system/cpu/cpu" + i + "/cpufreq/scaling_cur_freq";
        }
        this.lastProcStat = new ProcStat(0L, 0L, 0L);
        this.resetStat();
        this.initialized = true;
    }

    private synchronized void resetStat() {
        this.userCpuUsage.reset();
        this.systemCpuUsage.reset();
        this.totalCpuUsage.reset();
        this.frequencyScale.reset();
        this.lastStatLogTimeMs = SystemClock.elapsedRealtime();
    }

    private int getBatteryLevel() {
        Intent intent = this.appContext.registerReceiver(null, new IntentFilter("android.intent.action.BATTERY_CHANGED"));
        int batteryLevel = 0;
        int batteryScale = intent.getIntExtra("scale", 100);
        if (batteryScale > 0) {
            batteryLevel = (int)(100.0f * (float)intent.getIntExtra("level", 0) / (float)batteryScale);
        }
        return batteryLevel;
    }

    private synchronized boolean sampleCpuUtilization() {
        ProcStat procStat;
        long lastSeenMaxFreq = 0L;
        long cpuFreqCurSum = 0L;
        long cpuFreqMaxSum = 0L;
        if (!this.initialized) {
            this.init();
        }
        if (this.cpusPresent == 0) {
            return false;
        }
        this.actualCpusPresent = 0;
        for (int i = 0; i < this.cpusPresent; ++i) {
            this.curFreqScales[i] = 0.0;
            if (this.cpuFreqMax[i] == 0L) {
                long cpufreqMax = this.readFreqFromFile(this.maxPath[i]);
                if (cpufreqMax > 0L) {
                    LogUtils.DLog((String)TAG, (String)("Core " + i + ". Max frequency: " + cpufreqMax));
                    lastSeenMaxFreq = cpufreqMax;
                    this.cpuFreqMax[i] = cpufreqMax;
                    this.maxPath[i] = null;
                }
            } else {
                lastSeenMaxFreq = this.cpuFreqMax[i];
            }
            long cpuFreqCur = this.readFreqFromFile(this.curPath[i]);
            if (cpuFreqCur == 0L && lastSeenMaxFreq == 0L) continue;
            if (cpuFreqCur > 0L) {
                ++this.actualCpusPresent;
            }
            cpuFreqCurSum += cpuFreqCur;
            cpuFreqMaxSum += lastSeenMaxFreq;
            if (lastSeenMaxFreq <= 0L) continue;
            this.curFreqScales[i] = (double)cpuFreqCur / (double)lastSeenMaxFreq;
        }
        if (cpuFreqCurSum == 0L || cpuFreqMaxSum == 0L) {
            LogUtils.ELog((String)TAG, (String)"Could not read max or current frequency for any CPU");
            return false;
        }
        double currentFrequencyScale = (double)cpuFreqCurSum / (double)cpuFreqMaxSum;
        if (this.frequencyScale.getCurrent() > 0.0) {
            currentFrequencyScale = (this.frequencyScale.getCurrent() + currentFrequencyScale) * 0.5;
        }
        if ((procStat = this.readProcStat()) == null) {
            return false;
        }
        long diffUserTime = procStat.userTime - this.lastProcStat.userTime;
        long diffSystemTime = procStat.systemTime - this.lastProcStat.systemTime;
        long diffIdleTime = procStat.idleTime - this.lastProcStat.idleTime;
        long allTime = diffUserTime + diffSystemTime + diffIdleTime;
        if (currentFrequencyScale == 0.0 || allTime == 0L) {
            return false;
        }
        this.frequencyScale.addValue(currentFrequencyScale);
        double currentUserCpuUsage = (double)diffUserTime / (double)allTime;
        this.userCpuUsage.addValue(currentUserCpuUsage);
        double currentSystemCpuUsage = (double)diffSystemTime / (double)allTime;
        this.systemCpuUsage.addValue(currentSystemCpuUsage);
        double currentTotalCpuUsage = (currentUserCpuUsage + currentSystemCpuUsage) * currentFrequencyScale;
        this.totalCpuUsage.addValue(currentTotalCpuUsage);
        this.lastProcStat = procStat;
        return true;
    }

    private int doubleToPercent(double d) {
        return (int)(d * 100.0 + 0.5);
    }

    private synchronized String getStatString() {
        StringBuilder stat = new StringBuilder();
        stat.append("CPU User: ").append(this.doubleToPercent(this.userCpuUsage.getCurrent())).append("/").append(this.doubleToPercent(this.userCpuUsage.getAverage())).append(". System: ").append(this.doubleToPercent(this.systemCpuUsage.getCurrent())).append("/").append(this.doubleToPercent(this.systemCpuUsage.getAverage())).append(". Freq: ").append(this.doubleToPercent(this.frequencyScale.getCurrent())).append("/").append(this.doubleToPercent(this.frequencyScale.getAverage())).append(". Total usage: ").append(this.doubleToPercent(this.totalCpuUsage.getCurrent())).append("/").append(this.doubleToPercent(this.totalCpuUsage.getAverage())).append(". Cores: ").append(this.actualCpusPresent);
        stat.append("( ");
        for (int i = 0; i < this.cpusPresent; ++i) {
            stat.append(this.doubleToPercent(this.curFreqScales[i])).append(" ");
        }
        stat.append("). Battery: ").append(this.getBatteryLevel());
        if (this.cpuOveruse) {
            stat.append(". Overuse.");
        }
        return stat.toString();
    }

    private long readFreqFromFile(String fileName) {
        long number = 0L;
        try (FileInputStream stream = new FileInputStream(fileName);
             InputStreamReader streamReader = new InputStreamReader((InputStream)stream, Charset.forName("UTF-8"));
             BufferedReader reader = new BufferedReader(streamReader);){
            String line = reader.readLine();
            number = CpuMonitor.parseLong(line);
        }
        catch (FileNotFoundException fileNotFoundException) {
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return number;
    }

    private static long parseLong(String value) {
        long number = 0L;
        try {
            number = Long.parseLong(value);
        }
        catch (NumberFormatException e) {
            LogUtils.ELog((String)TAG, (String)("parseLong error:" + e.getLocalizedMessage()));
        }
        return number;
    }

    private ProcStat readProcStat() {
        long userTime = 0L;
        long systemTime = 0L;
        long idleTime = 0L;
        try (FileInputStream stream = new FileInputStream("/proc/stat");
             InputStreamReader streamReader = new InputStreamReader((InputStream)stream, Charset.forName("UTF-8"));
             BufferedReader reader = new BufferedReader(streamReader);){
            String line = reader.readLine();
            String[] lines = line.split("\\s+");
            int length = lines.length;
            if (length >= 5) {
                userTime = CpuMonitor.parseLong(lines[1]);
                userTime += CpuMonitor.parseLong(lines[2]);
                systemTime = CpuMonitor.parseLong(lines[3]);
                idleTime = CpuMonitor.parseLong(lines[4]);
            }
            if (length >= 8) {
                userTime += CpuMonitor.parseLong(lines[5]);
                systemTime += CpuMonitor.parseLong(lines[6]);
                systemTime += CpuMonitor.parseLong(lines[7]);
            }
        }
        catch (FileNotFoundException e) {
            LogUtils.ELog((String)TAG, (String)("Cannot open /proc/stat for reading: " + e.getLocalizedMessage()));
            return null;
        }
        catch (Exception e) {
            LogUtils.ELog((String)TAG, (String)("Problems parsing /proc/stat: " + e.getLocalizedMessage()));
            return null;
        }
        return new ProcStat(userTime, systemTime, idleTime);
    }

    private static class MovingAverage {
        private final int size;
        private double sum;
        private double currentValue;
        private double[] circBuffer;
        private int circBufferIndex;

        public MovingAverage(int size) {
            if (size <= 0) {
                throw new AssertionError((Object)"Size value in MovingAverage ctor should be positive.");
            }
            this.size = size;
            this.circBuffer = new double[size];
        }

        public void reset() {
            Arrays.fill(this.circBuffer, 0.0);
            this.circBufferIndex = 0;
            this.sum = 0.0;
            this.currentValue = 0.0;
        }

        public void addValue(double value) {
            this.sum -= this.circBuffer[this.circBufferIndex];
            this.circBuffer[this.circBufferIndex++] = value;
            this.currentValue = value;
            this.sum += value;
            if (this.circBufferIndex >= this.size) {
                this.circBufferIndex = 0;
            }
        }

        public double getCurrent() {
            return this.currentValue;
        }

        public double getAverage() {
            return this.sum / (double)this.size;
        }
    }

    private static class ProcStat {
        final long userTime;
        final long systemTime;
        final long idleTime;

        ProcStat(long userTime, long systemTime, long idleTime) {
            this.userTime = userTime;
            this.systemTime = systemTime;
            this.idleTime = idleTime;
        }
    }
}

