package de.tavendo.autobahn;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.Log;

import com.xdja.poc.common.utils.LogUtils;
import com.xdja.poc.common.utils.NetUtils;

import java.lang.ref.WeakReference;
import java.net.Socket;
import java.net.URI;

import de.tavendo.autobahn.WebSocket.WebSocketConnectionObserver.WebSocketCloseNotification;
import de.tavendo.autobahn.WebSocketMessage.BinaryMessage;
import de.tavendo.autobahn.WebSocketMessage.ClientHandshake;
import de.tavendo.autobahn.WebSocketMessage.Close;
import de.tavendo.autobahn.WebSocketMessage.ConnectionLost;
import de.tavendo.autobahn.WebSocketMessage.Ping;
import de.tavendo.autobahn.WebSocketMessage.Pong;
import de.tavendo.autobahn.WebSocketMessage.ProtocolViolation;
import de.tavendo.autobahn.WebSocketMessage.Quit;
import de.tavendo.autobahn.WebSocketMessage.RawTextMessage;
import de.tavendo.autobahn.WebSocketMessage.ServerHandshake;
import de.tavendo.autobahn.WebSocketMessage.TextMessage;


public class WebSocketConnection implements WebSocket {
    private static final String TAG = WebSocketConnection.class.getName();
    private static final String WS_URI_SCHEME = "ws";
    private static final String WSS_URI_SCHEME = "wss";
    private static final String WS_WRITER = "WebSocketWriter";
    private static final String WS_READER = "WebSocketReader";
    private final Handler mHandler;
    private Context mContext;
    private WebSocketReader mWebSocketReader;
    private WebSocketWriter mWebSocketWriter;
    private Socket mSocket;
    private SocketThread mSocketThread;
    private URI mWebSocketURI;
    private String[] mWebSocketSubprotocols;
    private WeakReference<WebSocketConnectionObserver> mWebSocketConnectionObserver;
    private WebSocketOptions mWebSocketOptions;
    private boolean mPreviousConnection = false;

    public WebSocketConnection() {
        LogUtils.DLog(TAG, "WebSocket connection created.");
        this.mHandler = new ThreadHandler(this);
    }

    public WebSocketConnection(Context context) {
        this.mContext = context;
        this.mHandler = new ThreadHandler(this);
    }

    @Override
    public void sendTextMessage(String payload) {
        //mWebSocketWriter 不能为null
        if (this.mWebSocketWriter != null) {
            this.mWebSocketWriter.forward(new TextMessage(payload));
        }
    }

    @Override
    public void sendRawTextMessage(byte[] payload) {
        //mWebSocketWriter不能为null
        if (this.mWebSocketWriter != null) {
            this.mWebSocketWriter.forward(new RawTextMessage(payload));
        }
    }

    @Override
    public void sendBinaryMessage(byte[] payload) {
        //mWebSocketWriter不能为null
        if (this.mWebSocketWriter != null) {
            this.mWebSocketWriter.forward(new BinaryMessage(payload));
        }
    }

    @Override
    public boolean isConnected() {
        return this.mSocket != null && this.mSocket.isConnected() && !this.mSocket.isClosed();
    }

    private void failConnection(WebSocketCloseNotification code, String reason) {
        LogUtils.DLog(TAG, "fail connection [code = " + code + ", reason = " + reason);
        if (this.mWebSocketReader != null&& this.mWebSocketReader.isAlive()) {
            this.mWebSocketReader.quit();

            try {
                this.mWebSocketReader.join();
            } catch (InterruptedException var5) {
                var5.printStackTrace();
            }
        } else {
            LogUtils.DLog(TAG, "mReader already NULL");

            this.onClose(code, reason);
            LogUtils.DLog(TAG, "worker threads stopped");
            return;
        }

        if (this.mWebSocketWriter != null && this.mWebSocketWriter.isAlive()) {
            if (this.mWebSocketWriter.forward(new Quit())){
                try {
                    this.mWebSocketWriter.join();
                } catch (InterruptedException var4) {
                    var4.printStackTrace();
                }
            }
        } else {
            LogUtils.DLog(TAG, "mWriter already NULL");
            this.onClose(code, reason);
            return;
        }

        if (this.mSocket != null && this.mSocketThread.getHandler() != null && this.mSocketThread.isAlive()) {
            this.mSocketThread.getHandler().post(new Runnable() {
                @Override
                public void run() {
                    WebSocketConnection.this.mSocketThread.stopConnection();
                }
            });
        } else {
            LogUtils.DLog(TAG, "mTransportChannel already NULL");
            this.onClose(code, reason);
            LogUtils.DLog(TAG, "worker threads stopped");
            return;
        }

        if (this.mSocket != null && this.mSocketThread.getHandler() != null) {
            this.mSocketThread.getHandler().post(new Runnable() {
                public void run() {
                    Looper.myLooper().quit();
                }
            });
        }
        this.onClose(code, reason);
        LogUtils.DLog(TAG, "worker threads stopped");
    }

    @Override
    public void connect(URI webSocketURI, WebSocketConnectionObserver connectionObserver) throws WebSocketException {
        this.connect(webSocketURI, connectionObserver, new WebSocketOptions());
    }

    @Override
    public void connect(URI webSocketURI, WebSocketConnectionObserver connectionObserver, WebSocketOptions options) throws WebSocketException {
        this.connect(webSocketURI, (String[]) null, connectionObserver, options);
    }

    public void connect(URI webSocketURI, String[] subprotocols, WebSocketConnectionObserver connectionObserver, WebSocketOptions options) throws WebSocketException {
        if (this.isConnected()) {
            throw new WebSocketException("already connected");
        } else if (webSocketURI == null) {
            throw new WebSocketException("WebSockets URI null.");
        } else {
            this.mWebSocketURI = webSocketURI;
            if (!this.mWebSocketURI.getScheme().equals(WS_URI_SCHEME) && !this.mWebSocketURI.getScheme().equals(WSS_URI_SCHEME)) {
                throw new WebSocketException("unsupported scheme for WebSockets URI");
            } else {
                this.mWebSocketSubprotocols = subprotocols;
                this.mWebSocketConnectionObserver = new WeakReference(connectionObserver);
                this.mWebSocketOptions = new WebSocketOptions(options);
                this.connect();
            }
        }
    }

    @Override
    public void disconnect() {
        if (this.mWebSocketWriter != null && this.mWebSocketWriter.isAlive()) {
            this.mWebSocketWriter.forward(new Close());
        } else {
            LogUtils.DLog(TAG, "Could not send WebSocket Close .. writer already null");
        }

        this.mPreviousConnection = false;
    }

    public boolean reconnect() {
        if (!this.isConnected() && this.mWebSocketURI != null) {
            this.connect();
            return true;
        } else {
            return false;
        }
    }

    private void connect() {
        this.mSocketThread = new SocketThread(this.mWebSocketURI, this.mWebSocketOptions);
        this.mSocketThread.setContext(mContext);
        this.mSocketThread.start();
        SocketThread var1 = this.mSocketThread;
        synchronized (this.mSocketThread) {
            try {
                this.mSocketThread.wait();
            } catch (InterruptedException var6) {
                var6.printStackTrace();
                LogUtils.ELog(TAG, "mSocketThread  wait ...ex: " + var6.getLocalizedMessage());
            }
        }
        LogUtils.ELog(TAG, "thread notify ...");
        if (this.mSocketThread.getHandler() != null) {
            this.mSocketThread.getHandler().post(new Runnable() {
                @Override
                public void run() {
                    try {
                        LogUtils.ELog(TAG, "mSocketThread start connect ...");
                        WebSocketConnection.this.mSocketThread.startConnection();
                    } catch (Exception ex) {
                        mSocketThread.notifyAll();//lyz@xdja.com add
                        LogUtils.ELog(TAG, "mSocketThread start connect ...ex: " + ex.getLocalizedMessage());
                    }
                }
            });
        }
        var1 = this.mSocketThread;
        synchronized (this.mSocketThread) {
            try {
                this.mSocketThread.wait();
            } catch (InterruptedException var4) {

            }
        }

        this.mSocket = this.mSocketThread.getSocket();
        if (this.mSocket == null) {
            //判断有没有网络
            if (NetUtils.isNetworkAvailable(mContext)) {
                this.onClose(WebSocketCloseNotification.CANNOT_CONNECT, this.mSocketThread.getFailureMessage());
            } else {
                WebSocketConnectionObserver observer = this.mWebSocketConnectionObserver.get();
                if (observer != null) {
                    observer.onError(new Exception("No available network"));
                }
            }
        } else if (this.mSocket.isConnected()) {
            try {
                this.createReader();
                this.createWriter();
                ClientHandshake clientHandshake = new ClientHandshake(this.mWebSocketURI, (URI) null, this.mWebSocketSubprotocols);
                this.mWebSocketWriter.forward(clientHandshake);
            } catch (Exception var3) {
                this.onClose(WebSocketCloseNotification.INTERNAL_ERROR, var3.getLocalizedMessage());
            }
        } else {
            this.onClose(WebSocketCloseNotification.CANNOT_CONNECT, "could not connect to WebSockets server");
        }

    }

    protected boolean scheduleReconnect() {
        int interval = this.mWebSocketOptions.getReconnectInterval();
        boolean shouldReconnect = this.mSocket != null && this.mSocket.isConnected() && this.mPreviousConnection && interval > 0;
        if (shouldReconnect) {
            LogUtils.DLog(TAG, "WebSocket reconnection scheduled");
            this.mHandler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    LogUtils.DLog(WebSocketConnection.TAG, "WebSocket reconnecting...");
                    WebSocketConnection.this.reconnect();
                }
            }, (long) interval);
        }

        return shouldReconnect;
    }

    private void onClose(WebSocketCloseNotification code, String reason) {
        boolean reconnecting = false;
        if (code == WebSocketCloseNotification.CANNOT_CONNECT || code == WebSocketCloseNotification.CONNECTION_LOST) {
            reconnecting = this.scheduleReconnect();
        }

        WebSocketConnectionObserver webSocketObserver = (WebSocketConnectionObserver) this.mWebSocketConnectionObserver.get();
        if (webSocketObserver != null) {
            try {
                if (reconnecting) {
                    webSocketObserver.onClose(WebSocketCloseNotification.RECONNECT, reason);
                } else {
//                    webSocketObserver.onClose(code, reason);
                    webSocketObserver.onError(new Exception(reason));
                }
            } catch (Exception var6) {
                var6.printStackTrace();
                webSocketObserver.onError(var6);
            }
        } else {
            LogUtils.DLog(TAG, "WebSocketObserver null");
            webSocketObserver.onError(new Exception("WebSocketObserver is null"));
        }

    }

    protected void processAppMessage(Object message) {
    }

    protected void createWriter() {
        this.mWebSocketWriter = new WebSocketWriter(this.mHandler, this.mSocket, this.mWebSocketOptions, WS_WRITER);
        this.mWebSocketWriter.start();
        WebSocketWriter var1 = this.mWebSocketWriter;
        synchronized (this.mWebSocketWriter) {
            try {
                this.mWebSocketWriter.wait();
            } catch (InterruptedException var3) {
                ;
            }
        }

        LogUtils.DLog(TAG, "WebSocket writer created and started.");
    }

    protected void createReader() {
        this.mWebSocketReader = new WebSocketReader(this.mHandler, this.mSocket, this.mWebSocketOptions, WS_READER);
        this.mWebSocketReader.start();
        WebSocketReader var1 = this.mWebSocketReader;
        synchronized (this.mWebSocketReader) {
            try {
                this.mWebSocketReader.wait();
            } catch (InterruptedException var3) {
                ;
            }
        }

        LogUtils.DLog(TAG, "WebSocket reader created and started.");
    }

    private void handleMessage(Message message) {
        WebSocketConnectionObserver webSocketObserver = (WebSocketConnectionObserver) this.mWebSocketConnectionObserver.get();
        if (message.obj instanceof TextMessage) {
            TextMessage textMessage = (TextMessage) message.obj;
            if (webSocketObserver != null) {
                webSocketObserver.onTextMessage(textMessage.mPayload);
            } else {
                LogUtils.DLog(TAG, "could not call onTextMessage() .. handler already NULL");
            }
        } else if (message.obj instanceof RawTextMessage) {
            RawTextMessage rawTextMessage = (RawTextMessage) message.obj;
            if (webSocketObserver != null) {
                webSocketObserver.onRawTextMessage(rawTextMessage.mPayload);
            } else {
                LogUtils.DLog(TAG, "could not call onRawTextMessage() .. handler already NULL");
            }
        } else if (message.obj instanceof BinaryMessage) {
            BinaryMessage binaryMessage = (BinaryMessage) message.obj;
            if (webSocketObserver != null) {
                webSocketObserver.onBinaryMessage(binaryMessage.mPayload);
            } else {
                LogUtils.DLog(TAG, "could not call onBinaryMessage() .. handler already NULL");
            }
        } else if (message.obj instanceof Ping) {
            Ping ping = (Ping) message.obj;
            LogUtils.DLog(TAG, "WebSockets Ping received");
            Pong pong = new Pong();
            pong.mPayload = ping.mPayload;
            this.mWebSocketWriter.forward(pong);
        } else if (message.obj instanceof Pong) {
            Pong pong = (Pong) message.obj;
            LogUtils.DLog(TAG, "WebSockets Pong received" + pong.mPayload);
        } else if (message.obj instanceof Close) {
            Close close = (Close) message.obj;
            LogUtils.DLog(TAG, "WebSockets Close received (" + close.getCode() + " - " + close.getReason() + ")");
            this.mWebSocketWriter.forward(new Close(1000));
        } else if (message.obj instanceof ServerHandshake) {
            ServerHandshake serverHandshake = (ServerHandshake) message.obj;
            LogUtils.DLog(TAG, "opening handshake received");
            if (serverHandshake.mSuccess) {
                if (webSocketObserver != null) {
                    webSocketObserver.onOpen();
                } else {
                    LogUtils.DLog(TAG, "could not call onOpen() .. handler already NULL");
                }

                this.mPreviousConnection = true;
            }
        } else if (message.obj instanceof ConnectionLost) {
            this.failConnection(WebSocketCloseNotification.CONNECTION_LOST, "WebSockets connection lost");
        } else if (message.obj instanceof ProtocolViolation) {
            this.failConnection(WebSocketCloseNotification.PROTOCOL_ERROR, "WebSockets protocol violation");
        } else if (message.obj instanceof WebSocketMessage.Error) {
            WebSocketMessage.Error error = (WebSocketMessage.Error) message.obj;
            this.failConnection(WebSocketCloseNotification.INTERNAL_ERROR, "WebSockets internal error (" + error.mException.toString() + ")");
        } else if (message.obj instanceof WebSocketMessage.ServerError) {
            WebSocketMessage.ServerError error = (WebSocketMessage.ServerError) message.obj;
            this.failConnection(WebSocketCloseNotification.SERVER_ERROR, "Server error " + error.mStatusCode + " (" + error.mStatusMessage + ")");
        } else {
            this.processAppMessage(message.obj);
        }

    }


    private static class ThreadHandler extends Handler {
        private final WeakReference<WebSocketConnection> mWebSocketConnection;

        public ThreadHandler(WebSocketConnection webSocketConnection) {
            this.mWebSocketConnection = new WeakReference(webSocketConnection);
        }

        @Override
        public void handleMessage(Message message) {
            WebSocketConnection webSocketConnection = (WebSocketConnection) this.mWebSocketConnection.get();
            if (webSocketConnection != null) {
                webSocketConnection.handleMessage(message);
            }

        }
    }
}
