package com.xdja.poc.sdk.ui.history;

import android.media.MediaPlayer;

import com.xdja.poc.sdk.config.Constants;
import com.xdja.poc.sdk.record.http.HttpClientFactory;
import com.xdja.poc.sdk.record.http.IHttpClient;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Request;
import okhttp3.Response;

/**
 * 录音播放
 * Created by xdjaxa on 2019/1/4.
 */

public class HistoryVoicePlayer {

    private MediaPlayer mediaPlayer;
    private PlayVoiceCallBack playVoiceCallBack;
    private String upVoiceUrl = "";
    private IHttpClient client;
    private boolean downloading = false;

    private static HistoryVoicePlayer instance = null;

    private HistoryVoicePlayer() {
    }

    public static HistoryVoicePlayer getInstance() {
        if (instance == null) {
            synchronized (HistoryVoicePlayer.class) {
                if (instance == null) {
                    instance = new HistoryVoicePlayer();
                }
            }
        }
        return instance;
    }

    /**
     * 外部调用方法，开始播放音频
     *
     * @param message
     * @param voiceCallBack
     */
    public void startPlay(POCHistoryMessage message, PlayVoiceCallBack voiceCallBack) {
        //有音频在下载不播放
        if (downloading) {
            return;
        }
        //上一播放路径与当前播放路径一致且正在播放，则暂停播放
        if (upVoiceUrl.equals(message.getFileUrl())) {
            if (mediaPlayer != null && mediaPlayer.isPlaying()) {
                releasePlay();
                if (playVoiceCallBack != null) {
                    playVoiceCallBack.onPlayStop();
                }
                return;
            }
        }
        //释放player，解决非法状态异常
        releasePlay();

        this.upVoiceUrl = message.getFileUrl();
        this.playVoiceCallBack = voiceCallBack;

        if (playVoiceCallBack != null) {
            playVoiceCallBack.onPlayStart();
        }
        //播放文件存在，则直接播放
        if (new File(Constants.HISTORY_RECORD_DOWNLOAD_PATH + message.id + "." + message.fileType).exists()) {
            realPlay(Constants.HISTORY_RECORD_DOWNLOAD_PATH + message.id + "." + message.fileType);
            return;
        }
        //播放文件不存在，下载后播放
        downloadFile(message);
    }

    /**
     * 外部调用方法，释放player
     */
    public void releasePlay() {
        if (mediaPlayer != null && mediaPlayer.isPlaying()) {
            mediaPlayer.stop();
            mediaPlayer.release();
        }
        mediaPlayer = null;
    }

    /**
     * 播放音频
     *
     * @param fileUri
     */
    private void realPlay(String fileUri) {
        if (mediaPlayer == null) {
            mediaPlayer = new MediaPlayer();
        }
        try {
            mediaPlayer.setDataSource(fileUri);
            mediaPlayer.setLooping(false);
            mediaPlayer.setOnErrorListener(new MediaPlayer.OnErrorListener() {
                @Override
                public boolean onError(MediaPlayer mp, int what, int extra) {
                    releasePlay();
                    if (playVoiceCallBack != null) {
                        playVoiceCallBack.onPlayStop();
                    }
                    return false;
                }
            });
            mediaPlayer.setOnCompletionListener(new MediaPlayer.OnCompletionListener() {
                @Override
                public void onCompletion(MediaPlayer mp) {
                    releasePlay();
                    if (playVoiceCallBack != null) {
                        playVoiceCallBack.onPlayStop();
                    }
                }
            });

            mediaPlayer.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
                @Override
                public void onPrepared(MediaPlayer mp) {
                    if (mp != null) {
                        mp.start();
                    }
                }
            });

            mediaPlayer.prepare();
        } catch (IOException e) {
            e.printStackTrace();
            releasePlay();
            if (playVoiceCallBack != null) {
                playVoiceCallBack.onPlayFailed();
            }
        }
    }

    /**
     * 下载录音文件
     *
     * @param message
     */
    private void downloadFile(POCHistoryMessage message) {
        downloading = true;
        final POCHistoryMessage currMessage = message;
        client = HttpClientFactory.createHttpClient();

        Request request = new Request.Builder()
                .url(currMessage.fileUrl)
                .build();

        client.downloadFile(request, new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                downloading = false;
                if (playVoiceCallBack != null) {
                    playVoiceCallBack.onPlayFailed();
                }
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                InputStream is = null;
                byte[] buff = new byte[1024];
                int len = 0;
                FileOutputStream fos = null;

                File dir = new File(Constants.HISTORY_RECORD_DOWNLOAD_PATH);
                if (!dir.exists()) {
                    dir.mkdirs();
                }
                File file = new File(dir, currMessage.id + "." + currMessage.fileType);
                try {
                    is = response.body().byteStream();
                    fos = new FileOutputStream(file);
                    while ((len = is.read(buff)) != -1) {
                        fos.write(buff);
                    }
                    fos.flush();
                } catch (IOException e) {
                    e.printStackTrace();
                    if (playVoiceCallBack != null) {
                        playVoiceCallBack.onPlayFailed();
                    }
                } finally {
                    try {
                        if (is != null) {
                            is.close();
                        }
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                    try {
                        if (fos != null) {
                            fos.close();
                        }
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
                downloading = false;
                if (!upVoiceUrl.equals(currMessage.getFileUrl())) {
                    return;
                }
                realPlay(file.getAbsolutePath());
            }
        });
    }

    /**
     * 播放音频回调
     */
    public interface PlayVoiceCallBack {

        void onPlayStart();

        void onPlayStop();

        void onPlayFailed();
    }
}
