package com.xdja.poc.sdk.ui.history;

import android.content.Context;
import android.graphics.drawable.AnimationDrawable;
import android.support.annotation.NonNull;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.jude.easyrecyclerview.adapter.BaseViewHolder;
import com.jude.easyrecyclerview.adapter.RecyclerArrayAdapter;
import com.xdja.poc.common.utils.DisplayUtil;
import com.xdja.poc.common.utils.PreferenceHelper;
import com.xdja.poc.common.utils.StringUtils;
import com.xdja.poc.common.utils.ToastUtils;
import com.xdja.poc.sdk.R;
import com.xdja.poc.sdk.bean.POCMemberInfo;
import com.xdja.poc.sdk.config.UserConfig;
import com.xdja.poc.sdk.db.entity.User;
import com.xdja.poc.sdk.proxy.POCUiKitProxyImpl;
import com.xdja.poc.sdk.utils.GlideUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;

/**
 * 历史消息适配器
 * Created by xdjaxa on 2019/1/4.
 */

public class POCHistoryMessageAdapter extends RecyclerArrayAdapter<POCHistoryMessage> {

    private AnimationDrawable drawable; //语音播放动画处理
    private Map<String, POCMemberInfo> mCacheUserinfoMap = null; //群组成员信息
    private User mCacheUserinfo = null; //我的信息
    private String upMessageTime; //上一条消息时间，用于判断是否显示消息时间
    private static final int MAX_MESSAGE_TIME_INTERVAL = 30 * 1000; //消息时间间隔

    public POCHistoryMessageAdapter(Context context) {
        super(context);
        mCacheUserinfoMap = POCUiKitProxyImpl.getInstance().getUserInfosCache();
        mCacheUserinfo = POCUiKitProxyImpl.getInstance().getUserInfo();
    }

    @Override
    public BaseViewHolder OnCreateViewHolder(ViewGroup parent, int viewType) {
        return new MessageViewHolder(parent);
    }

    @Override
    public void OnBindViewHolder(BaseViewHolder holder, int position) {
        //super，调用ViewHolder的setData方法绑定数据
        super.OnBindViewHolder(holder, position);
        if (holder instanceof MessageViewHolder) {
            //我的消息语音播放
            ((MessageViewHolder) holder).mMineVoiceLayout.setOnClickListener(v -> {
                HistoryVoicePlayer.getInstance().startPlay(getAllData().get(position), new HistoryVoicePlayer.PlayVoiceCallBack() {
                    @Override
                    public void onPlayFailed() {
                        ToastUtils.showToast("播放失败");
                        stopAnim();
                        ((MessageViewHolder) holder).mOtherVoice.setBackground(getContext().getDrawable(R.drawable.poc_anim_history_record_voice_blue));
                    }

                    @Override
                    public void onPlayStart() {
                        stopAnim();
                        drawable = (AnimationDrawable) ((MessageViewHolder) holder).mMineVoice.getBackground();
                        drawable.start();
                    }

                    @Override
                    public void onPlayStop() {
                        stopAnim();
                        ((MessageViewHolder) holder).mMineVoice.setBackground(getContext().getDrawable(R.drawable.poc_anim_history_record_voice_white));
                    }
                });
            });
            //其他人消息语音播放
            ((MessageViewHolder) holder).mOtherVoiceLayout.setOnClickListener(v -> {
                HistoryVoicePlayer.getInstance().startPlay(getAllData().get(position), new HistoryVoicePlayer.PlayVoiceCallBack() {
                    @Override
                    public void onPlayFailed() {
                        ToastUtils.showToast("播放异常");
                        stopAnim();
                        ((MessageViewHolder) holder).mOtherVoice.setBackground(getContext().getDrawable(R.drawable.poc_anim_history_record_voice_blue));
                    }

                    @Override
                    public void onPlayStart() {
                        stopAnim();
                        drawable = (AnimationDrawable) ((MessageViewHolder) holder).mOtherVoice.getBackground();
                        drawable.start();
                    }

                    @Override
                    public void onPlayStop() {
                        stopAnim();
                        ((MessageViewHolder) holder).mOtherVoice.setBackground(getContext().getDrawable(R.drawable.poc_anim_history_record_voice_blue));
                    }
                });
            });
        }
    }

    public class MessageViewHolder extends BaseViewHolder<POCHistoryMessage> {

        ImageView mMineIcon, mOtherIcon;
        TextView mMineTime, mOtherTime;
        TextView mMineName, mOtherName;
        RelativeLayout mMineLayout, mOtherLayout;
        TextView mNotSupportLayout, mTime;
        RelativeLayout mMineVoiceLayout, mOtherVoiceLayout;
        ImageView mMineVoice, mOtherVoice;

        public MessageViewHolder(ViewGroup group) {
            super(group, R.layout.poc_item_history_message);
            mMineIcon = $(R.id.iv_mine_message_icon);
            mMineLayout = $(R.id.rl_mine);
            mMineTime = $(R.id.tv_mine_message_time);
            mMineName = $(R.id.tv_mine_name);
            mMineVoiceLayout = $(R.id.rl_mine_voice_bg);
            mMineVoice = $(R.id.iv_mine_voice);

            mOtherIcon = $(R.id.iv_other_message_icon);
            mOtherLayout = $(R.id.rl_other);
            mOtherTime = $(R.id.tv_other_message_time);
            mOtherName = $(R.id.tv_other_name);
            mOtherVoiceLayout = $(R.id.rl_other_voice_bg);
            mOtherVoice = $(R.id.iv_other_voice);

            mNotSupportLayout = $(R.id.tv_not_support);
            mTime = $(R.id.iv_interval_time);
        }

        @Override
        public void setData(POCHistoryMessage message) {

            upMessageTime = getItem(getDataPosition() > 0 ? getDataPosition() - 1 : getDataPosition()).getCreateTime();
            //获取登录账号、消息账号
            String userAccount = mCacheUserinfo != null ? mCacheUserinfo.getAccount() : "";
            String messageAccount = message.owner;
            //账号为空，非法消息
            if (StringUtils.isEmpty(messageAccount) || StringUtils.isEmpty(userAccount)) {
                mNotSupportLayout.setVisibility(View.VISIBLE);
                mOtherLayout.setVisibility(View.GONE);
                mMineLayout.setVisibility(View.GONE);
                return;
            }
            //消息时间
            mTime.setText(formatTime(message.createTime));
            mTime.setVisibility(computeMessageTimeInterval(message.createTime) || getDataPosition() == 0 ? View.VISIBLE : View.GONE);
            //我的消息
            if (userAccount.equals(messageAccount)) {
                mOtherLayout.setVisibility(View.GONE);
                mNotSupportLayout.setVisibility(View.GONE);
                mMineLayout.setVisibility(View.VISIBLE);

                mMineName.setText(transation(messageAccount) == null ? messageAccount : transation(messageAccount).getName());

                int voiceTime = computeVoiceTime(message);
                mMineTime.setText(String.format(getContext().getString(R.string.poc_history_record_format_time), voiceTime + ""));

                RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) mMineVoiceLayout.getLayoutParams();
                params.width = computeLayoutWidth(voiceTime);
                mMineVoiceLayout.setLayoutParams(params);

                GlideUtils.setImageUseNetWithCircle(getContext(), transation(messageAccount) == null ? "" : transation(messageAccount).getIcon(), mMineIcon);
                return;
            }
            //其他人消息
            mMineLayout.setVisibility(View.GONE);
            mNotSupportLayout.setVisibility(View.GONE);
            mOtherLayout.setVisibility(View.VISIBLE);

            mOtherName.setText(transation(messageAccount) == null ? messageAccount : transation(messageAccount).getName());

            int voiceTime = computeVoiceTime(message);
            mOtherTime.setText(String.format(getContext().getString(R.string.poc_history_record_format_time), voiceTime + ""));

            mOtherVoiceLayout.setMinimumWidth(computeLayoutWidth(voiceTime));

            GlideUtils.setImageUseNetWithCircle(getContext(), transation(messageAccount) == null ? "" : transation(messageAccount).getIcon(), mOtherIcon);
        }

    }

    /**
     * 计算voice时长
     *
     * @param message
     * @return
     */
    private int computeVoiceTime(POCHistoryMessage message) {
        return Integer.valueOf(message.getFileSize()) / 1000 + 1;
    }

    /**
     * 计算录音布局长度
     *
     * @param voiceTime
     * @return
     */
    private int computeLayoutWidth(int voiceTime) {
        return DisplayUtil.dip2px(getContext(),
                64 + (voiceTime < 10 ? voiceTime * 3 : (voiceTime < 20 ? voiceTime * 2 : voiceTime / 2)));
    }

    /**
     * 计算与上一条消息时间间隔
     *
     * @param currMsgTime
     * @return
     */
    private boolean computeMessageTimeInterval(String currMsgTime) {
        SimpleDateFormat sourceFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        try {
            Date currMsgDate = sourceFormat.parse(currMsgTime);
            Date upMsgDate = sourceFormat.parse(upMessageTime);
            if (currMsgDate.getTime() - upMsgDate.getTime() > MAX_MESSAGE_TIME_INTERVAL) {
                return true;
            }
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * 获取消息人员信息
     *
     * @param messageAccount
     * @return
     */
    private POCMemberInfo transation(String messageAccount) {
        if (mCacheUserinfoMap == null) {
            return null;
        }
        return mCacheUserinfoMap.get(messageAccount);
    }

    /**
     * 格式化消息时间
     *
     * @param sourceTime
     * @return
     */
    private String formatTime(String sourceTime) {
        SimpleDateFormat sourceFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        SimpleDateFormat destFormat = new SimpleDateFormat("HH:mm");
        try {
            Date date = sourceFormat.parse(sourceTime);
            return getAppendTime(date) + destFormat.format(date);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return sourceTime;
    }


    /**
     * 获取消息附加时间信息
     *
     * @param messageDate
     * @return
     * @throws ParseException
     */
    private String getAppendTime(Date messageDate) throws ParseException {

        SimpleDateFormat msgDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        String msgDateStr = msgDateFormat.format(messageDate);
        Date msgResultDate = msgDateFormat.parse(msgDateStr);

        Date phoneDate = new Date();
        String phoneDateStr = msgDateFormat.format(phoneDate);
        Date phoneResultDate = msgDateFormat.parse(phoneDateStr);

        String append;
        if (phoneResultDate.getTime() - msgResultDate.getTime() == 0) {
            append = "";
        } else if (phoneResultDate.getTime() - msgResultDate.getTime() <= 24 * 60 * 60 * 1000) {
            append = "昨天" + " ";
        } else {
            SimpleDateFormat resultFormat = new SimpleDateFormat("yyyy年MM月dd日");
            append = resultFormat.format(msgResultDate) + " ";
        }
        return append;
    }

    /**
     * 结束动画
     */
    public void stopAnim() {
        if (drawable != null) {
            drawable.stop();
        }
    }
}
