package com.xdja.poc.sdk.business.ringplay;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothProfile;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.media.AudioManager;
import android.net.Uri;
import android.os.Vibrator;
import android.provider.Settings;
import android.text.TextUtils;

import com.xdja.poc.sdk.R;
import com.xdja.poc.common.utils.GlobalContext;
import com.xdja.poc.common.utils.LogUtils;
import com.xdja.poc.common.utils.ToastUtils;

import io.reactivex.Observer;
import io.reactivex.disposables.Disposable;

import static com.xdja.poc.common.utils.DeviceUtils.gainAudioFocus;
import static com.xdja.poc.common.utils.DeviceUtils.releaseAudioFocus;

/**
 * Created by gouhao on 6/27/2017.
 */

public class RingPlayer implements IRingPlayer {
    private final static String TAG = RingPlayer.class.getSimpleName();

    private static final long[] sVibratePattern = {0, 1000, 800, 1000, 800};

    private Context context;
    private AudioManager audioManager;
    private Vibrator vibrator;
    private RingConfig currentRingConfig;
    private RxAudioPlayer rxAudioPlayer;
    private RingStateReceiver ringStateReceiver;
    private Callback callback;
    public boolean isNeedPlayRing = false;
    private int mLooping = 0;
    private static RingPlayer instance;

    private static final int STOP = 1;
    private static final int CHANGE_AUDIO_MODE = 2;

    private int flag;

    private RingPlayer(Context context) {
        init(context);
    }

    public static RingPlayer getInstance(Context context) {
        if (instance == null) {
            synchronized (RingPlayer.class) {
                if (instance == null) {
                    instance = new RingPlayer(context);
                }
            }
        }
        return instance;
    }

    private void init(Context mContext) {
        this.context = mContext.getApplicationContext();
        this.audioManager = (AudioManager) context.getSystemService(Context.AUDIO_SERVICE);
        this.vibrator = (Vibrator) context.getSystemService(Context.VIBRATOR_SERVICE);
        this.rxAudioPlayer = RxAudioPlayer.getInstance();
        currentRingConfig = new RingConfig();
        registerRingStateReceiver();
        setFlag(true, STOP);
    }

    private void registerRingStateReceiver() {
        ringStateReceiver = new RingStateReceiver();
        IntentFilter mIntentFilter = new IntentFilter();
        mIntentFilter.addAction(AudioManager.RINGER_MODE_CHANGED_ACTION);
        context.registerReceiver(ringStateReceiver, mIntentFilter);
    }

    @Override
    public void playIncomingRing(boolean isOutGoing, boolean isVoice) {
        LogUtils.DLog(TAG, "playIncomingRing");
        setFlag(false, STOP);
        isNeedPlayRing = true;
        currentRingConfig.isOutgoing = isOutGoing;
        currentRingConfig.isVoice = isVoice;

        if (!isOutGoing && checkRingerMode()) {
            LogUtils.DLog(TAG, "playIncomingRing: is silent or vibrate mode");
            return;
        }

        int vibrateRing = Settings.System.getInt(GlobalContext.getContext().getContentResolver(), "vibrate_when_ringing", 0);
        // TODO: 2018/9/12 这地方控制是否震动,默认先打开
        if (!currentRingConfig.isOutgoing && vibrateRing != 0) {
        }
        startVibrate();

        playRing(R.raw.ring, true, new RingObserverAdapter() {
            @Override
            public void onError(final Throwable throwable) {
                LogUtils.ELog(TAG, "playIncoming error: " + throwable.toString());
                stopRing();
            }
        });
    }

    private boolean checkRingerMode() {
        int ringerMode = audioManager.getRingerMode();
        if (ringerMode == AudioManager.RINGER_MODE_SILENT) {
            LogUtils.DLog(TAG, "checkRingerMode: RINGER_MODE_SILENT");
            return true;
        } else if (ringerMode == AudioManager.RINGER_MODE_VIBRATE) {
            LogUtils.DLog(TAG, "checkRingerMode: RINGER_MODE_SILENT");
            startVibrate();
            return true;
        }
        return false;
    }

    @Override
    public void playNoAnswerRing() {
//        playFinishRing(R.raw.no_answer);
    }

    @Override
    public void playBusyRing() {
        ToastUtils.showToast(GlobalContext.getContext().getString(R.string.poc_user_is_calling));
//        playFinishRing(R.raw.user_busy);
    }

    @Override
    public void playErrorRing(String message) {
        if (!TextUtils.isEmpty(message)) {
            ToastUtils.showToast(message);
        }
//        playFinishRing(R.raw.connect_err);
    }

    private void playFinishRing(final int rawRes) {
        LogUtils.DLog(TAG, "playFinishRing");
        if (isStop()) {
            LogUtils.DLog(TAG, "playFinishRing: RingPlayer already stop");
            playFinishComplete();
            return;
        }
        isNeedPlayRing = false;
        currentRingConfig.isPlayErrorRing = true;
//        if (!(FloatingWindowManager.hasFloatingWindow() ||GlobalContext.getContext().getCurActivity() instanceof CallWaitingActivity)) {
//            LogUtils.DLog(TAG, "playFinishRing: communication is close, not need playFinishRing");
//            mLooping = 0;
//            return;
//        }
        rxAudioPlayer.stopPlay();
        playRing(rawRes, false, new RingObserverAdapter() {
            @Override
            public void onError(final Throwable throwable) {
                LogUtils.DLog(TAG, "playFinishRing error: " + throwable.toString());
                playFinishComplete();
            }

            @Override
            public void onNext(Boolean aBoolean) {
                mLooping++;
            }

            @Override
            public void onComplete() {
                if (mLooping < 2 && !isStop()) {
                    playFinishRing(rawRes);
                } else {
                    playFinishComplete();
                }
            }
        });
    }

    private void playFinishComplete() {
        if (callback != null) {
            callback.playErrorRingFinish();
        }
    }

    /**
     * 判断是否戴耳机或者戴蓝牙
     *
     * @return
     */
    public boolean isConnectedHeadsetOrBluetooth() {
        BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();
        int bluetoothState = adapter.getProfileConnectionState(BluetoothProfile.HEADSET);
        return bluetoothState == BluetoothProfile.STATE_CONNECTED ||
                bluetoothState == BluetoothProfile.STATE_CONNECTING ||
                audioManager.isWiredHeadsetOn();
    }

    /**
     * 设置外放模式还是听筒模式
     *
     * @param playMode
     */
    public void setAudioMode(boolean playMode) {
        if (audioManager != null) {
            audioManager.setSpeakerphoneOn(playMode);
        }
    }

    private void playRing(int rawRes, boolean isLooping, Observer<Boolean> observable) {
        if (!isStop()) {
            gainAudioFocus();
            int streamType = AudioManager.STREAM_RING;
            audioManager.setMode(AudioManager.MODE_NORMAL);
            audioManager.setSpeakerphoneOn(true);
            if (isConnectedHeadsetOrBluetooth()) {
                LogUtils.DLog(TAG, "playRing: from headset");
                if (currentRingConfig.isVoice && currentRingConfig.isOutgoing) {
                    LogUtils.DLog(TAG, "playRing: from headset is voice outgoing");
                    setFlag(true, CHANGE_AUDIO_MODE);
                    audioManager.setSpeakerphoneOn(false);
                    streamType = AudioManager.STREAM_VOICE_CALL;
                }
            } else {
                if (currentRingConfig.isVoice && currentRingConfig.isOutgoing) {
                    LogUtils.DLog(TAG, "playRing: from headphone");
                    setFlag(true, CHANGE_AUDIO_MODE);
                    audioManager.setMode(AudioManager.MODE_IN_COMMUNICATION);
                    audioManager.setSpeakerphoneOn(true);
                    streamType = AudioManager.STREAM_VOICE_CALL;
                }
            }
            rxAudioPlayer.play(PlayConfig.uri(GlobalContext.getContext(),
                    Uri.parse("android.resource://" + GlobalContext.getContext().getPackageName() + "/" + rawRes))
                    .streamType(streamType)
                    .looping(isLooping)
                    .build())
                    .subscribe(observable);
        }
    }

    private void startVibrate() {
        vibrator.vibrate(sVibratePattern, 1);
    }

    private void stopVibrate() {
        if (vibrator.hasVibrator()) {
            vibrator.cancel();
        }
    }

    public void stopRing() {
        if (isStop()) {
            LogUtils.ELog(TAG, "stopRing: is not init or already stop");
            return;
        }
        LogUtils.DLog(TAG, "stopRing");
        setFlag(true, STOP);
        currentRingConfig.isPlayErrorRing = false;
        mLooping = 0;
        isNeedPlayRing = false;
        stopVibrate();
        rxAudioPlayer.stopPlay();
        resetAudioMode();
        releaseAudioFocus();
    }

    private void resetAudioMode() {
        if (getFlag(CHANGE_AUDIO_MODE)) {
            setFlag(false, CHANGE_AUDIO_MODE);
            audioManager.setMode(AudioManager.MODE_NORMAL);
            audioManager.setSpeakerphoneOn(!isConnectedHeadsetOrBluetooth());
        }
    }

    public void resetAudioModeOn() {
        setFlag(false, CHANGE_AUDIO_MODE);
        if (audioManager != null) {
            audioManager.setMode(AudioManager.MODE_NORMAL);
            audioManager.setSpeakerphoneOn(!isConnectedHeadsetOrBluetooth());
        }
    }

    @Override
    public void release() {
        synchronized (instance) {
            this.rxAudioPlayer.stopPlay();
            stopVibrate();
            unregisterRingStateReceiver();
            resetAudioMode();
            releaseAudioFocus();
            this.context = null;
            this.audioManager = null;
            this.vibrator = null;
            this.rxAudioPlayer = null;
            this.currentRingConfig = null;
            this.callback = null;
            flag = 0;
            instance = null;
        }
    }

    @Override
    public void setCallback(Callback callback) {
        this.callback = callback;
    }

    private void unregisterRingStateReceiver() {
        if (ringStateReceiver != null) {
            context.unregisterReceiver(ringStateReceiver);
            ringStateReceiver = null;
        }
    }

    public class RingConfig {
        public boolean isOutgoing;
        public boolean isVoice;
        public boolean isPlayErrorRing = false;
    }

    public boolean isStop() {
        return getFlag(STOP);
    }

    private boolean getFlag(int mask) {
        synchronized (RingPlayer.class) {
            return (flag & mask) != 0;
        }
    }

    private void setFlag(boolean value, int mask) {
        synchronized (RingPlayer.class) {
            if (value) {
                flag |= mask;
            } else {
                flag &= ~mask;
            }
        }
    }

    public RingConfig getCurrentRingConfig() {
        return currentRingConfig;
    }

    private class RingStateReceiver extends BroadcastReceiver {

        @Override
        public void onReceive(Context context, Intent intent) {
            if (!isNeedPlayRing || currentRingConfig.isOutgoing || intent == null || TextUtils.isEmpty(intent.getAction())) {
                LogUtils.DLog(TAG, "RingStateReceiver: ignore event");
                return;
            }
            if (intent.getAction().equals(AudioManager.RINGER_MODE_CHANGED_ACTION)) {
                int ringerMode = audioManager.getRingerMode();
                LogUtils.DLog(TAG, "RingStateReceiver: ringerMode is " + ringerMode);
                switch (ringerMode) {
                    case AudioManager.RINGER_MODE_NORMAL:
                        playIncomingRing(currentRingConfig.isOutgoing, currentRingConfig.isVoice);
                        break;
                    case AudioManager.RINGER_MODE_VIBRATE:
                        rxAudioPlayer.stopPlay();
                        startVibrate();
                        break;
                    case AudioManager.RINGER_MODE_SILENT:
                        stopVibrate();
                        rxAudioPlayer.stopPlay();
                        break;
                }
            }
        }
    }

    private class RingObserverAdapter implements Observer<Boolean> {

        @Override
        public void onSubscribe(Disposable disposable) {
        }

        @Override
        public void onNext(Boolean aBoolean) {
        }

        @Override
        public void onError(Throwable throwable) {
        }

        @Override
        public void onComplete() {
        }
    }
}
