/*
 *  Copyright 2018 The WebRTC project authors. All Rights Reserved.
 *
 *  Use of this source code is governed by a BSD-style license
 *  that can be found in the LICENSE file in the root of the source
 *  tree. An additional intellectual property rights grant can be found
 *  in the file PATENTS.  All contributing project authors may
 *  be found in the AUTHORS file in the root of the source tree.
 */

package com.xdja.poc.sdk.business.webrtc.apprtc;

import android.media.AudioFormat;
import android.os.Environment;
import android.util.Log;

import com.czt.mp3recorder.DataEncodeThread;
import com.czt.mp3recorder.util.LameUtil;
import com.czt.mp3recorder.util.RecordFileUtils;
import com.xdja.poc.common.utils.LogUtils;
import com.xdja.poc.sdk.config.Constants;

import org.webrtc.audio.JavaAudioDeviceModule;
import org.webrtc.audio.JavaAudioDeviceModule.SamplesReadyCallback;
import org.webrtc.voiceengine.WebRtcAudioRecord;
import org.webrtc.voiceengine.WebRtcAudioRecord.WebRtcAudioRecordSamplesReadyCallback;

import java.io.File;
import java.io.FileNotFoundException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.ShortBuffer;
import java.util.concurrent.ExecutorService;

/**
 * Implements the AudioRecordSamplesReadyCallback interface and writes
 * recorded raw audio samples to an output file.
 */
public class RecordedAudioToFileController
        implements SamplesReadyCallback, WebRtcAudioRecordSamplesReadyCallback {
    private static final String TAG = "RecordedAudioToFile";
    private final Object lock = new Object();
    private final ExecutorService executor;
    //保存录音数据
    private DataEncodeThread mEncodeThread;

    /**
     * 以下三项为默认配置参数。Google Android文档明确表明只有以下3个参数是可以在所有设备上保证支持的。
     */
    private static final int DEFAULT_SAMPLING_RATE = 16000;//
    //======================Lame Default Settings=====================
    private static final int DEFAULT_LAME_MP3_QUALITY = 7;
    /**
     * 与DEFAULT_CHANNEL_CONFIG相关，因为是mono单声，所以是1
     */
    private static final int DEFAULT_LAME_IN_CHANNEL = 1;
    /**
     * Encoded bit rate. MP3 file will be encoded with bit rate 16kbps
     */
    private static final int DEFAULT_LAME_MP3_BIT_RATE = 16;
    /**
     * 默认BufferSize，依据WebrtcAudioRecord初始化参数
     * 2560是实际WebrtcAudioRecord计算出的buffer大小
     */
    private static final int DEFAULT_BUFFER_ZISE = 2560;

    private OnSaveRecordCallBack saveRecordCallBack;
    private static double audioVolumn;
    //==================================================================

    public RecordedAudioToFileController(ExecutorService executor) {
        LogUtils.DLog(TAG, "ctor");
        this.executor = executor;
    }

    public RecordedAudioToFileController(ExecutorService executor, OnSaveRecordCallBack saveRecordCallBack) {
        LogUtils.DLog(TAG, "ctor");
        this.executor = executor;
        this.saveRecordCallBack = saveRecordCallBack;
    }

    /**
     * Should be called on the same executor thread as the one provided at
     * construction.
     */
    public boolean start() {
        LogUtils.DLog(TAG, "start");
        return true;
    }

    /**
     * Should be called on the same executor thread as the one provided at
     * construction.
     */
    public void stop() {
        LogUtils.DLog(TAG, "stop");
    }

    // Checks if external storage is available for read and write.
    private boolean isExternalStorageWritable() {
        String state = Environment.getExternalStorageState();
        return Environment.MEDIA_MOUNTED.equals(state);
    }

    // Called when new audio samples are ready.
    @Override
    public void onWebRtcAudioRecordSamplesReady(WebRtcAudioRecord.AudioSamples samples) {
        if (!isExternalStorageWritable()) {
            LogUtils.ELog(TAG, "Writing to external media is not possible");
            return;
        }
        onWebRtcAudioRecordSamplesReady(new JavaAudioDeviceModule.AudioSamples(samples.getAudioFormat(),
                samples.getChannelCount(), samples.getSampleRate(), samples.getData()));
    }

    // Called when new audio samples are ready.
    @Override
    public void onWebRtcAudioRecordSamplesReady(final JavaAudioDeviceModule.AudioSamples samples) {
        // The native audio layer on Android should use 16-bit PCM format.
        if (samples.getAudioFormat() != AudioFormat.ENCODING_PCM_16BIT) {
            LogUtils.ELog(TAG, "Invalid audio format");
            return;
        }
        // Append the recorded 16-bit audio samples to the open output file.
        if (samples.getData() == null) {
            return;
        }
        byte[] data = new byte[samples.getData().length];
        System.arraycopy(samples.getData(), 0, data, 0, samples.getData().length);
        short[] audioData = bytesToShort_(data);
        if (audioData != null && audioData.length > 0) {
            try {
                //计算分贝
                long r = audioData.length;
                long v = 0;
                // 将 buffer 内容取出，进行平方和运算
                for (short anAudioData : audioData) {
                    v += anAudioData * anAudioData;
                }
                // 平方和除以数据总长度，得到音量大小。
                double mean = v / (double) r;
                double volume = 10 * Math.log10(mean);
                audioVolumn = volume;
                Log.d(TAG, "分贝值:" + audioVolumn + "  mean: " + mean + " r: " + r + "  v: " + v + " length= " + audioData.length);
            } catch (Exception e) {
                LogUtils.ELog(TAG, "e: " + e.getLocalizedMessage());
            }
        }
        executor.execute(new Runnable() {
            @Override
            public void run() {
                if (mEncodeThread != null) {
                    mEncodeThread.addTask(bytesToShort(data), data.length / 2);
                    mEncodeThread.processData();
//                    mEncodeThread.addByteTask(data, samples.getData().length);
//                    mEncodeThread.writeData(data);
                }
            }
        });
    }


    @Override
    public void onWebRtcAudioRecordSamplesStart() {
        Log.e(TAG, "onWebRtcAudioRecordSamplesStart: ");
        if (!isExternalStorageWritable()) {
            LogUtils.ELog(TAG, "Writing to external media is not possible");
            return;
        }
        try {
            synchronized (lock) {
                LameUtil.init(DEFAULT_SAMPLING_RATE, DEFAULT_LAME_IN_CHANNEL, DEFAULT_SAMPLING_RATE, DEFAULT_LAME_MP3_BIT_RATE, DEFAULT_LAME_MP3_QUALITY);
                File path = new File(Constants.HISTORY_RECORD_FILE_PATH);
                if (!path.exists()) {
                    path.mkdirs();
                }
                String mp3FileName = new RecordFileUtils().getRecordFileName();
                File mp3File = new File(path, mp3FileName);
                mEncodeThread = new DataEncodeThread(mp3File, DEFAULT_BUFFER_ZISE, new DataEncodeThread.DataEncodeCallBack() {
                    @Override
                    public void onEncodeStart() {
                        if (saveRecordCallBack != null) {
                            saveRecordCallBack.onStartSave(mp3File.getAbsolutePath());
                            Log.d(TAG, "onEncodeStart: " + mp3File.getAbsolutePath());
                        }
                    }

                    @Override
                    public void onEncodeFinish() {
                        if (saveRecordCallBack != null) {
                            saveRecordCallBack.onEndSave();
                        }
                    }
                });
                mEncodeThread.start();
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onWebRtcAudioRecordSamplesStop() {
        Log.e(TAG, "onWebRtcAudioRecordSamplesStop: ");
        if (!isExternalStorageWritable()) {
            LogUtils.ELog(TAG, "Writing to external media is not possible");
            return;
        }
        if (mEncodeThread != null) {
            synchronized (lock) {
                mEncodeThread.sendStopMessage();
            }
        }
    }

    /**
     * 将录音数据byte[]转为Lame能处理的short[]
     *
     * @param data
     * @return
     */
    private short[] bytesToShort(byte[] data) {

        int dataLength = data.length;

        //byte[]转成short[]，数组长度缩减一半
        int shortLength = dataLength / 2;

        //把byte[]数组装进byteBuffer缓冲
        ByteBuffer byteBuffer = ByteBuffer.wrap(data, 0, dataLength);

        //将byteBuffer转成小端序并获取shortBuffer
        ShortBuffer shortBuffer = byteBuffer.order(ByteOrder.LITTLE_ENDIAN).asShortBuffer();
        short[] shortSamples = new short[shortLength];

        //取出shortBuffer中的short数组
        shortBuffer.get(shortSamples, 0, shortLength);

        return shortSamples;
    }

    /**
     * 保存录音文件监听回调接口
     */
    public interface OnSaveRecordCallBack {

        void onStartSave(String recordFileUri);

        void onEndSave();
    }

    public static double getAudioVolumn() {
        return audioVolumn;
    }

    public short[] bytesToShort_(byte[] bytes) {
        if (bytes == null) {
            return null;
        }
        short[] shorts = new short[bytes.length / 2];
        ByteBuffer.wrap(bytes).order(ByteOrder.LITTLE_ENDIAN).asShortBuffer().get(shorts);
        return shorts;
    }

}
