/*
 *  Copyright 2014 The WebRTC Project Authors. All rights reserved.
 *
 *  Use of this source code is governed by a BSD-style license
 *  that can be found in the LICENSE file in the root of the source
 *  tree. An additional intellectual property rights grant can be found
 *  in the file PATENTS.  All contributing project authors may
 *  be found in the AUTHORS file in the root of the source tree.
 */

package com.xdja.poc.sdk.business.webrtc.janusclientapi;

import android.os.Handler;
import android.os.HandlerThread;
import android.util.Log;

import com.xdja.poc.common.utils.GlobalContext;
import com.xdja.poc.common.utils.LogUtils;
import com.xdja.poc.sdk.utils.Utils;

import org.json.JSONException;
import org.json.JSONObject;

import java.math.BigInteger;
import java.net.URI;
import java.net.URISyntaxException;

import de.tavendo.autobahn.WebSocket.WebSocketConnectionObserver;
import de.tavendo.autobahn.WebSocketConnection;
import de.tavendo.autobahn.WebSocketException;
import de.tavendo.autobahn.WebSocketOptions;

/**
 * Negotiates signaling for chatting with https://appr.tc "rooms".
 * Uses the client<->server specifics of the apprtc AppEngine webapp.
 * <p>
 * <p>To use: create an instance of this object (registering a message handler) and
 * call connectToRoom().  Once room connection is established
 * onConnectedToRoom() callback with room parameters is invoked.
 * Messages to other party (with local Ice candidates and answer SDP) can
 * be sent after WebSocket connection is established.
 */
public class JanusWebSocketClient implements IJanusMessenger {
    private static final String TAG = "JanusWebSocketClient";
    public static final int NETWORK_ERROR = -1;
    public static final int INTERNAL_ERROR = -2;
    public static final int JSON_ERROR = -3;
    public static final int TOKEN_ERROR = -4;
    private final Handler handler;
    private final IJanusMessageObserver eventHandler;
    private final String uri;
    private WebSocketConnection ws;
    private WebSocketObserver wsObserver;

    private final JanusMessengerType type = JanusMessengerType.websocket;

    public JanusWebSocketClient(String uri, IJanusMessageObserver events) {
        this.eventHandler = events;
        this.uri = uri;

        final HandlerThread handlerThread = new HandlerThread(TAG);
        handlerThread.start();
        handler = new Handler(handlerThread.getLooper());

    }

    // --------------------------------------------------------------------
    // WebSocketChannelEvents interface implementation.
    // All events are called by WebSocketChannelClient on a local looper thread
    // (passed to WebSocket client constructor).

    // --------------------------------------------------------------------
    // Helper functions.
    private void reportError(final int err, final String errorMessage) {
        LogUtils.ELog(TAG, errorMessage);
        handler.post(new Runnable() {
            @Override
            public void run() {
                eventHandler.onError(err, errorMessage);
            }
        });
    }


    @Override
    public void connect() {
        handler.post(new Runnable() {
            @Override
            public void run() {
                try {
                    checkIfCalledOnValidThread();

                    ws = new WebSocketConnection(GlobalContext.getContext());
                    wsObserver = new WebSocketObserver();

                    ws.connect(new URI(uri), new String[]{"janus-protocol"}, wsObserver, new WebSocketOptions());
                } catch (URISyntaxException e) {
                    reportError(NETWORK_ERROR, "URI error: " + e.getMessage());
                } catch (WebSocketException e) {
                    reportError(NETWORK_ERROR, "WebSocket connection error: " + e.getMessage());
                }catch (IllegalStateException e){
                    reportError(NETWORK_ERROR, "current thread not equals handler thread" + e.getMessage());
                }
            }
        });

    }

    @Override
    public void disconnect() {
        try {
            checkIfCalledOnValidThread();
            if (ws != null){
                ws.disconnect();
            }
        }catch (Exception e){
            reportError(INTERNAL_ERROR, "current thread not equals handler thread" + e.getMessage());
        }
    }
    @Override
    public void sendMessage(String message) {
        try {
            JSONObject obj = new JSONObject(message);
            obj.put("token", Utils.genToken(message));
            message = obj.toString();
            LogUtils.DLog(TAG, "sendMessage ---with token :" + message);
            ws.sendTextMessage(message);
        } catch (JSONException e) {
            reportError(JSON_ERROR,"json is wrong");
            e.printStackTrace();
        }


    }

    @Override
    public void sendMessage(String message, BigInteger session_id) {
        //LogUtils.DLog(TAG, "sendMessage --- message :" + message + " session_id " + session_id);
        sendMessage(message);
    }

    @Override
    public void sendMessage(String message, BigInteger session_id, BigInteger handle_id) {
        //LogUtils.DLog(TAG, "sendMessage --- message :" + message + " session_id " + session_id  + " handle_id" + handle_id);
        sendMessage(message);
    }

    @Override
    public void receivedMessage(String message) {
        LogUtils.DLog(TAG, "receivedMessage  message :" + message);
        try {
            JSONObject obj = new JSONObject(message);
            eventHandler.receivedNewMessage(obj);
        } catch (Exception ex) {
            eventHandler.onError(JSON_ERROR, ex.getMessage());
        }
    }

    @Override
    public JanusMessengerType getMessengerType() {
        return type;
    }


    private class WebSocketObserver implements WebSocketConnectionObserver {
        @Override
        public void onOpen() {
            LogUtils.DLog(TAG, "WebSocket connection opened---");
            handler.post(new Runnable() {
                @Override
                public void run() {
                    eventHandler.onOpen();
                }
            });
        }

        @Override
        public void onClose(WebSocketCloseNotification webSocketCloseNotification, String s) {
            LogUtils.DLog(TAG, "WebSocketObserver WebSocket " + s);
            handler.post(new Runnable() {
                @Override
                public void run() {
                    eventHandler.onClose();
                }
            });
        }

        @Override
        public void onTextMessage(String payload) {
            LogUtils.DLog(TAG, "WebSocketObserver onTextMessage " + payload);
            final String message = payload;
            handler.post(new Runnable() {
                @Override
                public void run() {
                    receivedMessage(message);
                }
            });
        }

        @Override
        public void onRawTextMessage(byte[] bytes) {
        }

        @Override
        public void onBinaryMessage(byte[] bytes) {
        }

        @Override
        public void onError(Exception e) {
            eventHandler.onError(NETWORK_ERROR, e.getMessage());
        }
    }

    private void checkIfCalledOnValidThread() {
        if (Thread.currentThread() != handler.getLooper().getThread()) {
            //throw new IllegalStateException("WebSocket method is not called on valid thread");
        }
    }

}
