package com.xdja.poc.sdk.record.fdfs;

import com.xdja.poc.common.utils.LogUtils;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Created by gouhao on 10/30/2018.
 */

public class ThreadPoolManager {
    private static final String TAG = ThreadPoolManager.class.getSimpleName();

    /**
     * 每个线程存活时间为1分钟
     */
    private static final int KEEP_ALIVE_TIME = 1;
    private static final int MAX_EXECUTOR_THREADS = Runtime.getRuntime().availableProcessors() * 2 + 1;
    private static final int MAX_HTTP_THREADS = Runtime.getRuntime().availableProcessors() + 1;
    private static final int MAX_SCHEDULED_THREADS = Runtime.getRuntime().availableProcessors();

    private static final ThreadPoolManager INSTANCE = new ThreadPoolManager();

    private ExecutorService executorService;
    private ScheduledExecutorService scheduledExecutorService;
    private ExecutorService httpRequestExecutorService;

    public static ThreadPoolManager getInstance() {
        return INSTANCE;
    }
    private boolean init;

    public void init(){
        if(init){
            LogUtils.DLog(TAG, "init: already init. " + this);
            return;
        }
        init = true;
        LogUtils.DLog(TAG, "init: " + this);
        executorService = new ThreadPoolExecutor(MAX_EXECUTOR_THREADS, MAX_EXECUTOR_THREADS,
                KEEP_ALIVE_TIME, TimeUnit.MINUTES,
                new LinkedBlockingQueue<>(),new ThreadPoolExecutor.DiscardPolicy());
        scheduledExecutorService = new ScheduledThreadPoolExecutor(MAX_SCHEDULED_THREADS, new ThreadPoolExecutor.DiscardPolicy());
        httpRequestExecutorService = new ThreadPoolExecutor(MAX_HTTP_THREADS, MAX_HTTP_THREADS,
                KEEP_ALIVE_TIME, TimeUnit.MINUTES,
                new LinkedBlockingQueue<>(),new ThreadPoolExecutor.DiscardPolicy());
    }


    /**
     * 只用于OK_HTTP请求，其它地方不许用
     * @return
     */
    public ExecutorService getHttpRequestExecutorService(){
        return httpRequestExecutorService;
    }

    public void execute(Runnable task){
        submit(task);
    }

    public <T> Future<T> submit(Callable<T> task){
        if (checkInvalidBeforeExec(task)) return null;
        return executorService.submit(task);
    }

    public Future<?> submit(Runnable task){
        if (checkInvalidBeforeExec(task)) return null;
        return executorService.submit(task);
    }

    public ScheduledFuture<?> schedule(Runnable command, long delay, TimeUnit unit){
        if (checkInvalidBeforeExec(command)) return null;
        return scheduledExecutorService.schedule(command, delay, unit);
    }
    private boolean checkInvalidBeforeExec(Object task) {
        if(task == null) return true;
        if(!init){
            LogUtils.ELog(TAG, "checkInvalidBeforeExec: not init");
            return true;
        }
        if(executorService.isShutdown()){
            LogUtils.ELog(TAG, "checkInvalidBeforeExec: executorService is shutdown");
            return true;
        }
        return false;
    }

    public void shutDown(){
        if(!init) {
            LogUtils.DLog(TAG, "shutDown: already init. " + this);
            return;
        }
        init = false;
        LogUtils.DLog(TAG, "shutDown: " + this);
        executorService.shutdownNow();
        scheduledExecutorService.shutdownNow();
        httpRequestExecutorService.shutdownNow();
    }
}
