package com.xdja.poc.sdk.ui.history;

import android.media.MediaPlayer;

import com.xdja.poc.common.utils.DateUtils;
import com.xdja.poc.common.utils.LogUtils;
import com.xdja.poc.common.utils.StringUtils;
import com.xdja.poc.common.utils.ToastUtils;
import com.xdja.poc.sdk.config.Constants;
import com.xdja.poc.sdk.greendao.HistoryMessageDBHelper;
import com.xdja.poc.sdk.record.http.HttpClientFactory;
import com.xdja.poc.sdk.record.http.IHttpClient;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Request;
import okhttp3.Response;

/**
 * 录音播放
 * Created by xdjaxa on 2019/1/4.
 */

public class HistoryVoicePlayer {

    private static final String TAG = HistoryVoicePlayer.class.getSimpleName();
    private MediaPlayer mediaPlayer;
    private PlayVoiceCallBack playVoiceCallBack;
    private String upVoiceUrl = "";
    private IHttpClient client;
    private boolean downloading = false;

    private static HistoryVoicePlayer instance = null;

    private HistoryVoicePlayer() {
    }

    public static HistoryVoicePlayer getInstance() {
        if (instance == null) {
            synchronized (HistoryVoicePlayer.class) {
                if (instance == null) {
                    instance = new HistoryVoicePlayer();
                }
            }
        }
        return instance;
    }

    /**
     * 外部调用方法，开始播放音频
     *
     * @param message
     * @param voiceCallBack
     */
    public void startPlay(POCHistoryMessage message, PlayVoiceCallBack voiceCallBack) {
        //上一播放路径与当前播放路径一致
        if (upVoiceUrl.equals(message.getFileUrl())) {
            //音频在下载，则提示
            if (downloading) {
                ToastUtils.showToast("语音正在加载，请稍后");
                return;
            }
            //正在播放，则暂停播放
            if (mediaPlayer != null && mediaPlayer.isPlaying()) {
                releasePlay();
                if (playVoiceCallBack != null) {
                    playVoiceCallBack.onPlayStop();
                }
                return;
            }
        }

        this.upVoiceUrl = message.getFileUrl();
        this.playVoiceCallBack = voiceCallBack;

        if (playVoiceCallBack != null) {
            playVoiceCallBack.onPlayStart();
        }
        //播放文件存在，则直接播放
        String destPath = message.getLocalCacheUri();
        if (!StringUtils.isEmpty(destPath) && new File(destPath).exists()) {
            realPlay(destPath);
            return;
        }
        //播放文件不存在，下载后播放
        downloadFile(message);
    }

    /**
     * 外部调用方法，释放player
     */
    public void releasePlay() {
        if (mediaPlayer != null && mediaPlayer.isPlaying()) {
            mediaPlayer.stop();
            mediaPlayer.release();
        }
        mediaPlayer = null;
    }

    /**
     * 外部调用方法,是否正在下载
     *
     * @return
     */
    public boolean isDownloading() {
        return downloading;
    }

    /**
     * 播放音频
     *
     * @param fileUri
     */
    private void realPlay(String fileUri) {
        //释放player，解决非法状态异常
        releasePlay();

        mediaPlayer = new MediaPlayer();

        try {
            mediaPlayer.setDataSource(fileUri);
            mediaPlayer.setLooping(false);
            mediaPlayer.setOnErrorListener(new MediaPlayer.OnErrorListener() {
                @Override
                public boolean onError(MediaPlayer mp, int what, int extra) {
                    releasePlay();
                    if (playVoiceCallBack != null) {
                        playVoiceCallBack.onPlayStop();
                    }
                    return false;
                }
            });
            mediaPlayer.setOnCompletionListener(new MediaPlayer.OnCompletionListener() {
                @Override
                public void onCompletion(MediaPlayer mp) {
                    releasePlay();
                    if (playVoiceCallBack != null) {
                        playVoiceCallBack.onPlayStop();
                    }
                }
            });

            mediaPlayer.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
                @Override
                public void onPrepared(MediaPlayer mp) {
                    if (mp != null) {
                        mp.start();
                    }
                }
            });

            mediaPlayer.prepare();
        } catch (IOException e) {
            e.printStackTrace();
            releasePlay();
            if (playVoiceCallBack != null) {
                playVoiceCallBack.onPlayFailed();
            }
        }
    }

    /**
     * 下载录音文件
     *
     * @param message
     */
    private void downloadFile(POCHistoryMessage message) {
        downloading = true;
        final POCHistoryMessage currMessage = message;

        Observable.create(new ObservableOnSubscribe<String>() {
            @Override
            public void subscribe(ObservableEmitter<String> emitter) throws Exception {
                if (client == null) {
                    client = HttpClientFactory.createHttpClient();
                }

                Request request = new Request.Builder()
                        .url(currMessage.fileUrl)
                        .build();

                client.downloadFile(request, new Callback() {
                    @Override
                    public void onFailure(Call call, IOException e) {
                        downloading = false;
                        emitter.onError(e);
                    }

                    @Override
                    public void onResponse(Call call, Response response) throws IOException {
                        boolean hasError = false;
                        InputStream is = null;
                        byte[] buff = new byte[1024];
                        BufferedOutputStream fos = null;

                        File dir = new File(createRecordPath());
                        if (!dir.exists()) {
                            dir.mkdirs();
                        }
                        File file = new File(dir, currMessage.id + "." + currMessage.fileType);
                        try {
                            is = response.body().byteStream();
                            fos = new BufferedOutputStream(new FileOutputStream(file));
                            while (( is.read(buff)) != -1) {
                                fos.write(buff);
                            }
                            fos.flush();
                        } catch (IOException e) {
                            e.printStackTrace();
                            emitter.onError(e);
                            hasError = true;
                        } finally {
                            try {
                                if (is != null) {
                                    is.close();
                                }
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                            try {
                                if (fos != null) {
                                    fos.close();
                                }
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                        }
                        downloading = false;
                        if (!hasError) {
                            emitter.onNext(file.getAbsolutePath());
                        }
                    }
                });
            }
        }).subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(voiceUri -> {
                    if (!upVoiceUrl.equals(currMessage.getFileUrl())) {
                        return;
                    }
                    currMessage.setLocalCacheUri(voiceUri);
                    HistoryMessageDBHelper.getInstance().insertOrUpdate(currMessage);

                    realPlay(voiceUri);
                }, e -> {
                    if (!upVoiceUrl.equals(currMessage.getFileUrl())) {
                        return;
                    }
                    if (playVoiceCallBack != null) {
                        playVoiceCallBack.onPlayFailed();
                    }
                });
    }

    /**
     * 生成本地保存录音路径
     *
     * @return
     */
    private String createRecordPath() {
        //根据时间生成文件路径，方便设置过期时限删除
        return Constants.HISTORY_RECORD_DOWNLOAD_PATH + getTime() + File.separator;
    }

    /**
     * 获取yyyyMMdd时间戳
     *
     * @return
     */
    private long getTime() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd", Locale.CHINA);
        long time = 0L;
        try {
            Date destDate = dateFormat.parse(dateFormat.format(new Date()));
            time = destDate.getTime();
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return time;
    }

    /**
     * 播放音频回调
     */
    public interface PlayVoiceCallBack {

        void onPlayStart();

        void onPlayStop();

        void onPlayFailed();
    }
}
