package com.xdja.poc.sdk.utils;

import android.animation.ObjectAnimator;
import android.content.Context;
import android.content.res.AssetFileDescriptor;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.media.MediaPlayer.OnCompletionListener;
import android.media.MediaPlayer.OnPreparedListener;
import android.support.annotation.IntDef;
import android.util.Log;
import android.util.Property;
import android.util.SparseArray;
import android.util.SparseBooleanArray;

import com.xdja.poc.common.utils.LogUtils;
import com.xdja.poc.sdk.BuildConfig;
import com.xdja.poc.sdk.R;
import com.xdja.poc.sdk.ui.conversion.POCWarningToneCompleteCallBack;

import java.io.IOException;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * 按下松开声音播放
 */
public class MusicHelper {
    public static final int TYPE_END_GO = 1;
    public static final int TYPE_READY_GO = 2;
    private SparseArray<MusicPlayer> mPlayers = new SparseArray<>();
    private SparseBooleanArray mDisableMusics = new SparseBooleanArray();
    private Context mAppContext;
    private static MusicHelper sSelf;

    @Retention(RetentionPolicy.SOURCE)
    @IntDef({TYPE_END_GO, TYPE_READY_GO})
    public @interface Type {
    }

    public static String typeToString(@Type int type) {
        switch (type) {
            case TYPE_END_GO:
                return "end-background";
            case TYPE_READY_GO:
                return "ready-go";
            default:
                return "unknow-" + type;
        }
    }

    public static MusicHelper getInstance(Context context) {
        if (sSelf != null) return sSelf;
        synchronized (MusicHelper.class) {
            if (sSelf == null) sSelf = new MusicHelper(context);
        }
        return sSelf;
    }

    private MusicHelper(Context context) {
        mAppContext = context.getApplicationContext();
        mPlayers.append(TYPE_READY_GO, new MusicPlayer(R.raw.ppt_start,
                typeToString(TYPE_READY_GO), mAppContext, false));
        mPlayers.append(TYPE_END_GO, new MusicPlayer(R.raw.ppt_stop,
                typeToString(TYPE_END_GO), mAppContext, false));


    }

    public synchronized MusicPlayer playMusic(@Type int type, boolean fadeIn) {
        MusicPlayer player = mPlayers.get(type);
        if (player == null) {
            throw new IllegalArgumentException("Unsupport type:" + typeToString(type));
        }
        if (mDisableMusics.get(type)) return player;
        player.play(fadeIn);
        return player;
    }

    public MusicHelper pauseMusic(@Type int type, boolean fadeOut) {
        MusicPlayer player = mPlayers.get(type);
        if (player == null) {
            throw new IllegalArgumentException("Unsupport type:" + typeToString(type));
        }
        if (mDisableMusics.get(type)) return this;
        player.pause(fadeOut);
        return this;
    }

    public void disableMusic(@Type int... music) {
        if (music == null || music.length == 0) return;
        for (@Type int type : music) {
            mDisableMusics.put(type, true);
        }
    }

    public static class MusicPlayer {
        private static final String TAG = "MusicPlayer";
        private static final int STATE_STOPED = 1;
        private static final int STATE_PREPARING = 2;
        private static final int STATE_PLAYING_FADING = 3;
        private static final int STATE_PLAYING = 4;
        private static final int STATE_PAUSE_FADING = 5;
        private static final int STATE_PAUSED = 6;
        private MediaPlayer mMediaPlayer;
        private final int mMusicId;
        private final String mType;
        private final Context mAppContext;
        private final boolean mLoop;
        @State
        private int mState = STATE_STOPED;
        private boolean mFade;
        private float mCurrentVolume;
        private ObjectAnimator mVolumeAnimator;
        private POCWarningToneCompleteCallBack mCompleteListener;

        public void setCompleteListener(POCWarningToneCompleteCallBack callBack) {
            mCompleteListener = callBack;
        }

        private OnPreparedListener mPreparedListener = new OnPreparedListener() {
            @Override
            public void onPrepared(MediaPlayer mp) {
                LogUtils.ELog(TAG, "mState: " + mState);
                if (mState != STATE_PREPARING) {
                    if (mCompleteListener != null) {
                        mCompleteListener.completeCallBack();
                    }
                    return;
                }
                LogUtils.ELog(TAG, "mLoop: " + mLoop);
                if (mLoop) mp.setLooping(true);
                LogUtils.ELog(TAG, "mFade: " + mFade);
                if (mFade) {
                    mp.setVolume(1.0f, 1.0f);
                    setState(STATE_PLAYING_FADING);
                    fade(true);
                } else {
                    mMediaPlayer.setVolume(0.5f, 0.5f);//for half volume.
                    setState(STATE_PLAYING);
                }
                mp.start();
            }
        };
        private OnCompletionListener mCompletionListener = new OnCompletionListener() {
            @Override
            public void onCompletion(MediaPlayer mp) {
                if (mState != STATE_STOPED) {
                    if (mLoop) {
                        mp.start();
                    } else {
                        setState(STATE_STOPED);
                    }
                }
                releasePlayer();
                if (mCompleteListener != null) {
                    mCompleteListener.completeCallBack();
                }
            }
        };

        @Retention(RetentionPolicy.SOURCE)
        @IntDef({STATE_STOPED, STATE_PREPARING, STATE_PLAYING_FADING, STATE_PLAYING,
                STATE_PAUSE_FADING, STATE_PAUSED})
        public @interface State {
        }

        private void clearVolumeAnimator() {
            if (mVolumeAnimator != null) {
                mVolumeAnimator.cancel();
                mVolumeAnimator = null;
            }
        }

        private void fade(final boolean in) {
          /*  clearVolumeAnimator();
            if (in) {
                mVolumeAnimator = ObjectAnimator.ofFloat(this, new VolumeProperty(), mCurrentVolume, 1f)
                        .setDuration(C.SCENE_ANIMATION_DURATION);
            } else {
                mVolumeAnimator = ObjectAnimator.ofFloat(this, new VolumeProperty(), mCurrentVolume, 0f)
                        .setDuration(C.SCENE_ANIMATION_DURATION);
            }
            mVolumeAnimator.addListener(new SimpleAnimatorListener() {
                @Override
                public void onAnimationEnd(Animator animation) {
                    mVolumeAnimator = null;
                    if (in) setState(STATE_PLAYING);
                    else {
                        setState(STATE_PAUSED);
                        mMediaPlayer.pause();
                    }
                }
            });
            mVolumeAnimator.start();*/
        }

        public void setVolume(float volume) {
            mCurrentVolume = volume;
            if (mMediaPlayer != null) mMediaPlayer.setVolume(volume, volume);
        }

        public float getVolume() {
            return mCurrentVolume;
        }

        public static String stateToString(@State int state) {
            switch (state) {
                case STATE_STOPED:
                    return "stoped";
                case STATE_PAUSED:
                    return "paused";
                case STATE_PAUSE_FADING:
                    return "pause-fading";
                case STATE_PLAYING_FADING:
                    return "playing-fading";
                case STATE_PLAYING:
                    return "playing";
                case STATE_PREPARING:
                    return "preparing";
                default:
                    return "unknow-" + state;
            }
        }

        private void setState(@State int state) {
            if (state == mState) return;
            if (!BuildConfig.DEBUG) {
                LogUtils.DLog("MusicHelper", String.format("setState for %s, %s -> %s", mType, stateToString(mState),
                        stateToString(state)));
            }
            mState = state;
        }

        public MusicPlayer(int mMusicId, String mType, Context mAppContext, boolean mLoop) {
            this.mMusicId = mMusicId;
            this.mType = mType;
            this.mAppContext = mAppContext;
            this.mLoop = mLoop;
        }

        public void releasePlayer() {
            try {
                if (mMediaPlayer != null) {
                    mMediaPlayer.stop();
                    mMediaPlayer.release();
                    mMediaPlayer.setOnCompletionListener(null);
                    mMediaPlayer.setOnPreparedListener(null);
                    mMediaPlayer = null;
                }
                setState(STATE_STOPED);
                clearVolumeAnimator();
            } catch (Exception ex) {

            }
            mCurrentVolume = 0;
        }

        public synchronized void play(boolean fade) {
            mFade = fade;
            switch (mState) {
                case STATE_PREPARING:
                case STATE_PLAYING:
                    return;
                case STATE_STOPED:
                    releasePlayer();
                    mMediaPlayer = new MediaPlayer();
                    AssetFileDescriptor afd = mAppContext.getResources().openRawResourceFd(mMusicId);
                    try {
                        mMediaPlayer.setAudioStreamType(AudioManager.STREAM_RING);
                        mMediaPlayer.setDataSource(afd.getFileDescriptor(), afd.getStartOffset(),
                                afd.getLength());
                        mMediaPlayer.setOnPreparedListener(mPreparedListener);
                        mMediaPlayer.setOnCompletionListener(mCompletionListener);
                        mMediaPlayer.setOnErrorListener(new MediaPlayer.OnErrorListener() {
                            @Override
                            public boolean onError(MediaPlayer mediaPlayer, int i, int i1) {
                                releasePlayer();
                                if (mCompleteListener != null) {
                                    mCompleteListener.completeCallBack();
                                }
                                return false;
                            }
                        });
                        setState(STATE_PREPARING);
                        mMediaPlayer.prepareAsync();
                    } catch (Exception e) {
                        releasePlayer();
                        if (mCompleteListener != null) {
                            mCompleteListener.completeCallBack();
                        }
                        setState(STATE_STOPED);
                        e.printStackTrace();
                    }
                    break;
                case STATE_PAUSED:
                case STATE_PAUSE_FADING:
                case STATE_PLAYING_FADING:
                    try {
                        if (fade) {
                            mMediaPlayer.setVolume(0f, 0f);
                            mMediaPlayer.start();
                            setState(STATE_PLAYING_FADING);
                            fade(true);
                        } else {
                            mMediaPlayer.setVolume(0.5f, 0.5f);//for half volume.
                            setState(STATE_PLAYING);
                            mMediaPlayer.start();
                        }
                    } catch (Exception e) {
                        LogUtils.ELog(TAG, e.getLocalizedMessage());
                        releasePlayer();
                        if (mCompleteListener != null) {
                            mCompleteListener.completeCallBack();
                        }
                        setState(STATE_STOPED);

                    }
                    break;
            }
        }

        public synchronized void pause(boolean fade) {
            mFade = fade;
            switch (mState) {
                case STATE_PLAYING:
                case STATE_PLAYING_FADING:
                    if (fade) {
                        setState(STATE_PAUSE_FADING);
                        fade(false);
                    } else {
                        setState(STATE_PAUSED);
                        mMediaPlayer.pause();
                    }
                    break;
                case STATE_PREPARING:
                case STATE_STOPED:
                case STATE_PAUSED:
                case STATE_PAUSE_FADING:
                    break;
            }
        }
    }

    public static class VolumeProperty extends Property<MusicPlayer, Float> {
        public VolumeProperty() {
            super(Float.class, "volume");
        }

        @Override
        public Float get(MusicPlayer object) {
            return object.getVolume();
        }

        @Override
        public void set(MusicPlayer object, Float value) {
            object.setVolume(value);
        }
    }
}
