package com.xdja.poc.sdk.greendao;

import com.xdja.poc.common.utils.LogUtils;
import com.xdja.poc.common.utils.StringUtils;
import com.xdja.poc.sdk.greendao.dao.POCHistoryMessageDao;
import com.xdja.poc.sdk.greendao.manager.GreenDaoManager;
import com.xdja.poc.sdk.ui.history.POCHistoryMessage;

import java.io.File;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Locale;

/**
 * Created by xdjaxa on 2019/1/11.
 */

public class HistoryMessageDBHelper {

    private static final String TAG = HistoryMessageDBHelper.class.getSimpleName();
    private final GreenDaoManager daoManager;
    private static HistoryMessageDBHelper dbHelper;
    //最大缓存消息时间间隔
    private static final long MAX_CACHE_MESSAGE_TIME_INTERVAL = 6 * 24 * 60 * 60 * 1000;

    private HistoryMessageDBHelper() {
        daoManager = GreenDaoManager.getInstance();
    }

    public static HistoryMessageDBHelper getInstance() {
        if (dbHelper == null) {
            synchronized (HistoryMessageDBHelper.class) {
                if (dbHelper == null) {
                    dbHelper = new HistoryMessageDBHelper();
                }
            }
        }
        return dbHelper;
    }

    /**
     * 插入或更新单个对象
     *
     * @param historyMessage
     */
    public synchronized void insertOrUpdate(POCHistoryMessage historyMessage) {
        delete();
        if (historyMessage == null) {
            return;
        }
        getPocHistoryMessageDao().insertOrReplaceInTx(historyMessage);
        LogUtils.DLog(TAG, "insert success");
    }

    /**
     * 插入或更新集合数据
     *
     * @param historyMessages
     */
    public synchronized void insertOrUpdate(List<POCHistoryMessage> historyMessages) {
        delete();
        if (historyMessages == null || historyMessages.size() == 0) {
            return;
        }
        long time = getTime();
        for (POCHistoryMessage message : historyMessages) {
            message.setLocalCacheTime(String.valueOf(time));
        }
        getPocHistoryMessageDao().insertOrReplaceInTx(historyMessages);
        LogUtils.DLog(TAG, "insert success");
    }

    /**
     * 根据房间号、大小、时间获取本地消息
     *
     * @param room
     * @param pageSize
     * @param time
     * @return
     */
    public synchronized List<POCHistoryMessage> query(String room, int pageSize, long time) {
        delete();
        List<POCHistoryMessage> result;
        if (time == 0) {
            result = getPocHistoryMessageDao().queryBuilder()
                    .where(POCHistoryMessageDao.Properties.GroupUri.eq(room))
                    .orderDesc(POCHistoryMessageDao.Properties.SyncTime)
                    .limit(pageSize)
                    .build()
                    .list();
        } else {
            result = getPocHistoryMessageDao().queryBuilder().where(POCHistoryMessageDao.Properties.SyncTime.lt(time),
                    POCHistoryMessageDao.Properties.GroupUri.eq(room))
                    .orderAsc(POCHistoryMessageDao.Properties.SyncTime)
                    .limit(pageSize)
                    .build()
                    .list();
        }
        Collections.sort(result, new Comparator<POCHistoryMessage>() {
            @Override
            public int compare(POCHistoryMessage o1, POCHistoryMessage o2) {
                return o1.getSyncTime().compareTo(o2.getSyncTime());
            }
        });
        LogUtils.DLog(TAG, "query success");
        return result;
    }
    public synchronized List<POCHistoryMessage> getAllData(String room) {
        delete();
        List<POCHistoryMessage> result;
        result = getPocHistoryMessageDao().queryBuilder().where(
                POCHistoryMessageDao.Properties.GroupUri.eq(room))
                .orderAsc(POCHistoryMessageDao.Properties.SyncTime)
                .build()
                .list();
        Collections.sort(result, new Comparator<POCHistoryMessage>() {
            @Override
            public int compare(POCHistoryMessage o1, POCHistoryMessage o2) {
                return o1.getSyncTime().compareTo(o2.getSyncTime());
            }
        });
        LogUtils.DLog(TAG, "query success");
        return result;
    }
    /**
     * 根据消息缓存时间删除本地消息
     */
    public synchronized void delete() {
        //按照消息拉取时间删除缓存
        long time = getTime() - MAX_CACHE_MESSAGE_TIME_INTERVAL;
        List<POCHistoryMessage> result = getPocHistoryMessageDao().queryBuilder()
                .where(POCHistoryMessageDao.Properties.LocalCacheTime.le(time))
                .build()
                .list();
        if (result != null && result.size() > 0) {
            LogUtils.DLog(TAG, "缓存消息超过最大缓存期限，触发自动删除机制");
            deleteLocalFile(result);
            getPocHistoryMessageDao().deleteInTx(result);
        }
        LogUtils.DLog(TAG, "delete success");
    }

    /**
     * 根据房间号统计消息数
     *
     * @param room
     * @return
     */
    public long count(String room) {
        delete();
        return getPocHistoryMessageDao()
                .queryBuilder()
                .where(POCHistoryMessageDao.Properties.GroupUri.eq(room))
                .build()
                .list()
                .size();
    }

    /**
     * 获取数据库操作对象
     *
     * @return
     */
    private POCHistoryMessageDao getPocHistoryMessageDao() {
        return daoManager.getDaoSession().getPOCHistoryMessageDao();
    }

    /**
     * 删除本地缓存过期文件及文件夹
     *
     * @param result
     */
    private void deleteLocalFile(List<POCHistoryMessage> result) {
        for (POCHistoryMessage message : result) {
            String uri = message.getLocalCacheUri();
            if (!StringUtils.isEmpty(uri)) {
                //删除过期文件
                File file = new File(uri);
                if (file.exists()) {
                    file.delete();
                }
                //删除过期文件所在文件夹（无文件）
                File dir = new File(uri.substring(0, uri.lastIndexOf("/") - 1));
                if (dir.list() != null && dir.list().length == 0){
                    dir.delete();
                }
            }
        }
    }

    /**
     * 获取yyyyMMdd时间戳
     *
     * @return
     */
    private long getTime() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd", Locale.CHINA);
        long time = 0L;
        try {
            Date destDate = dateFormat.parse(dateFormat.format(new Date()));
            time = destDate.getTime();
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return time;
    }
}
