package com.xdja.poc.sdk.record.http;

import android.text.TextUtils;
import android.util.SparseArray;

import com.google.gson.Gson;
import com.xdja.poc.common.utils.LogUtils;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

/**
 * Created by gouhao on 5/28/2018.
 */

public abstract class BaseHttpClient implements IHttpClient {
    protected String TAG = getClass().getSimpleName();

    protected static final int DEFAULT_HTTP_TIMEOUT = 30; //30S
    protected static final String DEFAULT_CONTENT_TYPE = "application/json";
    protected static final String CONTENT_TYPE_FORM = "multipart/form-data";
    protected static final String TYPE_UPLOAD_FILE = "application/octet-stream";
    protected static final int HTTP_RESPONSE_OK = 200;

    protected OkHttpClient httpClient;
    private int requestCounter = 0;
    private SparseArray<Call> requestCallMap = new SparseArray<>();
    private SparseArray<UploadFileRequestBody> uploadRequestBodyMap = new SparseArray<>();

    @Override
    public int request(HttpRequest request, Callback callback) {
        checkNotNull(request);
        return dealRequest(request.getMethod(), request.getUrl(), request.getParamJsonString(), request.getHeaders(), callback);
    }

    protected void checkNotNull(Object obj) {
        if (obj == null) {
            throw new NullPointerException("checkNotNull: obj is null");
        }
    }

    @Override
    public int exec(String url, Map<String, Object> params, Map<String, String> headers, Callback callback) {
        if (TextUtils.isEmpty(url)) {
            LogUtils.ELog(TAG, "url is null");
            return 0;
        }
        //默认为post请求
        return exec(METHOD_TYPE_POST, url, params, headers, callback);
    }

    @Override
    public int exec(String methodType, String url, Object body, Map<String, String> headers, Callback callback) {
        if (TextUtils.isEmpty(url)) {
            LogUtils.ELog(TAG, "url is null");
            return 0;
        }
        return dealRequest(methodType, url, body, headers, callback);
    }

//
//    private boolean checkTokenTimeout(String responseBody) {
//        SharedPreferenceUtil sp = SharedPreferenceUtil.getInstance(Xypolice.getAppContext());
//        boolean isLogin = sp.getBoolean(Constants.PREF_IS_LOGIN);
//        if(!isLogin) {
//            LogUtil.e(TAG, "checkTokenTimeout: user is logout");
//            return false;
//        }
//        //只对MIS请求做TOKEN检测，IM的群组返回的直接是个ARRAY
//        if(!responseBody.startsWith("{")){
//            return false;
//        }
//        try {
//            JSONObject jsonObject = new JSONObject(responseBody);
//            if(jsonObject.has(MisKey.Response.KEY_ERROR_CODE) && MisKey.Response.ERROR_TOKEN_TIMEOUT.equals(jsonObject.getString(MisKey.Response.KEY_ERROR_CODE))){
//                //TOKEN过期
//                LogUtil.e(TAG, "checkTokenTimeout: token time out");
//                EventBus.getDefault().post(new CommonCommand(CommonCommand.CODE_TOKEN_TIMEOUT));
//                return true;
//            } else{
//                return false;
//            }
//        }catch (Exception e) {
//            LogUtil.e(TAG, "checkTokenTimeout: error: " + e.toString());
//        }
//        return false;
//    }

    @Override
    public int execFormData(String url, Map<String, String> params, Map<String, String> headers, Callback callback) {
        if (TextUtils.isEmpty(url)) {
            LogUtils.ELog(TAG, "url is null");
            throw new NullPointerException("url is null");
        }

        Request.Builder builder = new Request.Builder();

        //添加URL
        builder.url(url);
        LogUtils.DLog(TAG, "execFormData: url: " + url);

        //如果有Header，添加Header
        if (headers != null && !headers.isEmpty()) {
            LogUtils.DLog(TAG, "execFormData: headers:" + headers.toString());
            addHeader(builder, headers);
        }

        //创建请求参数
        RequestBody requestBody = null;
        if (params != null && !params.isEmpty()) {
            requestBody = createRequestBody(params);
        }

        //根据请求参数，构造POST请求
        builder.post(requestBody);
        int requestId = getRequestId();
        realRequest(requestId, builder.build(), callback);
        return requestId;
    }

    private RequestBody createRequestBody(Map<String, String> params) {
        if (params == null || params.isEmpty()) return null;
        MultipartBody.Builder builder = new MultipartBody.Builder();
        for (String key : params.keySet()) {
            String value = params.get(key);
            builder.addFormDataPart(key, value);
        }
        return builder.build();
    }

    @Override
    public int execGet(String url, Map<String, String> headers, Callback callback) {
        if (TextUtils.isEmpty(url)) {
            LogUtils.ELog(TAG, "url is null");
            throw new NullPointerException("url is null");
        }
        return exec(METHOD_TYPE_GET, url, null, headers, callback);
    }

    /**
     * 处理请求操作
     *
     * @param methodType POST,GET,PUT,DELETE,
     * @param url
     * @param obj
     * @param headers
     * @param callback
     */
    private int dealRequest(String methodType, String url, Object obj, Map<String, String> headers, Callback callback) {
        Request.Builder builder = new Request.Builder();

        //添加URL
        builder.url(url);
        LogUtils.DLog(TAG, "exec: url: " + url);

        //如果有Header，添加Header
        if (headers != null && !headers.isEmpty()) {
            LogUtils.DLog(TAG, "exec: headers:" + headers.toString());
            addHeader(builder, headers);
        }

        //创建请求参数
        String requestJson = "{}";
        if (obj instanceof String) {
            String body = (String) obj;
            if (!TextUtils.isEmpty(body)) {
                requestJson = body;
                LogUtils.DLog(TAG, "exec: request: " + requestJson);
            }
        } else if (obj instanceof Map) {
            Map<String, Object> params = (Map<String, Object>) obj;
            if (params != null && !params.isEmpty()) {
                requestJson = new Gson().toJson(params);
                LogUtils.DLog(TAG, "exec: request: " + requestJson);
            }
        }
        //根据请求参数，构造Method类型
        switch (methodType) {
            case METHOD_TYPE_POST:
                builder.post(RequestBody.create(MediaType.parse(HttpClient.DEFAULT_CONTENT_TYPE), requestJson));
                break;
            case METHOD_TYPE_GET:
                builder.get();
                break;
            case METHOD_TYPE_PUT:
                builder.put(RequestBody.create(MediaType.parse(HttpClient.DEFAULT_CONTENT_TYPE), requestJson));
                break;
            case METHOD_TYPE_DELETE:
                builder.delete(RequestBody.create(MediaType.parse(HttpClient.DEFAULT_CONTENT_TYPE), requestJson));
                break;
            case METHOD_TYPE_PATHCH:
                builder.patch(RequestBody.create(MediaType.parse(HttpClient.DEFAULT_CONTENT_TYPE), requestJson));
                break;
            default:
        }

        Request request = builder.build();
        int requestId = getRequestId();
        realRequest(requestId, request, callback);
        return requestId;
    }

    private int realRequest(final int requestId, Request request, Callback callback) {
        Call call = httpClient.newCall(request);
        requestCallMap.put(requestId, call);

        call.enqueue(new okhttp3.Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                LogUtils.ELog(TAG, "exec: onFailure: " + e.toString());
                requestCallMap.delete(requestId);
                if (callback != null) {
                    callback.onFailed(requestId, E_IO, e.toString());
                }
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                requestCallMap.delete(requestId);
                dealOnResponse(requestId, response, callback);
            }
        });
        return requestId;
    }

    private synchronized int getRequestId() {
        return ++requestCounter;
    }

    @Override
    public void cancelRequest(int requestId) {
        Call call = requestCallMap.get(requestId);
        if (call != null) {
            UploadFileRequestBody uploadFileRequestBody = uploadRequestBodyMap.get(requestId);
            LogUtils.DLog(TAG, "cancelRequest: requestId: " + requestId);
            if (uploadFileRequestBody != null) {
                LogUtils.DLog(TAG, "cancelRequest: cancel upload");
                uploadFileRequestBody.setCanceled(true);
            } else {
                call.cancel();
                requestCallMap.delete(requestId);
            }
        }
    }

    /**
     * deal response result
     *
     * @param requestId
     * @param response
     * @param callback
     */
    private void dealOnResponse(int requestId, Response response, Callback callback) {
        if (response == null || response.body() == null) {
            LogUtils.ELog(TAG, "exec: response: response is null");
            if (callback != null) {
                callback.onFailed(requestId, E_RESPONSE_BODY, "response is null");
            }
            return;
        }
        try {
            final byte[] responseBytes = response.body().bytes();
            String responseBody = new String(responseBytes);
            int responseCode = response.code();
            LogUtils.DLog(TAG, "exec: response: responseCode: " + responseCode + ", responseBody: " + responseBody);
            if (responseCode == HttpClient.HTTP_RESPONSE_OK) {
//                if(checkTokenTimeout(responseBody)){
//                    return;
//                }
                if (callback != null) {
                    callback.onSuccess(requestId, responseBody);
                }
            } else {
                if (callback != null) {
                    callback.onFailed(requestId, responseCode, responseBody);
                }
            }
        } catch (Exception e) {
            LogUtils.ELog(TAG, "dealOnResponse: error: " + e);
            if (callback != null) {
                callback.onFailed(requestId, E_RESPONSE_BODY, "unknown error: " + e.toString());
            }
        }
    }

    private void addHeader(Request.Builder builder, Map<String, String> headers) {
        for (String key : headers.keySet()) {
            String value = headers.get(key);
            builder.addHeader(key, value);
        }
    }

    @Override
    public int uploadFile(HttpUploadFileRequest uploadFileRequest, UploadFileCallback callback) {
        checkNotNull(uploadFileRequest);
        Request.Builder builder = new Request.Builder();

        //添加URL
        String url = uploadFileRequest.getUrl();
        builder.url(url);
        LogUtils.DLog(TAG, "exec: url: " + url);

        //如果有Header，添加Header
        Map<String, String> headers = uploadFileRequest.getHeaders();
        if (headers != null && !headers.isEmpty()) {
            LogUtils.DLog(TAG, "exec: headers:" + headers.toString());
            addHeader(builder, headers);
        }
        File file = new File(uploadFileRequest.getFilePath());
        int requestId = getRequestId();
        UploadFileRequestBody body = new UploadFileRequestBody(requestId, file, uploadFileRequest.getLastUploadSize(), uploadFileRequest.getBlockSize(), callback);
        builder.post(body);

        Request request = builder.build();

        realRequest(requestId, request, callback);
        uploadRequestBodyMap.put(requestId, body);
        return requestId;
    }

    @Override
    public int downloadFile(Request request, okhttp3.Callback callback) {
        httpClient.newCall(request).enqueue(callback);
        return 0;
    }
}
