package com.parfoismeng.slidebacklib;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.graphics.Color;
import android.util.DisplayMetrics;
import android.view.MotionEvent;
import android.view.View;
import android.widget.FrameLayout;
import com.parfoismeng.slidebacklib.callback.SlideBackCallBack;
import com.parfoismeng.slidebacklib.widget.SlideBackIconView;

/**
 * author : ParfoisMeng
 * time   : 2018/12/19
 * desc   : SlideBack管理器
 */
public class SlideBackManager {
    private Activity activity;
    private SlideBackCallBack callBack;
    private boolean isSideSlide=false;
    private float maxSlideLength;

    private SlideBackIconView slideBackIconView;

    /**
     * 需要使用滑动的页面注册
     *
     * @param activity 页面Act
     * @param callBack 回调
     */
    @SuppressLint("ClickableViewAccessibility")
    SlideBackManager register(Activity activity, SlideBackCallBack callBack) {
        this.activity = activity;
        this.callBack = callBack;

        DisplayMetrics dm = activity.getResources().getDisplayMetrics();
        float screenWidth = dm.widthPixels;
        float screenHeight = dm.heightPixels;

        maxSlideLength = screenWidth / 13; // 这里我设置为 屏宽/12

        // 初始化SlideBackIconView
        slideBackIconView = new SlideBackIconView(activity);
        slideBackIconView.setBackViewColor(Color.BLACK);
        slideBackIconView.setBackViewHeight(screenHeight / 4);
        slideBackIconView.setArrowSize(dp2px(5));
        slideBackIconView.setMaxSlideLength(maxSlideLength);

        // 获取decorView并设置OnTouchListener监听
        FrameLayout container = (FrameLayout) activity.getWindow().getDecorView();
        container.addView(slideBackIconView);
        container.setOnTouchListener(new View.OnTouchListener() {
            private float downX = 0; // 按下的X轴坐标

            @Override
            public boolean onTouch(View v, MotionEvent event) {
                switch (event.getAction()) {
                    case MotionEvent.ACTION_DOWN: // 按下
                        downX = event.getRawX(); // 更新按下点的X轴坐标
                        if (downX <= maxSlideLength) { // 检验是否从边缘开始滑动
                            isSideSlide = true;
                        }
                        break;
                    case MotionEvent.ACTION_MOVE: // 移动
                        if(downX==0){
                            downX = event.getRawX(); // 更新按下点的X轴坐标
                        }
                        if (downX <= maxSlideLength) { // 检验是否从边缘开始滑动
                            isSideSlide = true;
                        }
                        if (isSideSlide) { // 是从边缘开始滑动
                            float moveX = event.getRawX() - downX; // 获取X轴位移距离
                            if (Math.abs(moveX) <= maxSlideLength * 2) {
                                // 如果位移距离在可拉动距离内，更新SlideBackIconView的当前拉动距离并重绘
                                slideBackIconView.updateSlideLength(Math.abs(moveX) / 2);
                            }
                            // 根据Y轴位置给SlideBackIconView定位
                            setSlideBackPosition(slideBackIconView, (int) (event.getRawY()));
                        }
                        break;
                    case MotionEvent.ACTION_UP: // 抬起
                        // 是从边缘开始滑动 且 抬起点的X轴坐标大于某值(4倍最大滑动长度)
                        if (isSideSlide && event.getRawX() >= maxSlideLength * 1.1) {
                            if (null != SlideBackManager.this.callBack) {
                                // 不为空则响应回调事件
                                SlideBackManager.this.callBack.onSlideBack();
                            }
                        }
                        isSideSlide = false; // 从边缘开始滑动结束
                        slideBackIconView.updateSlideLength(0); // 恢复0
                        downX=0;
                        break;
                }
                return isSideSlide;
            }
        });

        return this;
    }

    /**
     * 页面销毁时记得解绑
     * 其实就是置空防止内存泄漏
     */
    void unregister() {
        activity = null;
        callBack = null;
        maxSlideLength = 0;
        slideBackIconView = null;
    }

    /**
     * 给SlideBackIconView设置topMargin，起到定位效果
     *
     * @param view     SlideBackIconView
     * @param position 触点位置
     */
    public void setSlideBackPosition(SlideBackIconView view, int position) {
        // 触点位置减去SlideBackIconView一半高度即为topMargin
        int topMargin = (int) (position - (view.getBackViewHeight() / 2));
        FrameLayout.LayoutParams layoutParams = new FrameLayout.LayoutParams(view.getLayoutParams());
        layoutParams.topMargin = topMargin;
        view.setLayoutParams(layoutParams);
    }
    public SlideBackIconView getSlideView(){
        return slideBackIconView;
    }

    public void updatePosition(float position){
        if(slideBackIconView!=null){
            slideBackIconView.updateSlideLength(position);
        }
    }
    private float dp2px(float dpValue) {
        return dpValue * activity.getResources().getDisplayMetrics().density + 0.5f;
    }
    public void setSlideStatus(boolean status){
        isSideSlide=status;
    }
}