package com.xdja.aircert.device;

import android.content.Context;
import android.content.res.AssetManager;
import android.text.TextUtils;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.XDJA_FILE;
import com.xdja.aircert.config.AppConfig;
import com.xdja.cryptodev.CryptoDevInfo;
import com.xdja.cryptodev.CryptoDevManager;
import com.xdja.cryptodev.CryptoDevType;
import com.xdja.cryptodev.devapi.CryptoInstance;
import com.xdja.cryptodev.driver.chipmanager.ChipManagerDriver;
import com.xdja.multichip.param.CertBean;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Principal;
import java.security.Security;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by Administrator on 2019/9/24 0024.
 */

public class DeviceWrapper {

    private static void scanningDevice() {
        boolean isScanningDevice = CryptoDevManager.getInstance().isScanning();
        while (isScanningDevice) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            isScanningDevice = CryptoDevManager.getInstance().isScanning();
        }
    }

    public static Map<Integer, CryptoDevInfo> getDevInfo() {

        scanningDevice();

        Map<Integer, CryptoDevInfo> infoMap = new HashMap<>();
        CryptoDevInfo sdDev;
        CryptoDevInfo chipDev;
        CryptoDevInfo simKeyDev;
        CryptoDevInfo softDev;

        List<CryptoDevInfo> devList = getDeviceList();
        if (devList == null || devList.size() <= 0) {
            return null;
        }
        //        ModuleLog.e("获取设备列表devList.size() " + devList.size());
        for (CryptoDevInfo devInfo : devList) {
            switch (devInfo.getType()) {
                case DEV_TYPE_SdCard:
                    sdDev = devInfo;
                    infoMap.put(CryptoDevType.DEV_TYPE_SdCard.getType(), sdDev);
                    break;
                case DEV_TYPE_Chip:
                    chipDev = devInfo;
                    infoMap.put(CryptoDevType.DEV_TYPE_Chip.getType(), chipDev);
                    break;
                case DEV_TYPE_SimKey:
                    simKeyDev = devInfo;
                    infoMap.put(CryptoDevType.DEV_TYPE_SimKey.getType(), simKeyDev);
                    break;
                case DEV_TYPE_Soft:
                    softDev = devInfo;
                    infoMap.put(CryptoDevType.DEV_TYPE_Soft.getType(), softDev);
                    break;

            }
        }
        return infoMap;
    }

    /**
     * 获取加密设备的列表
     * author jff
     *
     * @return
     */
    private static List<CryptoDevInfo> getDeviceList() {
        List<CryptoDevInfo> devList = CryptoDevManager.getInstance().getCryptoDevInfos();
        if (devList.size() <= 0) {
            return null;
        }
        return devList;
    }

    private static CertSKF splitCertStr(CryptoDevType type, String certStr) {

        if (certStr.equals("") || certStr.length() <= 0) {
            return null;
        }

        //“cert”: “签名证书Base64#加密证书Base64#加密密钥” //证书信息
        certStr = certStr.replaceAll("\r", "");
        certStr = certStr.replaceAll("\n", "");
        certStr = certStr.replaceAll("-----BEGIN CERTIFICATE-----", "");
        certStr = certStr.replaceAll("-----END CERTIFICATE-----", "");
        String[] pieces = certStr.split("#");
        //        ModuleLog.e("cert分隔后数据长度 " + pieces.length);

        CertSKF certSKF = new CertSKF();

        switch (pieces.length) {

            case 4://签名证书base64#加密证书base64#私钥结构体#项目标签
                if (!TextUtils.isEmpty(pieces[0])) {
                    certSKF.setSignCert(Base64.decode(pieces[0].getBytes()));
                }
                if (!TextUtils.isEmpty(pieces[1])) {
                    certSKF.setEnCodeCert(Base64.decode(pieces[1].getBytes()));
                }

                if (!TextUtils.isEmpty(pieces[2])) {
                    certSKF.setKeyPair(Base64.decode(pieces[2].getBytes()));
                }
                if (!TextUtils.isEmpty(pieces[3])) {
                    certSKF.setProjectLabel(pieces[3]);
                }
                certSKF.setLen(4);
                break;
        }
        return certSKF;
    }

    /**
     * 将SKF结构的数据写入卡
     *
     * @param type    设备类型
     * @param certStr SKF结构数据
     * @return -4 标识服务端跟客户端的证书机制不一致
     */
    public static int importCert(CryptoDevType type, String certStr) {
        CertSKF certSKF = splitCertStr(type, certStr);
        if (certSKF == null) {
            return -1;
        }
        if (certSKF.getErrorCode() == 4) {
            return -4;
        }

        String pin = AppConfig.getInstance().getPin();
        int role = 0x01;
        int containerNo = 0;

        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        CertBean bean = CertBean.createDoubleSM2CertSkf(role, pin, containerNo, certSKF.getSignCert(), certSKF.getEnCodeCert(), certSKF.getKeyPair());

        int importCertRet = cryptoInstance.importCert(bean);

        if (importCertRet != 0) {
            return importCertRet;
        }

        int writeProjectLabelRet = writeProjectLabel(type, certSKF.getProjectLabel());

        if (writeProjectLabelRet != 0) {
            return writeProjectLabelRet;
        } else {
            return 0;
        }
    }

    /**
     * 根据实际情况获取CryptoInstance实例
     * author jff
     *
     * @return
     */
    protected static CryptoInstance getCryptoInstance(CryptoDevType type) {

        scanningDevice();

        CryptoInstance cryptoDev;

        try {
            cryptoDev = CryptoDevManager.getInstance()
                    .createInstance(ChipManagerDriver.DRIVER_NAME, type);
        } catch (Exception e) {
            return null;
        }

        if (cryptoDev == null) {
            return null;
        }
        int result = cryptoDev.open();
        if (result != 0) {
            return null;
        }
        return cryptoDev;
    }

    private static int writeProjectLabel(CryptoDevType type, String labStr) {
        byte[] labelFid = new byte[]{0x00, (byte) 0x9f};
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        byte[] labStrBytes = labStr.getBytes();
        int writeLabel = -1;

        int createRet = createFile(type);
        if (createRet == 0 || createRet == -23) {//创建文件成功或者文件已存在
            writeLabel = cryptoInstance.writeFile(labelFid, 0, 64, labStrBytes);
        }

        return writeLabel;
    }

    private static int createFile(CryptoDevType type) {
        byte[] labelFid = new byte[]{0x00, (byte) 0x9f};

        XDJA_FILE file = new XDJA_FILE();
        file.read_Acl = (byte) 0xf1;
        file.write_Acl = (byte) 0xf1;
        file.room = 0x200;
        file.type = JNIAPI.FILE_BINARY;
        file.id = labelFid;


        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        int ret = cryptoInstance.createFile(file);
        //        ModuleLog.e("创建ret " + ret + " type " + type);
        return ret;
    }

    /**
     * @param type
     * @return -1 设备获取失败
     * 0 验证成功
     */
    public static int verifyPin(CryptoDevType type, String pinInput) {
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        AppConfig module = AppConfig.getInstance();
        String pin;
        if (pinInput.isEmpty()) {
            pin = module.getPin();
        } else {
            pin = pinInput;
        }

        int result = cryptoInstance.verifyPIN(pin, 0x01);

        return result;
    }

    public static String getSn(CryptoDevType type, String certStr) {
        CertSKF certSKF = splitCertStr(type, certStr);
        if (certSKF == null) {
            return "";
        }
        CertDetailInfo info;
        byte[][] fid = FidUtil.getCertFid(0);
        byte[] signCert;
        byte[] encodeCert;
        encodeCert = certSKF.getEnCodeCert();
        int encodeCertLen = encodeCert != null ? encodeCert.length : 0;
        int[] enCodeCertLen = new int[]{encodeCertLen};
        info = readBytesToCert(fid[1], encodeCert, enCodeCertLen);

        if (info == null) {
            return "";
        }
        return info.sn;
    }

    /**
     * 将证书byte[]解析为可读的字符串
     * author jff
     *
     * @param fid     设备内证书id
     * @param certBuf 待解析的证书byte[]
     * @return 证书实体
     */
    private static CertDetailInfo readBytesToCert(byte[] fid, byte[] certBuf, int[] certLen) {
        if (fid.length == 0 || certBuf.length == 0 || certLen.length == 0) {
            return null;
        }
        try {
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy.MM.dd");
            SimpleDateFormat formatterValid = new SimpleDateFormat("yyyy.MM.dd HH:mm:ss");
            ByteArrayInputStream bin = new ByteArrayInputStream(certBuf);
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            X509Certificate cert = (X509Certificate) cf.generateCertificate(bin);
            CertDetailInfo certDetailInfo = new CertDetailInfo();

            certDetailInfo.issuerOrg = cert.getIssuerX500Principal().getName();
            certDetailInfo.issuerCn = cert.getIssuerDN().getName();
            certDetailInfo.subjectOrg = cert.getSubjectX500Principal().getName();
            certDetailInfo.subjectCn = cert.getSubjectDN().getName();


            certDetailInfo.alg = cert.getSigAlgName();
            certDetailInfo.ver = cert.getVersion();

            System.arraycopy(fid, 0, certDetailInfo.certId, 0, 2);
            certDetailInfo.indate = formatter.format(cert.getNotBefore()) + " - " + formatter.format(cert.getNotAfter());
            certDetailInfo.notBefore = cert.getNotBefore();
            certDetailInfo.notAfter = cert.getNotAfter();

            //sn应该是16进制的
            certDetailInfo.sn = cert.getSerialNumber().toString(16);
            certDetailInfo.certLen = certLen;
            bin.close();

            return certDetailInfo;
        } catch (CertificateException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }

    /**
     * 将数据用指定设备内0号容器内私钥签名
     *
     * @param type 指定私钥对应的设备
     * @param str  待签名内容
     * @return 签名后的内容
     */
    public static String signData(CryptoDevType type, String str) {

        //        ModuleLog.e("待签名信息 str = " + str);
        if (str.equals("")) {
            return null;
        }

        byte[][] keyPair = FidUtil.getKeyPairFid(0);
        byte[] pubFid = keyPair[0];
        byte[] priFid = keyPair[1];

        byte[] signData;
        int[] signDataLen = new int[1];
        byte[] certInfoBytes = str.getBytes();
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return null;
        }

        AppConfig module = AppConfig.getInstance();

        int ret = verifyPin(type, module.getPin());
        if (ret != 0) {
            //            ModuleLog.d(" 验证Pin码result = " + ret + CryptoDevManager.getInstance().getErrorText(ret));
            return "errorCode" + CryptoDevManager.getInstance().getErrorText(ret);
        }

//        只保留SM2
        signData = new byte[64];
        ret = cryptoInstance.SM2Sign(pubFid, priFid, 1, certInfoBytes, certInfoBytes.length, signData, signDataLen);

        byte[] data = new byte[signDataLen[0]];

        //        ModuleLog.e("签名前数据str =" + str);
        //        verifySM2Sign(type, str, signData);
        System.arraycopy(signData, 0, data, 0, data.length);
        //        ModuleLog.e("签名后 数据 = " + new String(Base64.encode(data)));
        if (ret == 0) {
            return new String(Base64.encode(data));
        }
        return null;
    }

    /**
     * 读取设备ID
     * author jff
     *
     * @return 指定类型的设备ID
     */
    public static String getCardNum(CryptoDevType type) {
        String cardId;

        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return null;
        }
        cardId = cryptoInstance.getCardId();
        cryptoInstance.close();

        return cardId;
    }

    /**
     * 清除指定容器内的所有内容
     *
     * @param type
     * @return
     */
    public static int clearContainer(CryptoDevType type) {
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }
        byte[] labelFid = new byte[]{0x00, (byte) 0x9f};

        AppConfig module = AppConfig.getInstance();
        int ret = cryptoInstance.verifyPIN(module.getPin(), 0x01);
        if (ret == 0) {
            ret = cryptoInstance.clearContainer(0x01, module.getPin(), 0);
            //此处屏蔽是因为发现删除文件返回错误码是-15，跟芯片管家确认目前的cos是不支持删除文件的，可以复写文件
            //            if (ret == 0) {
            //                verifyPin(type, "");
            //                ret = cryptoInstance.deleteFile(labelFid);
            //                Log.e("jff", "DeviceWrapper  clearContainer:  deleteFile  ret =  " + ret);
            //            }

        }
        return ret;
    }

    /**
     * 从容其中读取证书
     *
     * @param type        设备类型
     * @param containerNo 容器ID
     * @param certType    证书类型    0  签名证书       1 交换证书
     * @return
     */
    public static CertDetailInfo readCertFromContainer(CryptoDevType type, int containerNo) {

        byte[] fid = null;
        byte[][] certFid = FidUtil.getCertFid(containerNo);

        fid = certFid[0];
        return readCertFromFid(type, fid);

    }

    private static CertDetailInfo readCertFromFid(CryptoDevType type, byte[] fid) {
        byte[] certBuf = new byte[2048];
        int[] certLen = new int[1];
        byte[] bytes = readCertBytes(type, fid, certBuf, certLen);

        if (bytes == null) {
            return null;
        }

        return readBytesToCert(fid, certBuf, certLen);
    }

    /**
     * 读取指定设备0号容器内的签名证书
     *
     * @param type 要读取的设备类型
     * @param fid  读取的位置
     * @return
     */
    public static byte[] readCertBytes(CryptoDevType type, byte[] fid, byte[] certBuf, int[] certLen) {
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return null;
        }
        int result = cryptoInstance.readCert(fid, certBuf, certLen);
        cryptoInstance.close();

        if (result != 0) {
            return null;
        }
        return certBuf;
    }

    /**
     * 用根证书验证写入卡里的证书
     *
     * @return
     */

    public static boolean verifyCertByRoot(Context context, CryptoDevType type, int containerNo) {

        BouncyCastleProvider provider = new BouncyCastleProvider();

        //Android中自带的 BouncyCastleProvider 不包含 SM3WITHSM2签名算法，所以此处删除该Provider，然后添加引用jar里的 BouncyCastleProvider
        Security.removeProvider("BC");
        Security.addProvider(provider);


        //读取申请的证书
        X509Certificate cert = getX509Certificate(readCertFromType(type, containerNo), provider);

        // TODO: 2019/6/28 貴州，先把這個地方 改成只從assets目錄下讀取試一下 ‘

        //读取根证书
        X509Certificate root = getX509Certificate(readRootCertFromAssets("root.cer"), provider);

        if (cert == null || root == null) {
            return false;
        }

        //待验证证书的发布者是否是根证的所有者
        Principal principalIssuer = cert.getIssuerDN();   //获取待验证证书的发布者
        Principal principalSubject = root.getSubjectDN(); //根证书的所有者

        if (!principalIssuer.equals(principalSubject)) {
            return false;
        }
        try {
            cert.verify(root.getPublicKey());
            return true;
        } catch (CertificateException e) {
            e.printStackTrace();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (InvalidKeyException e) {
            e.printStackTrace();
        } catch (NoSuchProviderException e) {
            e.printStackTrace();
        } catch (SignatureException e) {
            e.printStackTrace();
        }
        return false;

    }

    /**
     * 根据芯片类型读取证书
     * 只验证根证书使用
     *
     * @param type
     * @param containerNo
     * @return
     */
    private static InputStream readCertFromType(CryptoDevType type, int containerNo) {

        byte[][] fid = FidUtil.getCertFid(containerNo);


        byte[] certBuf = new byte[2048];
        int[] certLen = new int[1];
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return null;
        }
        int result = -1;

        //因为该方法目前只在验证根证书时使用，并且在现场即存在交换证书也存在签名证书的时候需要必须读出来证书
//        if (AppConfig.getInstance().getCertType() == AppConfig.SIGN_CERT
//                || AppConfig.getInstance().getCertType() == -1) {
        result = cryptoInstance.readCert(fid[0], certBuf, certLen);
//        } else if (AppConfig.getInstance().getCertType() == AppConfig.EXCHANGE_CERT) {
        result = cryptoInstance.readCert(fid[1], certBuf, certLen);
//        }

        cryptoInstance.close();
        if (result == 0) {
            return new ByteArrayInputStream(certBuf);
        } else {
            return null;
        }
    }

    /**
     * 将流转化为 X509Certificate 对象
     *
     * @param is
     * @param provider
     * @return
     */
    public static X509Certificate getX509Certificate(InputStream is, BouncyCastleProvider provider) {
        if (is == null) {
            return null;
        }
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509", provider);
            X509Certificate certificate = (X509Certificate) cf.generateCertificate(is);
            is.close();
            return certificate;
        } catch (CertificateException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }


    /**
     * 读取根证书
     *
     * @param fileName
     * @return
     */
    public static InputStream readRootCertFromAssets(String fileName) {
        Context context = AppConfig.getInstance().getContext();
        AssetManager assetManager = context.getAssets();
        try {

            return assetManager.open(fileName);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }


}
