package com.xdja.updatelibrary;

import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.content.FileProvider;
import android.text.TextUtils;
import android.util.Log;

import com.xdja.update.CheckCallback;
import com.xdja.update.CheckTask;
import com.xdja.update.DownloadCallback;
import com.xdja.update.UpdateManager;
import com.xdja.update.bean.CheckResult;
import com.xdja.update.enums.DownLoadFileError;
import com.xdja.updatelibrary.data.UpdateInfo;
import com.xdja.updatelibrary.utils.UpdateSharePreferenceUtils;

import java.io.File;
import java.io.IOException;

/**
 * Created by hyh on 2018/5/2.
 */

public class XDJAUpdateService implements CheckCallback {
    private static XDJAUpdateService mXdjaUpdate = null;
    private CheckResult mCheckResult;
    private UpdateInterface mUpdateInterface;
    private boolean mIsCheckingOrLoading;

    public interface UpdateInterface {
        void onUpdateCheckResult(String result, UpdateInfo info);

        void onUpdatedownloadStart();

        void onUpdatedownloadProgress(long totalSize, long progress);

        void onUpdatedownloadSuccess();

        void onUpdatedownloadError();

    }

    private XDJAUpdateService() {
    }

    public static XDJAUpdateService getInstance() {
        if (mXdjaUpdate == null) {
            synchronized (XDJAUpdateService.class) {
                if (mXdjaUpdate == null) {
                    mXdjaUpdate = new XDJAUpdateService();
                }
            }
        }
        return mXdjaUpdate;
    }



    /**
     * 检查是否需要升级
     */
    public void onStartCheckUpdate(UpdateInterface updateInterface) {
        mUpdateInterface = updateInterface;
        mIsCheckingOrLoading = true;
        if (XDJAUpdateManager.getConfig().getIsReadAssets()) {
            UpdateManager.getInstance().checkUpdate(this);
        } else {
            if (XDJAUpdateManager.getConfig().getServerPathBean() == null) {
                CheckResult result = new CheckResult();
                result.setResult(CheckResult.RESULT_UPDATE_ERROR);
                onCheckResult(result);
            } else {
                UpdateManager.getInstance().checkUpdate(this, XDJAUpdateManager.getConfig().getServerPathBean().getServerIp(), XDJAUpdateManager.getConfig().getServerPathBean().getServerPort());
            }
        }
    }

    @Override
    public void onChecking() {
        mIsCheckingOrLoading = true;
    }

    @Override
    public void onCheckResult(@NonNull CheckResult checkResult) {
        Log.i(CheckTask.TAG, "result=" + checkResult.toString());
        mIsCheckingOrLoading = false;
        mCheckResult = checkResult;
        UpdateInfo updateInfo = new UpdateInfo();
        if (mCheckResult.getResult().equals(CheckResult.RESULT_UPDATE_POSITIVE)) {
            updateInfo.setVersion(checkResult.getUpdates().get(0).getVersion());
            updateInfo.setInfo(checkResult.getUpdates().get(0).getComment());
            if (checkResult.getUpdates().get(0).getUpdateTag() == 0) {
                updateInfo.setForceUpdate(false);//提醒升级
            } else {
                updateInfo.setForceUpdate(true);// 强制升级
            }
            updateInfo.setUpdate(true);
            UpdateSharePreferenceUtils.onChangeUpdateState(UpdateManager.getInstance().getContext(), updateInfo);
        } else if (mCheckResult.getResult().equals(CheckResult.RESULT_UPDATE_NEGATIVE)) {//防止当前是最新版本，sp没有修改过来
            UpdateSharePreferenceUtils.onChangeVersion(UpdateManager.getInstance().getContext());
        }
        if (mUpdateInterface != null) {
            mUpdateInterface.onUpdateCheckResult(checkResult.getResult(), updateInfo);
            onReleaseResource();
        }
    }


    /**
     * 下载Apk文件
     */
    public boolean onDownLoadApkFile(UpdateInterface updateInterface) {
        mUpdateInterface = updateInterface;
        if (mCheckResult == null || !mCheckResult.getResult().equals(CheckResult.RESULT_UPDATE_POSITIVE) || mCheckResult.getUpdates() == null || mCheckResult.getUpdates().size() == 0 || mCheckResult.getUpdates().get(0).getFiles() == null || mCheckResult.getUpdates().get(0).getFiles().size() == 0) {
            return true;
        }
        UpdateManager.getInstance().downloadFile(mCheckResult.getUpdates().get(0).getFiles().get(0), new DownloadCallback() {
            @Override
            public void onStart(@NonNull String fileId) {
                mIsCheckingOrLoading = true;
                if (mUpdateInterface != null) {
                    mUpdateInterface.onUpdatedownloadStart();
                }
                onDeleteUnFinishedApkFile(getDownLoadApkPath(downloadDir(), fileId), mCheckResult.getUpdates().get(0).getFiles().get(0));
            }

            @Override
            public void onProgress(@NonNull String fileId, long totalSize, long cSize) {
                if (mUpdateInterface != null) {
                    mUpdateInterface.onUpdatedownloadProgress(totalSize, cSize);
                }
            }

            @Override
            public void onStop(@NonNull String fileId) {
                mIsCheckingOrLoading = false;
                onDeleteApkFile(getDownLoadApkPath(downloadDir(), fileId));
                mCheckResult = null;
                onReleaseResource();
            }

            @Override
            public void onError(
                    @NonNull String fileId,
                    @Nullable DownLoadFileError errorMsg, @Nullable Exception ex) {
                mIsCheckingOrLoading = false;
                onDeleteApkFile(getDownLoadApkPath(downloadDir(), fileId));
                if (mUpdateInterface != null) {
                    mUpdateInterface.onUpdatedownloadError();
                }
                mCheckResult = null;
                onReleaseResource();
            }

            @Override
            public void onComplete(@NonNull String fileId, @NonNull String filePath) {
                mIsCheckingOrLoading = false;
                if (mUpdateInterface != null) {
                    mUpdateInterface.onUpdatedownloadSuccess();
                }
                onInstallApk(filePath, UpdateManager.getInstance().getContext());
                onReleaseResource();
            }

            @Override
            public int unit() {
                return 1024 * 1024;
            }

            @NonNull
            @Override
            public String downloadDir() {
//                                            return MainActivity.this.getFilesDir().getPath();
                return UpdateManager.getInstance().getContext().getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS).getPath();
            }
        });
        return false;
    }

    /**
     * 停止下载
     */
    public void onStopLoad() {
        mIsCheckingOrLoading = false;
        UpdateManager.getInstance().stopDownload();
    }

    /**
     * 得到更新状态:是否正在检查或者下载
     *
     * @return
     */
    public boolean isCheckingOrLoading() {
        return mIsCheckingOrLoading;
    }

    /**
     * 安装apk
     *
     * @param filePath
     */
    private void onInstallApk(String filePath, Context context) {
        File file = new File(filePath);
        if (file.exists()) {
            Intent intent = new Intent(Intent.ACTION_VIEW);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
//                Uri uri = FileProvider.getUriForFile(context, context.getPackageName() + ".fileprovider", file);
//                String auth = "com.xdja.aircert.fileprovider";
                String auth = !BuildConfig.isAAR ? "com.xdja.aircert.standalone.fileprovider" : "com.xdja.aircert.fileprovider";
                Uri uri = FileProvider.getUriForFile(context, auth, file);
                intent.setDataAndType(uri, "application/vnd.android.package-archive");
            } else {
                intent.setDataAndType(Uri.fromFile(file), "application/vnd.android.package-archive");
            }
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            context.startActivity(intent);
        }
    }

    /**
     * 当进程关闭，所未下载完成的apk删除
     */
    private void onDeleteUnFinishedApkFile(String path, CheckResult.Update.File file) {
        if (!XDJAUpdateManager.getConfig().isNeedBreakPointResume()) {
            try {
                File targetFile = new File(path);
                long offset = getOffset(targetFile);
                if (offset >= file.getfSize() - 1) {
                } else {
                    onDeleteApkFile(path);
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    private long getOffset(File of) throws IOException {
        long offset = 0;
        if (!of.exists()) {
            if (!of.createNewFile()) {
            }
        } else {
            offset = of.length() - 1;
        }
        return offset < 0 ? 0 : offset;
    }

    /**
     * 下载失败或者取消下载
     */
    private void onDeleteApkFile(String fileName) {
        if (!XDJAUpdateManager.getConfig().isNeedBreakPointResume()) {
            if (!TextUtils.isEmpty(fileName)) {
                File file = new File(fileName);
                if (file.exists()) {
                    file.delete();
                }
            }
        }
    }


    /**
     * 下载apk文件路径
     */
    private String getDownLoadApkPath(String path, String fileId) {
        StringBuilder builder = new StringBuilder();
        builder.append(path).append("/").append(fileId).append(".apk");
        return builder.toString();
    }

    private void onReleaseResource() {
        if (mUpdateInterface != null) {
            mUpdateInterface = null;
        }
    }
}
