package com.xdja.safeclient.certcreation.device;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.res.AssetManager;
import android.os.Build;
import android.os.Environment;
import android.provider.Settings;
import android.support.v4.app.ActivityCompat;
import android.support.v4.content.ContextCompat;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.aircert.util.ModuleLog;
import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.XDJA_FILE;
import com.xdja.cryptodev.CryptoDevInfo;
import com.xdja.cryptodev.CryptoDevManager;
import com.xdja.cryptodev.CryptoDevType;
import com.xdja.cryptodev.devapi.CryptoInstance;
import com.xdja.cryptodev.driver.chipmanager.ChipManagerDriver;
import com.xdja.multichip.jniapi.Arithmetic;
import com.xdja.multichip.param.CertBean;
import com.xdja.safeclient.certcreation.AppConfig;
import com.xdja.safeclient.certcreation.ApplicationInit;
import com.xdja.safeclient.certcreation.bean.CertDetailInfo;
import com.xdja.safeclient.certcreation.bean.CertSKF;
import com.xdja.safeclient.certcreation.bean.device.CardType;
import com.xdja.safeclient.certcreation.config.CertRule;
import com.xdja.safeclient.certcreation.config.CertType;
import com.xdja.safeclient.certcreation.config.XDJAAlgParams;
import com.xdja.safeclient.certcreation.databases.CertConfigDataBase;
import com.xdja.safeclient.certcreation.util.FidUtil;
import com.xdja.safeclient.certcreation.util.OttoUtil;
import com.xdja.safeclient.certcreation.util.SharePreferencesUtil;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Principal;
import java.security.Security;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by wanghao on 2017/9/6.
 */

public class DeviceWrapper {

    private static DeviceWrapper DEVICE_WRAPPER = new DeviceWrapper();

    public static DeviceWrapper getInstance() {
        if (DEVICE_WRAPPER == null) {
            DEVICE_WRAPPER = new DeviceWrapper();
        }
        return DEVICE_WRAPPER;
    }


    private DeviceWrapper() {
    }

    private String certNo = "";

    private InputStream rootCer = null;


    /**
     * 申请证书流程结束的标志
     */
    private boolean applyCertEnd = false;


    public void setRootCer(InputStream rootCer) {
        this.rootCer = rootCer;
    }

    public void setInUse(String cardNo) {
        this.certNo = cardNo;
    }

    public String getInUse() {
        return certNo;
    }


    public boolean isApplyCertEnd() {
        return applyCertEnd;
    }

    public void setApplyCertEnd(boolean applyCertEnd) {
        this.applyCertEnd = applyCertEnd;
    }


    /**
     * 读取设备ID
     * author jff
     *
     * @return 指定类型的设备ID
     */
    public String getCardID(CryptoDevType type) {
        String cardId;
        if (type == CryptoDevType.DEV_TYPE_Soft
                && AppConfig.getInstance().getCertConfig().getVHSMType().equals(CardType.ONLINE_VHSM)) {
            cardId = NetVHSMHandle.getNetVhsmCardId(AppConfig.getInstance().getContext());
        } else {
            CryptoInstance cryptoInstance = getCryptoInstance(type);
            if (cryptoInstance == null) {
                Log.e("jff", "128 DeviceWrapper getCardID : cryptoInstance == null");
                return null;
            }
            cardId = cryptoInstance.getCardId();
            cryptoInstance.close();
        }

        return cardId;
    }


    /**
     * 读取设备IMEI
     * author jff
     *
     * @return
     */
    public String getIMEI() {
        Context context = AppConfig.getInstance().getContext();
        String imei = "";

        if (ContextCompat.checkSelfPermission(context,
                Manifest.permission.READ_PHONE_STATE) == PackageManager.PERMISSION_GRANTED) {

            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                imei = telephonyManager.getImei();
            } else {
                imei = telephonyManager.getDeviceId();
            }
        }
        if (imei == null)
            return "";
        else
            return imei;

////                // TODO: 2019/11/13 jff 山东政法委初始版本使用androidid作为唯一标识用来生成卡号
//                imei = Settings.System.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
//                return imei;
    }


    /**
     * 读取SIM卡的imsi
     * author jff
     *
     * @return
     */
    public String getIMSI() {
        Context context = AppConfig.getInstance().getContext();
        TelephonyManager mTelephonyMgr = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
        if (ActivityCompat.checkSelfPermission(context, Manifest.permission.READ_PHONE_STATE) != PackageManager.PERMISSION_GRANTED) {
            return "";
        }
        String imsi = mTelephonyMgr.getSubscriberId();
        if (imsi == null)
            return "";
        else
            return imsi;
    }


    private static void scanningDevice() {
        boolean isScanningDevice = CryptoDevManager.getInstance().isScanning();
        while (isScanningDevice) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            isScanningDevice = CryptoDevManager.getInstance().isScanning();
        }
    }


    /**
     * 根据实际情况获取CryptoInstance实例
     * author jff
     *
     * @return
     */
    protected static CryptoInstance getCryptoInstance(CryptoDevType type) {

        scanningDevice();

        CryptoInstance cryptoDev;

        try {
            cryptoDev = CryptoDevManager.getInstance().createInstance(ChipManagerDriver.DRIVER_NAME, type);
        } catch (Exception e) {
            return null;
        }

        if (cryptoDev == null) {
            return null;
        }
        int result = cryptoDev.open();
        if (result != 0) {
            return null;
        }
        return cryptoDev;
    }

    /**
     * 获取加密设备的列表
     * author jff
     *
     * @return
     */
    private List<CryptoDevInfo> getDeviceList() {
        List<CryptoDevInfo> devList = CryptoDevManager.getInstance().getCryptoDevInfos();
        if (devList.size() <= 0) {
            return null;
        }
        return devList;
    }


    public Map<Integer, CryptoDevInfo> getDevInfo() {

        scanningDevice();

        Map<Integer, CryptoDevInfo> infoMap = new HashMap<>();
        CryptoDevInfo sdDev;
        CryptoDevInfo chipDev;
        CryptoDevInfo simKeyDev;
        CryptoDevInfo softDev;

        List<CryptoDevInfo> devList = getDeviceList();
        if (devList == null || devList.size() <= 0) {
            return null;
        }
        //        ModuleLog.e("获取设备列表devList.size() " + devList.size());
        for (CryptoDevInfo devInfo : devList) {
            switch (devInfo.getType()) {
                case DEV_TYPE_SdCard:
                    sdDev = devInfo;
                    infoMap.put(CryptoDevType.DEV_TYPE_SdCard.getType(), sdDev);
                    break;
                case DEV_TYPE_Chip:
                    chipDev = devInfo;
                    infoMap.put(CryptoDevType.DEV_TYPE_Chip.getType(), chipDev);
                    break;
                case DEV_TYPE_SimKey:
                    simKeyDev = devInfo;
                    infoMap.put(CryptoDevType.DEV_TYPE_SimKey.getType(), simKeyDev);
                    break;
                case DEV_TYPE_Soft:
                    softDev = devInfo;
                    infoMap.put(CryptoDevType.DEV_TYPE_Soft.getType(), softDev);
                    break;

            }
        }
        return infoMap;
    }


    //    /**
    //     * 读取4号容器签名证书，走https请求时需要
    //     *
    //     * @return
    //     */
    //    public InputStream readCertInput(CryptoDevType type) {
    //        byte fid[] = {0x00, 0x48};//4号容器签名证书
    //        byte[] certBuf = new byte[2048];
    //        int[] certLen = new int[1];
    //        CryptoInstance cryptoInstance = DeviceWrapper.getInstance().getCryptoInstance(type);
    //        if (cryptoInstance == null) {
    //            return null;
    //        }
    //        int result = cryptoInstance.readCert(fid, certBuf, certLen);
    //        if (result == 0) {
    //            return new ByteArrayInputStream(certBuf);
    //        }
    //        return null;
    //    }

    /**
     * @param type
     * @return -1 设备获取失败
     * 0 验证成功
     */
    public static int verifyPin(CryptoDevType type, int containerNum, String pinInput) {
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        AppConfig module = AppConfig.getInstance();
        String pin;
        if (pinInput.isEmpty()) {
            pin = module.getDefaultConfig().getPin();
        } else {
            pin = pinInput;
        }

        com.aircert.util.ModuleLog.time("库验证PIN开始");

        int result = cryptoInstance.verifyPIN(pin, module.getRole(type, containerNum));

        com.aircert.util.ModuleLog.time("库验证PIN结束");

        return result;
    }

    /**
     * 卡库错误码对应的描述
     *
     * @param errorCode
     * @return 错误码的描述
     */
    private String errorMsg(int errorCode) {
        return CryptoDevManager.getInstance().getErrorText(errorCode);
    }


    /**
     * 清除指定容器内的所有内容，通过调用芯片管家aar实现删除容器该功能
     *
     * @param type
     * @return
     */
    public int clearContainer(CryptoDevType type, int containerNum) {
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }
        byte[] labelFid = new byte[]{0x00, (byte) 0x9f};

        AppConfig module = AppConfig.getInstance();
        int ret = cryptoInstance.verifyPIN(module.getDefaultConfig().getPin(), module.getRole(type, containerNum));
        if (ret == 0) {
            ret = cryptoInstance.clearContainer(module.getRole(type, containerNum), module.getDefaultConfig().getPin(), containerNum);
            //此处屏蔽是因为发现删除文件返回错误码是-15，跟芯片管家确认目前的cos是不支持删除文件的，可以复写文件
            //            if (ret == 0) {
            //                verifyPin(type, "");
            //                ret = cryptoInstance.deleteFile(labelFid);
            //                Log.e("jff", "DeviceWrapper  clearContainer:  deleteFile  ret =  " + ret);
            //            }

        }
        return ret;
    }


    /**
     * 通过调用crytoDev的写文件接口将相关内容覆写本地实现删除容器的功能 2019年10月15日17:15:00
     *
     * @param type
     * @param containerId
     * @return
     */
    public int clearContainerLocal(CryptoDevType type, int containerId) {

        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        int result = -1;
        try {
            int ret;
            byte[] exchangeCertFid = Arithmetic.getExchangeCertFid(containerId);
            byte[] signCertFid = Arithmetic.getSignCertFid(containerId);
            byte[] exchangePubFid = Arithmetic.getExchangePubKeyFid(containerId);
            byte[] exchangePriFid = Arithmetic.getExchangePriKeyFid(containerId);
            byte[] signPubFid = Arithmetic.getSignPubKeyFid(containerId);
            byte[] signPriFid = Arithmetic.getSignPriKeyFid(containerId);
            ArrayList<Pair<byte[], Integer>> list = new ArrayList<>();
            list.add(Pair.create(exchangeCertFid, 1));
            list.add(Pair.create(exchangePubFid, 2));
            list.add(Pair.create(exchangePriFid, 2));
            list.add(Pair.create(signCertFid, 1));
            list.add(Pair.create(signPubFid, 2));
            list.add(Pair.create(signPriFid, 2));

            byte[] certBytes = new byte[200];
            Arrays.fill(certBytes, (byte) 0xff);

            ret = verifyPin(type, containerId, "");
            if (ret != 0) {
                return result;
            }
            for (Pair<byte[], Integer> pair : list) {
                ret = cryptoInstance.writeFile(pair.first, 0, certBytes.length, certBytes);
                if (ret != 0) {
                    return result;
                }
            }

            byte[] containerInfo = new byte[5];
            Arrays.fill(containerInfo, (byte) 0x00);
            byte[] infoFid = {0x00, 0x03};
            int writePos = containerId * 5;
            result = cryptoInstance.writeFile(infoFid, writePos, 5, containerInfo);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return result;


    }

    /**
     * 证书信息更新——只写证书信息，不写公私钥
     * 该写证书方法只写证书不包含秘钥结构体，包含项目标签
     * 单证  签名证书#项目标签
     * 双证  签名证书#加密证书#项目标签
     *
     * @param type
     * @param certStr
     * @return
     */
    public int writeCert(CryptoDevType type, int containerNum, String certStr) {
        CertSKF certSKF = splitCertStr(type, containerNum, certStr);
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }
        if (certSKF.getErrorCode() == 4) {
            return -4;
        }
        byte[][] certFid = FidUtil.getCertFid(containerNum);

        String certRule = AppConfig.getInstance().getCertConfig().getCertRule(type, containerNum);

        if (AppConfig.getInstance().getCertConfig().getCertRule(type, containerNum).equals(CertRule.SINGLE)) {

            if (certRule.equals(CertType.SIGNING)
                    || TextUtils.isEmpty(AppConfig.getInstance().getCertConfig().getCertType(type, containerNum))) {

                int signCertRet = cryptoInstance.writeCert(certFid[0], certSKF.getSignCert(), certSKF.getSignCert().length);
                ModuleLog.e("写签名证书 = " + signCertRet);
                if (signCertRet != 0) {
                    return -1;
                }
            }

            if (AppConfig.getInstance().getCertConfig().getCertType(type, containerNum).equals(CertType.ENCRYPT)) {
                int exchangeCertRet = cryptoInstance.writeCert(certFid[0], certSKF.getEnCodeCert(), certSKF.getEnCodeCert().length);
                ModuleLog.e("写交换证书 = " + exchangeCertRet);
                if (exchangeCertRet != 0) {
                    return -1;
                }
            }
        }

        if (certSKF.getLen() == 3) {//双证更新
            int exchangeCertRet = cryptoInstance.writeCert(certFid[1], certSKF.getEnCodeCert(), certSKF.getEnCodeCert().length);
            ModuleLog.e("写交换证书 = " + exchangeCertRet);
            if (exchangeCertRet != 0) {
                return -1;
            }
        }

        int writeProjectLabelRet = writeProjectLabel(type, certSKF.getProjectLabel());
        ModuleLog.e("写项目标签 = " + writeProjectLabelRet);
        if (writeProjectLabelRet != 0) {
            return -1;
        }

        return 0;
    }


    /**
     * 将SKF结构的数据写入卡
     *
     * @param type    设备类型
     * @param certStr SKF结构数据
     * @return -4 标识服务端跟客户端的证书机制不一致
     */
    public int importCert(CryptoDevType type, int containerNum, String certStr) {

        String certType = AppConfig.getInstance().getCertConfig()
                .getCertType(type, containerNum);

        String certRule = AppConfig.getInstance().getCertConfig()
                .getCertRule(type, containerNum);

        String debugMsg = "当前设备:" + CardType.toString(type)
                + "\n" + "容器号：" + containerNum
                + "\n" + "证书机制" + certRule
                + "\n" + "证书类型:" + certType;

        ModuleLog.e(debugMsg);


        OttoUtil.showDebug(debugMsg);

        CertSKF certSKF = splitCertStr(type, containerNum, certStr);
        if (certSKF == null) {
            setApplyCertEnd(false);
            return -1;
        }
        if (certSKF.getErrorCode() == 4) {
            setApplyCertEnd(false);
            return -4;
        }

        ModuleLog.e("certSKF= " + certSKF.getLen());
        AppConfig module = AppConfig.getInstance();
        String pin = module.getDefaultConfig().getPin();
        int role = module.getRole(type, containerNum);
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            setApplyCertEnd(false);
            return -1;
        }

        CertBean bean = null;
        if (module.getCertConfig().getAlgType(type, containerNum).equals(XDJAAlgParams.CA_ALG_RSA)) {
            switch (AppConfig.getInstance().getCertConfig().getCertRule(type, containerNum)) {
                case CertRule.SINGLE:
                    if (AppConfig.getInstance().getCertConfig().getCertType(type, containerNum).equals(CertType.SIGNING)
                            || TextUtils.isEmpty(AppConfig.getInstance().getCertConfig().getCertType(type, containerNum))) {
                        bean = CertBean.createSingleRSASignCert(role, pin, containerNum, certSKF.getSignCert());
                    } else {
                        bean = CertBean.createSingleRSAExchangeCert(role, pin, containerNum, certSKF.getEnCodeCert());
                    }
                    break;
            }
        } else {
            switch (AppConfig.getInstance().getCertConfig().getCertRule(type, containerNum)) {
                case CertRule.SINGLE:
                    ModuleLog.e("单证写卡");
                    if (AppConfig.getInstance().getCertConfig().getCertType(type, containerNum).equals(CertType.SIGNING)
                            || TextUtils.isEmpty(AppConfig.getInstance().getCertConfig().getCertType(type, containerNum))) {

                        bean = CertBean.createSingleSM2SignCert(role, pin, containerNum, certSKF.getSignCert());
                    } else {
                        bean = CertBean.createSingleSM2ExchangeCert(role, pin, containerNum, certSKF.getEnCodeCert());
                    }
                    break;
                case CertRule.DOUBLE_CERT:
                    bean = CertBean.createDoubleSM2CertSkf(role, pin, containerNum, certSKF.getSignCert(), certSKF.getEnCodeCert(), certSKF.getKeyPair());
                    break;
            }
        }

        ModuleLog.e("import cert = " + bean.toString());
        int importCertRet = cryptoInstance.importCert(bean);
        ModuleLog.e("import cert = " + importCertRet + "\n" + DeviceWrapper.getInstance().errorMsg(importCertRet));

        if (importCertRet != 0) {
            setApplyCertEnd(false);
            return importCertRet;
        }

        int writeProjectLabelRet = writeProjectLabel(type, certSKF.getProjectLabel());
        ModuleLog.e("writeProjectLabelRet = " + writeProjectLabelRet);
        if (writeProjectLabelRet != 0) {
            setApplyCertEnd(false);
            return writeProjectLabelRet;
        } else {
            setApplyCertEnd(false);
            return 0;
        }
    }

    public String getSn(CryptoDevType type, int containerNum, String certStr) {
        CertSKF certSKF = splitCertStr(type, containerNum, certStr);
        if (certSKF == null) {
            return "";
        }
        CertDetailInfo info;
        byte[][] fid = FidUtil.getCertFid(containerNum);
        byte[] signCert;
        byte[] encodeCert;
        if (AppConfig.getInstance().getCertConfig().getCertType(type, containerNum).equals(CertType.SIGNING)
                || TextUtils.isEmpty(AppConfig.getInstance().getCertConfig().getCertType(type, containerNum))) {
            signCert = certSKF.getSignCert();
            int signCertLen = signCert != null ? signCert.length : 0;
            int[] signCodeCertLen = new int[]{signCertLen};
            info = readBytesToCert(fid[0], signCert, signCodeCertLen);
        } else {
            encodeCert = certSKF.getEnCodeCert();
            int encodeCertLen = encodeCert != null ? encodeCert.length : 0;
            int[] enCodeCertLen = new int[]{encodeCertLen};
            info = readBytesToCert(fid[1], encodeCert, enCodeCertLen);
        }

        if (info == null) {
            return "";
        }
        return info.sn;
    }


    private CertSKF splitCertStr(CryptoDevType type, int containerNum, String certStr) {
        if (certStr.equals("") || certStr.length() <= 0) {
            return null;
        }
        //        ModuleLog.e("certStr " + certStr);

        //“cert”: “签名证书Base64#加密证书Base64#加密密钥” //证书信息
        certStr = certStr.replaceAll("\r", "");
        certStr = certStr.replaceAll("\n", "");
        certStr = certStr.replaceAll("-----BEGIN CERTIFICATE-----", "");
        certStr = certStr.replaceAll("-----END CERTIFICATE-----", "");
        String[] pieces = certStr.split("#");
        //        ModuleLog.e("cert分隔后数据长度 " + pieces.length);

        CertSKF certSKF = new CertSKF();

        String certRule = AppConfig.getInstance().getCertConfig().getCertRule(type, containerNum);

        //如果配置项证书个数跟服务端返回的数据长度不一致
        //todo jff 之后做证书更新这一块应该做相应的变更
        if ((certRule.equals(CertRule.SINGLE) && pieces.length != 2)
                || (certRule.equals(CertRule.DOUBLE_CERT) && pieces.length != 4)) {
            certSKF.setErrorCode(4);
        }

        switch (pieces.length) {
            case 2:  //加密证书base64#项目标签（单证/单证更新延期）
                if (!TextUtils.isEmpty(pieces[0])) {
                    if (AppConfig.getInstance().getCertConfig().getCertType(type, containerNum).equals(CertType.SIGNING)
                            || TextUtils.isEmpty(AppConfig.getInstance().getCertConfig().getCertType(type, containerNum))) {

                        certSKF.setSignCert(Base64.decode(pieces[0].getBytes()));
                    } else if (AppConfig.getInstance().getCertConfig().getCertType(type, containerNum).equals(CertType.ENCRYPT)) {
                        certSKF.setEnCodeCert(Base64.decode(pieces[0].getBytes()));
                    }
                }
                if (!TextUtils.isEmpty(pieces[1])) {
                    certSKF.setProjectLabel(pieces[1]);
                }
                certSKF.setLen(2);
                break;
            case 3://签名证书base64#加密证书base64#项目标签（双证无私钥）
                if (!TextUtils.isEmpty(pieces[0])) {
                    certSKF.setSignCert(Base64.decode(pieces[0].getBytes()));
                }
                if (!TextUtils.isEmpty(pieces[1])) {
                    certSKF.setEnCodeCert(Base64.decode(pieces[1].getBytes()));
                }

                if (!TextUtils.isEmpty(pieces[2])) {
                    certSKF.setProjectLabel(pieces[2]);
                }
                certSKF.setLen(3);
            case 4://签名证书base64#加密证书base64#私钥结构体#项目标签
                if (!TextUtils.isEmpty(pieces[0])) {
                    certSKF.setSignCert(Base64.decode(pieces[0].getBytes()));
                }
                if (!TextUtils.isEmpty(pieces[1])) {
                    certSKF.setEnCodeCert(Base64.decode(pieces[1].getBytes()));
                }

                if (!TextUtils.isEmpty(pieces[2])) {
                    certSKF.setKeyPair(Base64.decode(pieces[2].getBytes()));
                }
                if (!TextUtils.isEmpty(pieces[3])) {
                    certSKF.setProjectLabel(pieces[3]);
                }
                certSKF.setLen(4);
                break;
        }
        return certSKF;
    }


    private int writeProjectLabel(CryptoDevType type, String labStr) {
        byte[] labelFid = new byte[]{0x00, (byte) 0x9f};
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        byte[] labStrBytes = labStr.getBytes();
        int writeLabel = -1;

        int createRet = createFile(type);
        if (createRet == 0 || createRet == -23) {//创建文件成功或者文件已存在
            writeLabel = cryptoInstance.writeFile(labelFid, 0, 64, labStrBytes);
            ModuleLog.e("写标签结果：= " + writeLabel);
            //            ModuleLog.e("写标签内容：= " + labSm3Str.getBytes());
        }

        return writeLabel;
        //        byte[] readFile = new byte[64];
        //        cryptoInstance.readFile(labelFid, 0, 64,  readFile);
        //        ModuleLog.e("读标签结果：= " + new String(readFile));
        //        ModuleLog.e("读标签结果：= " + Util.bytesToHexString(readFile));
    }


    private int createFile(CryptoDevType type) {
        byte[] labelFid = new byte[]{0x00, (byte) 0x9f};

        XDJA_FILE file = new XDJA_FILE();
        file.read_Acl = (byte) 0xf1;
        file.write_Acl = (byte) 0xf1;
        file.room = 0x200;
        file.type = JNIAPI.FILE_BINARY;
        file.id = labelFid;


        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        int ret = cryptoInstance.createFile(file);
        //        ModuleLog.e("创建ret " + ret + " type " + type);
        return ret;
    }


    /**
     * 从容其中读取证书
     *
     * @param type        设备类型
     * @param containerNo 容器ID
     * @param certType    证书类型    0  签名证书       1 交换证书
     * @return
     */
    public CertDetailInfo readCertFromContainer(CryptoDevType type, int containerNo, String certType) {

        byte[] fid = null;
        byte[][] certFid = FidUtil.getCertFid(containerNo);

        if (certType.equals(CertType.SIGNING)) {
            fid = certFid[0];
            return readCertFromFid(type, fid);
        } else if (certType.equals(CertType.ENCRYPT)) {
            fid = certFid[1];
            return readCertFromFid(type, fid);
        } else {
            CertDetailInfo info;
            info = readCertFromFid(type, certFid[0]);
            if (info == null) {
                info = readCertFromFid(type, certFid[1]);
            }
            return info;
        }

    }


    private CertDetailInfo readCertFromFid(CryptoDevType type, byte[] fid) {
        byte[] certBuf = new byte[2048];
        int[] certLen = new int[1];
        byte[] bytes = readCertBytes(type, fid, certBuf, certLen);
        if (bytes == null) {
            return null;
        }

        return readBytesToCert(fid, certBuf, certLen);
    }

    /**
     * 将X509的字符串导出文件到根目录
     *
     * @param x509Cert
     * @return
     */
    public static boolean convertCertFile(X509Certificate x509Cert) {
        FileOutputStream fos = null;
        try {
            if (x509Cert == null) {
                return false;
            }
            File dest = new File(Environment.getExternalStorageDirectory() + "/" + CertConfigDataBase.NAME + ".cer");
            fos = new FileOutputStream(dest);
            fos.write(x509Cert.getEncoded());
            fos.flush();
            fos.close();
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }


    /**
     * 读取指定设备0号容器内的签名证书
     *
     * @param type 要读取的设备类型
     * @param fid  读取的位置
     * @return
     */
    public byte[] readCertBytes(CryptoDevType type, byte[] fid, byte[] certBuf, int[] certLen) {
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return null;
        }
        int result = cryptoInstance.readCert(fid, certBuf, certLen);
        cryptoInstance.close();

        if (result != 0) {
            return null;
        }
        return certBuf;
    }


    /**
     * 将证书byte[]解析为可读的字符串
     * author jff
     *
     * @param fid     设备内证书id
     * @param certBuf 待解析的证书byte[]
     * @return 证书实体
     */
    private CertDetailInfo readBytesToCert(byte[] fid, byte[] certBuf, int[] certLen) {
        if (fid.length == 0 || certBuf.length == 0 || certLen.length == 0) {
            return null;
        }
        try {
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy.MM.dd");
            SimpleDateFormat formatterValid = new SimpleDateFormat("yyyy.MM.dd HH:mm:ss");
            ByteArrayInputStream bin = new ByteArrayInputStream(certBuf);
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            X509Certificate cert = (X509Certificate) cf.generateCertificate(bin);
            CertDetailInfo certDetailInfo = new CertDetailInfo();

            certDetailInfo.issuerOrg = cert.getIssuerX500Principal().getName();
            certDetailInfo.issuerCn = cert.getIssuerDN().getName();
            certDetailInfo.subjectOrg = cert.getSubjectX500Principal().getName();
            certDetailInfo.subjectCn = cert.getSubjectDN().getName();


            certDetailInfo.alg = cert.getSigAlgName();
            certDetailInfo.ver = cert.getVersion();

            System.arraycopy(fid, 0, certDetailInfo.certId, 0, 2);
            certDetailInfo.indate = formatter.format(cert.getNotBefore()) + " - " + formatter.format(cert.getNotAfter());
            certDetailInfo.notBefore = cert.getNotBefore();
            certDetailInfo.notAfter = cert.getNotAfter();

            //sn应该是16进制的
            certDetailInfo.sn = cert.getSerialNumber().toString(16);
            certDetailInfo.certLen = certLen;
            bin.close();

            return certDetailInfo;
        } catch (CertificateException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }

    /**
     * 将数据用指定设备内0号容器内私钥签名
     *
     * @param type 指定私钥对应的设备
     * @param str  待签名内容
     * @return 签名后的内容
     */
    public String signData(CryptoDevType type, int containerNum, String str) {

        //        ModuleLog.e("待签名信息 str = " + str);
        if (str.equals("")) {
            return null;
        }

        byte[][] keyPair = FidUtil.getKeyPairFid(type, containerNum);
        byte[] pubFid = keyPair[0];
        byte[] priFid = keyPair[1];

        byte[] signData;
        int[] signDataLen = new int[1];
        byte[] certInfoBytes = str.getBytes();
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            ModuleLog.e("待签名信息 cryptoInstance = null");
            return null;
        }

        AppConfig module = AppConfig.getInstance();

        int ret = verifyPin(type, containerNum, "");
        if (ret != 0) {
            //            ModuleLog.d(" 验证Pin码result = " + ret + CryptoDevManager.getInstance().getErrorText(ret));
            return "errorCode" + CryptoDevManager.getInstance().getErrorText(ret);
        }

        if (module.getCertConfig().getAlgType(type, containerNum).equals(XDJAAlgParams.CA_ALG_RSA)) {
            signData = new byte[256];
            ret = cryptoInstance.RSASign(XDJAAlgParams.RSA_KEY_LEN_1024, priFid, 1, certInfoBytes, certInfoBytes.length, signData, signDataLen);
        } else {
            signData = new byte[64];
            ret = cryptoInstance.SM2Sign(pubFid, priFid, 1, certInfoBytes, certInfoBytes.length, signData, signDataLen);
        }
        byte[] data = new byte[signDataLen[0]];

        /*
        测试代码导出证书
        SM2PubKey2PubKey(type, containerNum);

        ModuleLog.e("签名前数据str =" + str);
        Log.e("jff", "863 DeviceWrapper signData : " + signData.toString());
        AlgSignUtils.getAlgSignUtils().verifySM2Sign(type, str, signData);
        */
        System.arraycopy(signData, 0, data, 0, data.length);
        //        ModuleLog.e("签名后 数据 = " + new String(Base64.encode(data)));
        if (ret == 0) {
            return new String(Base64.encode(data));
        }
        return null;
    }


    /**
     * 用根证书验证写入卡里的证书
     *
     * @return
     */

    public boolean verifyCertByRoot(Context context, CryptoDevType type, int containerNo) {

        BouncyCastleProvider provider = new BouncyCastleProvider();

        //Android中自带的 BouncyCastleProvider 不包含 SM3WITHSM2签名算法，所以此处删除该Provider，然后添加引用jar里的 BouncyCastleProvider
        Security.removeProvider("BC");
        Security.addProvider(provider);


        //读取申请的证书
        X509Certificate cert = getX509Certificate(readCertFromType(type, containerNo), provider);

        // TODO: 2019/6/28 貴州，先把這個地方 改成只從assets目錄下讀取試一下 ‘

        //读取根证书
        X509Certificate root = getX509Certificate(readRootCertFromAssets("root.cer"), provider);

        if (cert == null || root == null) {
            return false;
        }

        //待验证证书的发布者是否是根证的所有者
        Principal principalIssuer = cert.getIssuerDN();   //获取待验证证书的发布者
        Principal principalSubject = root.getSubjectDN(); //根证书的所有者

        if (!principalIssuer.equals(principalSubject)) {
            return false;
        }
        try {
            cert.verify(root.getPublicKey());
            return true;
        } catch (CertificateException e) {
            e.printStackTrace();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (InvalidKeyException e) {
            e.printStackTrace();
        } catch (NoSuchProviderException e) {
            e.printStackTrace();
        } catch (SignatureException e) {
            e.printStackTrace();
        }
        return false;

    }

    private InputStream getRootCer(Context context) {

        InputStream is = null;
        if (rootCer != null) {
            is = rootCer;
        } else {
            is = ApplicationInit.string2IS(SharePreferencesUtil.getCER(context));
        }

        return is;
    }

    /**
     * 测试代码在ReasonActivity撤销证书时导出证书
     * 将流转化为 X509Certificate 对象
     *
     * @param is
     * @param provider
     * @return
     */
    public X509Certificate get0X509Certificate(InputStream is, BouncyCastleProvider provider) {
        if (is == null) {
            return null;
        }
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509", provider);
            X509Certificate certificate = (X509Certificate) cf.generateCertificate(is);

            convertCertFile(certificate);

        } catch (CertificateException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }


    /**
     * 将流转化为 X509Certificate 对象
     *
     * @param is
     * @param provider
     * @return
     */
    public X509Certificate getX509Certificate(InputStream is, BouncyCastleProvider provider) {
        if (is == null) {
            return null;
        }
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509", provider);
            X509Certificate certificate = (X509Certificate) cf.generateCertificate(is);
            is.close();
            return certificate;
        } catch (CertificateException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }


    /**
     * 读取根证书
     *
     * @param fileName
     * @return
     */
    public InputStream readRootCertFromAssets(String fileName) {
        Context context = AppConfig.getInstance().getContext();
        AssetManager assetManager = context.getAssets();
        try {

            return assetManager.open(fileName);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 根据芯片类型读取证书
     * 只验证根证书使用
     *
     * @param type
     * @param containerNum
     * @return
     */
    private InputStream readCertFromType(CryptoDevType type, int containerNum) {

        byte[][] fid = FidUtil.getCertFid(containerNum);


        byte[] certBuf = new byte[2048];
        int[] certLen = new int[1];
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return null;
        }
        int result = -1;

        //因为该方法目前只在验证根证书时使用，并且在现场即存在交换证书也存在签名证书的时候需要必须读出来证书
        if (AppConfig.getInstance().getCertConfig().getCertType(type, containerNum).equals(CertType.SIGNING)
                || TextUtils.isEmpty(AppConfig.getInstance().getCertConfig().getCertType(type, containerNum))) {
            result = cryptoInstance.readCert(fid[0], certBuf, certLen);
        } else if (AppConfig.getInstance().getCertConfig().getCertType(type, containerNum).equals(CertType.ENCRYPT)) {
            result = cryptoInstance.readCert(fid[1], certBuf, certLen);
        }

        cryptoInstance.close();
        if (result == 0) {
            return new ByteArrayInputStream(certBuf);
        } else {
            return null;
        }
    }


    //
    //
    //    //临时，测试完软证书公私钥对需删除该方法
    //    public  byte[] SM2PubKey2PubKey(CryptoDevType type) {
    //        SM2Pubkey      pubKey         = new SM2Pubkey();
    //        byte[]         pubFid         = new byte[]{0x00, 0x2D};
    //        CryptoInstance cryptoInstance = DeviceWrapper.getCryptoInstance(type);
    //        if (cryptoInstance == null) {
    //            return null;
    //        }
    //        int result = cryptoInstance.readSM2Pubkey(pubFid, pubKey);
    //        if (result != 0) {
    //            pubFid = new byte[]{0x00, 0x2A};
    //            result = cryptoInstance.readSM2Pubkey(pubFid, pubKey);
    //            if (result != 0) {
    //                return null;
    //            }
    //        }
    //        //获取公钥pub      SM2公钥byte[0]=4代表是SM2算法，紧跟着X,Y
    //        final byte pub[] = new byte[65];
    //        pub[0] = 4;
    //        System.arraycopy(pubKey.getX(), 0, pub, 1, pubKey.getX().length);
    //        System.arraycopy(pubKey.getY(), 0, pub, pubKey.getX().length + 1, pubKey.getY().length);
    ////        ModuleLog.e("芯片签名公钥" + new String(Base64.encode(pub)));
    ////        ModuleLog.e("芯片签名公钥x: " + new String(Base64.encode(pubKey.getX())));
    ////        ModuleLog.e("芯片签名公钥y: " + new String(Base64.encode(pubKey.getY())));
    //        return pub;
    //    }
    //
    //    public byte[] getChipCertPubicKey(CryptoDevType type, byte[] cert, int cert_len, com.xdja.cryptodev.devapi.CertDetailInfo certDetailInfo){
    //        CryptoInstance cryptoInstance = DeviceWrapper.getCryptoInstance(type);
    //        if (cryptoInstance == null) {
    //            return null;
    //        }
    //        byte pubicKey[] = cryptoInstance.getCertInfo(cert, cert_len, certDetailInfo);
    //        return pubicKey;
    //    }
}
