package com.xdja.safeclient.certcreation.device;

import android.text.TextUtils;

import com.aircert.util.ModuleLog;
import com.xdja.cryptodev.CryptoDevManager;
import com.xdja.cryptodev.CryptoDevType;
import com.xdja.cryptodev.devapi.CryptoInstance;
import com.xdja.cryptodev.devapi.SM2Pubkey;
import com.xdja.safeclient.certcreation.AppConfig;
import com.xdja.safeclient.certcreation.config.CertRule;
import com.xdja.safeclient.certcreation.config.CertType;
import com.xdja.safeclient.certcreation.config.XDJAAlgParams;
import com.xdja.safeclient.certcreation.util.FidUtil;

import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.DEROutputStream;
import org.bouncycastle.asn1.pkcs.CertificationRequest;
import org.bouncycastle.asn1.pkcs.CertificationRequestInfo;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.operator.DefaultSignatureAlgorithmIdentifierFinder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.util.encoders.Base64;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

/**
 * Created by jff on 2018/11/14.
 */

public class PKCS10ForSM2Utils extends PKCS10Utils {

    public static PKCS10ForSM2Utils pkcs10ForSM2Utils = new PKCS10ForSM2Utils();


    public PKCS10ForSM2Utils() {
    }

    public static PKCS10ForSM2Utils getInstance() {
        if (pkcs10ForSM2Utils == null) {
            pkcs10ForSM2Utils = new PKCS10ForSM2Utils();
        }
        return pkcs10ForSM2Utils;
    }

    /**
     * 生成SM2双证请求的p10
     *
     * @param type        卡类型
     * @param dn          主题
     * @param isApplyCert true-申请证书  false-更新证书信息
     * @return
     */
    @Override
    public String getP10RequestBC(CryptoDevType type, int containerNum, String dn, boolean isApplyCert) {
        ModuleLog.e("PKCS10ForSM2Utils getP10RequestBC");
        if (TextUtils.isEmpty(dn)) {
//            dn = "cn=李逍遥 452112199510214225,ou=00,ou=00,o=00,l=00,l=00,ST=41,C=CN";
            return "";
        }

        if (isApplyCert) {
            int keyResult = generateSM2Keys(type, containerNum);

            //TODO JFF 2018.7.20 需要跟zjc确认错误码处理
            if (keyResult != 0 && keyResult != -1) {
                return "errorCode" + CryptoDevManager.getInstance().getErrorText(keyResult);
            }
        }

        CryptoInstance cryptoInstance = DeviceWrapper.getCryptoInstance(type);
        if (cryptoInstance == null) {
            return "";
        }

        final byte pub[] = SM2PubKey2PubKey(type, containerNum);
        ASN1ObjectIdentifier eccOID = new ASN1ObjectIdentifier(XDJAAlgParams.OID_CURVE);//2.1是椭圆曲线的OID
        ASN1ObjectIdentifier sm2OID = new ASN1ObjectIdentifier(XDJAAlgParams.OID_SM2);//301是SM2曲线的OID
        SubjectPublicKeyInfo spi = new SubjectPublicKeyInfo(new AlgorithmIdentifier(eccOID, sm2OID), pub);
        //封装请求信息

        final X500Name x500Name = new X500Name(dn);
        CertificationRequestInfo cf1 = new CertificationRequestInfo(x500Name, spi, null);
        ByteArrayOutputStream outStream = new ByteArrayOutputStream();
        DEROutputStream derOutStream = new DEROutputStream(outStream);
        try {
            derOutStream.writeObject(cf1);
        } catch (IOException e) {
            e.printStackTrace();
        }
        byte[] certInfoBytes = outStream.toByteArray();

        //对CertificationRequestInfo进行签名
        byte[] signData = new byte[64];
        int[] signDataLen = new int[1];


        byte[][] keyPair = FidUtil.getKeyPairFid(type, containerNum);

        byte[] pubFid = keyPair[0];
        byte[] priFid = keyPair[1];
        int encodeRequestInfo = cryptoInstance.SM2Sign(pubFid, priFid, 1, certInfoBytes, certInfoBytes.length, signData, signDataLen);
        if (encodeRequestInfo != 0) {
//            ModuleLog.e("P10  encodeRequestInfo  " + encodeRequestInfo+ "    " + CryptoDevManager.getInstance().getErrorText(encodeRequestInfo));
            return null;
        }
        DERBitString dbs = new DERBitString(signData);
        AlgorithmIdentifier signIdentifier = new DefaultSignatureAlgorithmIdentifierFinder().find("SM3WITHSM2");
        CertificationRequest cr = new CertificationRequest(cf1, signIdentifier, dbs);
        PKCS10CertificationRequest request = new PKCS10CertificationRequest(cr);
        try {
            byte[] buffer = request.getEncoded();

            String p10 = new String(Base64.encode(buffer));
//            ModuleLog.e("P10请求字符串 ： " + p10);
//            checkP10(p10);
            return p10;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }


    /**
     * 生成SM2签名公私钥对
     *
     * @param type
     * @return -1 设备不存在
     * 0 生成公私钥对成功
     * 其他  根据错误码返回错误信息
     */
    private int generateSM2Keys(CryptoDevType type, int containerNum) {
        ModuleLog.e("PKCS10ForSM2Utils generateSM2Keys");
        int genSm2KeysPairResult = -1;
        CryptoInstance cryptoInstance = DeviceWrapper.getCryptoInstance(type);
        if (cryptoInstance == null) {
            return genSm2KeysPairResult;
        }
        AppConfig module = AppConfig.getInstance();
        //此处添加验证PIN，是因为生成公私钥操作返回-15（没有权限）
        int pinResult = DeviceWrapper.verifyPin(type, containerNum, "");
        if (pinResult == 0) {
            genSm2KeysPairResult = genSM2KeyPair(cryptoInstance, type, containerNum);
        } else {
            return pinResult;
        }

        if (genSm2KeysPairResult == 2030 || genSm2KeysPairResult == 2041) {//容器不存在
            //2030对应卡库错误码-20——文件不存在
            // 2041对应卡库错误码-31——密钥文件不存在
            if (cryptoInstance.createContainer(module.getRole(type, containerNum), module.getDefaultConfig().getPin(), containerNum, "") == 0) {
                genSm2KeysPairResult = genSM2KeyPair(cryptoInstance, type, containerNum);
            } else {
                return genSm2KeysPairResult;
            }
        }
        cryptoInstance.close();
        return genSm2KeysPairResult;
    }

    private int genSM2KeyPair(CryptoInstance cryptoInstance, CryptoDevType type, int containerNum) {
        ModuleLog.e("PKCS10ForSM2Utils genSM2KeyPair");
        int genSm2KeysPairResult;
        if (AppConfig.getInstance().getCertConfig().getCertRule(type, containerNum).equals(CertRule.SINGLE)) {
            if (AppConfig.getInstance().getCertConfig().getCertType(type, containerNum).equals(CertType.SIGNING)
                    || TextUtils.isEmpty(AppConfig.getInstance().getCertConfig().getCertType(type, containerNum))) {
                genSm2KeysPairResult = cryptoInstance.genSignSM2KeyPair(containerNum);
            } else {
                genSm2KeysPairResult = cryptoInstance.genExchangeSM2KeyPair(containerNum);
            }
        } else {
            genSm2KeysPairResult = cryptoInstance.genSignSM2KeyPair(containerNum);
        }
        return genSm2KeysPairResult;
    }


    private byte[] SM2PubKey2PubKey(CryptoDevType type, int containerNum) {
        SM2Pubkey pubKey = new SM2Pubkey();
//        byte[] pubFid = new byte[]{0x00, 0x2D};
        byte[][] pubFid = FidUtil.getKeyPairFid(type, containerNum);
        CryptoInstance cryptoInstance = DeviceWrapper.getCryptoInstance(type);
        if (cryptoInstance == null) {
            return null;
        }
        int result = cryptoInstance.readSM2Pubkey(pubFid[0], pubKey);
        if (result != 0) {
            return null;
        }
        //获取公钥pub      SM2公钥byte[0]=4代表是SM2算法，紧跟着X,Y
        final byte pub[] = new byte[65];
        pub[0] = 4;
        System.arraycopy(pubKey.getX(), 0, pub, 1, pubKey.getX().length);
        System.arraycopy(pubKey.getY(), 0, pub, pubKey.getX().length + 1, pubKey.getY().length);
//        ModuleLog.e("签名公钥" + new String(Base64.encode(pub)));
        return pub;
    }


    private void checkP10(CryptoDevType type, String p10) {
        if (p10 == null || p10.equals("")) {
            return;
        }

        try {
            byte[] bs = Base64.decode(p10);
            PKCS10CertificationRequest re = new PKCS10CertificationRequest(bs);
            CertificationRequest request = re.toASN1Structure();
            CertificationRequestInfo cerReqInfo = request.getCertificationRequestInfo();

            ByteArrayOutputStream outStream = new ByteArrayOutputStream();
            DEROutputStream derOutStream = new DEROutputStream(outStream);
            try {
                derOutStream.writeObject(cerReqInfo);
            } catch (IOException e) {
                e.printStackTrace();
            }
            byte[] certInfoBytes = outStream.toByteArray();

            byte[] signature = re.getSignature();
            SubjectPublicKeyInfo info = re.getSubjectPublicKeyInfo();
            DERBitString keyInfo = info.getPublicKeyData();
            byte[] bytes = keyInfo.getBytes();

            byte x[] = new byte[32];
            byte y[] = new byte[32];
            System.arraycopy(bytes, 1, x, 0, x.length);
            System.arraycopy(bytes, x.length + 1, y, 0, y.length);


            SM2Pubkey pubkey = new SM2Pubkey();
            pubkey.setX(x);
            pubkey.setY(y);

            byte pubFid[] = {0x00, 0x00};
            CryptoInstance cryptoInstance = DeviceWrapper.getCryptoInstance(type);
            int ret = -1;
            if (cryptoInstance != null) {
                ret = cryptoInstance.SM2SignVerify(pubFid, 1, pubkey, certInfoBytes, certInfoBytes.length, signature);

            }
//            ModuleLog.d("验签结果： = " + ret);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


}
