package com.xdja.widget.datetimepicker;

import android.graphics.drawable.ColorDrawable;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v4.app.DialogFragment;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentManager;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.xdja.widget.datetimepicker.fragment.CalendarFragment;
import com.xdja.widget.datetimepicker.utils.CalendarUtils;
import com.xdja.widget.datetimepicker.view.calendarview.Calendar;
import com.xdja.widget.datetimepicker.view.calendarview.CalendarView;

import java.util.Date;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

/**
 * desc:日期区间选择
 * author: lyl
 * 2019/4/10
 */
public class DateSectionPickDialog extends DialogFragment {
    private List<Fragment> fragmentList = new ArrayList<>();
    private CalendarFragment startCalendarFragment, finishCalendarFragment;
    private TextView tvDateStart;
    private TextView tvDateFinish;
    private RelativeLayout rlTableStart;
    private RelativeLayout rlTableFinish;
    private View viewStartLine;
    private View viewFinishLine;
    private Button btnConfirm;
    private FragmentManager manager;
    /**
     * 默认 日期格式
     */
    public static final String PARAM_DATE_FORMAT = "PARAM_DATE_FORMAT";
    /**
     * 初始化开始时间
     */
    public static final String PARAM_INIT_START_TIME = "param_init_start_time";
    /**
     * 结束时间
     */
    public static final String PARAM_INIT_END_TIME = "param_init_end_time";
    /**
     * 默认年月日 显示形式
     */
    private String DATE_FORMATE_DEFAULT = "yyyy-MM-dd";

    private static final int ERROR_STARTE_CODE = 0;
    private static final int ERROR_FINISH_CODE = 1;

    /**
     * * 创建dialog对象
     * * @param dateFormate 年月日tab显示格式,null则为"yyyy-MM-dd"
     * * @return
     */
    public static DateSectionPickDialog newInstance(String dateFormat) {
        Bundle bundle = new Bundle();
        if (!TextUtils.isEmpty(dateFormat)) {
            bundle.putString(PARAM_DATE_FORMAT, dateFormat);
        }
        return newInstance(bundle);
    }

    /**
     * * 创建dialog对象
     * * @param dateFormat 年月日tab显示格式,null则为"yyyy-MM-dd"
     * * @return
     */
    public static DateSectionPickDialog newInstance(Bundle bundle) {
        DateSectionPickDialog pickDialog = new DateSectionPickDialog();
        if (bundle != null) {
            pickDialog.setArguments(bundle);
        }
        return pickDialog;
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.datetimepicker_dialog_date_section_picker, container, false);
        return view;
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        initUi(view);
        initTab();
        initFragment();
    }

    @Override
    public void onStart() {
        super.onStart();
        Window win = getDialog().getWindow();
        // 一定要设置Background，如果不设置，window属性设置无效
        win.setBackgroundDrawable(new ColorDrawable(getResources().getColor(R.color.datetimepicker_transport)));
        DisplayMetrics dm = new DisplayMetrics();
        getActivity().getWindowManager().getDefaultDisplay().getMetrics(dm);
        WindowManager.LayoutParams params = win.getAttributes();
        params.gravity = Gravity.BOTTOM;
        // 使用ViewGroup.LayoutParams，以便Dialog 宽度充满整个屏幕
        getDialog().setCancelable(true);
        params.width = ViewGroup.LayoutParams.MATCH_PARENT;
        params.height = ViewGroup.LayoutParams.WRAP_CONTENT;
        win.setAttributes(params);
    }

    private void initUi(View dialog) {
        //注册view
        rlTableStart = (RelativeLayout) dialog.findViewById(R.id.rl_tab1);
        rlTableFinish = (RelativeLayout) dialog.findViewById(R.id.rl_tab2);
        tvDateStart = (TextView) dialog.findViewById(R.id.tv_tab1);
        tvDateFinish = (TextView) dialog.findViewById(R.id.tv_tab2);
        btnConfirm = (Button) dialog.findViewById(R.id.btn_confirm);
        viewStartLine = dialog.findViewById(R.id.view_line1);
        viewFinishLine = dialog.findViewById(R.id.view_line2);
        btnConfirm.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (onResultListener != null) {
                    long startTime = getReturnDate(startCalendarFragment);
                    long endTime = getReturnDate(finishCalendarFragment);

                    SimpleDateFormat formatter = CalendarUtils.getDateFormat(getDateFormate());
                    onResultListener.onResult(DateSectionPickDialog.this, formatter.format(startTime), startTime, formatter.format(endTime), endTime);
                    DateSectionPickDialog.this.dismiss();
                }
            }
        });
        rlTableStart.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                selectTab(0);
            }
        });
        rlTableFinish.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                selectTab(1);
            }
        });
    }

    private void initFragment() {
        manager = getChildFragmentManager();
        for (Fragment obj : fragmentList) {
            manager.beginTransaction().add(R.id.framlayout, obj).hide(obj).commit();
        }
        setCurrentItem(0);
    }

    /**
     * 获取返回的日期
     *
     * @return
     */
    private long getReturnDate(CalendarFragment calendarFragment) {
        java.util.Calendar calendar = CalendarUtils.getCalendar();
        calendar.set(calendarFragment.getCurrentYear(), calendarFragment.getCurrentMonth() - 1, calendarFragment.getCurrentDay());
        return calendar.getTime().getTime();
    }

    private Date getSelectDate(CalendarFragment calendarFragment) {
        java.util.Calendar calendar = CalendarUtils.getCalendar();
        calendar.set(calendarFragment.getCurrentYear(), calendarFragment.getCurrentMonth() - 1, calendarFragment.getCurrentDay());
        return calendar.getTime();
    }


    /**
     * 切换Tab状态转换
     *
     * @param tablePosition
     */
    private void selectTab(int tablePosition) {
        switch (tablePosition) {
            case 0:
                viewStartLine.setVisibility(View.VISIBLE);
                viewFinishLine.setVisibility(View.GONE);
                break;
            case 1:
                viewStartLine.setVisibility(View.GONE);
                viewFinishLine.setVisibility(View.VISIBLE);
                break;
        }
        setCurrentItem(tablePosition);
    }

    private void initTab() {
        //fragment对象和tab初始化
        fragmentList.clear();
        tvDateStart.setText(initTabTitle(getInitStartTime()));
        tvDateFinish.setText(initTabTitle(getInitEndTime()));
        startCalendarFragment = new CalendarFragment();
        finishCalendarFragment = new CalendarFragment();
        fragmentList.add(startCalendarFragment);
        fragmentList.add(finishCalendarFragment);

        Bundle arguments = getArguments();


        if (startCalendarFragment != null) {
            startCalendarFragment.setOnDateSelectedListener(new CalendarView.OnCalendarSelectListener() {
                @Override
                public void onCalendarOutOfRange(Calendar calendar) {
                }

                @Override
                public void onCalendarSelect(Calendar calendar, boolean isClick) {
                    if (!isClick) {//未点击返回
                        return;
                    }
                    verifyDate(0, calendar);
                }
            });
            //初始化年月日
            java.util.Calendar calendar = CalendarUtils.getCalendar();
            calendar.setTimeInMillis(getInitStartTime());
            startCalendarFragment.setInitDate(calendar.get(java.util.Calendar.YEAR), calendar.get(java.util.Calendar.MONTH) + 1, calendar.get(java.util.Calendar.DAY_OF_MONTH));
        }
        if (finishCalendarFragment != null) {
            finishCalendarFragment.setOnDateSelectedListener(new CalendarView.OnCalendarSelectListener() {
                @Override
                public void onCalendarOutOfRange(Calendar calendar) {
                }

                @Override
                public void onCalendarSelect(Calendar calendar, boolean isClick) {
                    if (!isClick) {//未点击返回
                        return;
                    }
                    verifyDate(1, calendar);
                }
            });
            //初始化年月日
            java.util.Calendar calendar = CalendarUtils.getCalendar();
            calendar.setTimeInMillis(getInitEndTime());
            finishCalendarFragment.setInitDate(calendar.get(java.util.Calendar.YEAR), calendar.get(java.util.Calendar.MONTH) + 1, calendar.get(java.util.Calendar.DAY_OF_MONTH));
        }
    }

    /**
     * 时间正确性校验
     *
     * @return
     */
    private void verifyDate(int tabPosition, Calendar calendar) {
        switch (tabPosition) {
            case 0:
                if (getSelectDate(startCalendarFragment).after(getSelectDate(finishCalendarFragment))) {
                    startCalendarFragment.setConstraintSelectDate(finishCalendarFragment.getCurrentYear(),
                            finishCalendarFragment.getCurrentMonth(), finishCalendarFragment.getCurrentDay());
                    if (onResultListener != null) {
                        onResultListener.onSelectDateError(ERROR_STARTE_CODE);
                    }
                    updateTabTitle(startCalendarFragment.getCurrentYear(), startCalendarFragment.getCurrentMonth() - 1, startCalendarFragment.getCurrentDay(), 0);
                } else {
                    updateTabTitle(calendar.getYear(), calendar.getMonth() - 1, calendar.getDay(), 0);
                    //  selectTab(1);
                }
                break;
            case 1:
                if (getSelectDate(finishCalendarFragment).before(getSelectDate(startCalendarFragment))) {
                    finishCalendarFragment.setConstraintSelectDate(startCalendarFragment.getCurrentYear(),
                            startCalendarFragment.getCurrentMonth(), startCalendarFragment.getCurrentDay());
                    if (onResultListener != null) {
                        onResultListener.onSelectDateError(ERROR_FINISH_CODE);
                    }
                    updateTabTitle(finishCalendarFragment.getCurrentYear(), finishCalendarFragment.getCurrentMonth() - 1, finishCalendarFragment.getCurrentDay(), 1);
                } else {
                    updateTabTitle(calendar.getYear(), calendar.getMonth() - 1, calendar.getDay(), 1);
                }
                break;
        }
    }

    /**
     * 切换Fragment界面
     *
     * @param index
     */
    private void setCurrentItem(int index) {
        for (int i = 0; i < fragmentList.size(); i++) {
            Fragment obj = fragmentList.get(i);
            if (index == i) {//同一个显示
                manager.beginTransaction().show(obj).commit();
            } else {//隐藏其他
                manager.beginTransaction().hide(obj).commit();
            }
        }
    }

    /**
     * 根据日历 更新tab标题
     *
     * @param year
     * @param month
     * @param day
     * @param tabPosition
     */
    private void updateTabTitle(int year, int month, int day, int tabPosition) {
        java.util.Calendar calendar = CalendarUtils.getCalendar();
        calendar.set(year, month, day);
        SimpleDateFormat formatter = CalendarUtils.getDateFormat(getDateFormate());
        String dateString = formatter.format(calendar.getTime());
        if (tabPosition == 0) {
            tvDateStart.setText(dateString);
        } else {
            tvDateFinish.setText(dateString);
        }
    }

    /**
     * 初始化tab标题
     *
     * @return
     */
    private String initTabTitle(long initTime) {
        java.util.Calendar calendar = CalendarUtils.getCalendar();
        calendar.setTimeInMillis(initTime);
        SimpleDateFormat formatter = CalendarUtils.getDateFormat(getDateFormate());
        String st = formatter.format(calendar.getTimeInMillis());
        return st;
    }

    private long getInitStartTime() {
        Bundle arguments = getArguments();
        if (arguments != null) {
            return arguments.getLong(PARAM_INIT_START_TIME, getDateTime());
        }
        return getDateTime();
    }

    private long getInitEndTime() {
        Bundle arguments = getArguments();
        if (arguments != null) {
            return arguments.getLong(PARAM_INIT_END_TIME, getDateTime());
        }
        return getDateTime();
    }

    /**
     * 获取默认时间
     *
     * @return
     */
    private long getDateTime() {
        return System.currentTimeMillis();
    }

    /**
     * 获取日期显示形式
     *
     * @return
     */
    private String getDateFormate() {
        String tmp = DATE_FORMATE_DEFAULT;
        if (getArguments() != null) {
            tmp = getArguments().getString(PARAM_DATE_FORMAT, tmp);
        }
        return tmp;
    }

    /**
     * 确定时结果回调
     */
    private OnResultListener onResultListener;

    public void setOnResultListener(OnResultListener onResultListener) {
        this.onResultListener = onResultListener;
    }

    public interface OnResultListener {
        /**
         * 返回值
         */
        void onResult(DateSectionPickDialog dialog, String startDate, long startTimeMillis, String finishDate, long endTimeMillis);

        /**
         * 0  开始日期不能大于结束日期；1 结束日期不能小于开始日期
         *
         * @param errorCode
         */
        void onSelectDateError(int errorCode);
    }
}
