package com.xdja.location;

import android.content.Intent;
import android.graphics.drawable.AnimationDrawable;
import android.location.Location;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.annotation.StringRes;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AppCompatActivity;
import android.text.TextUtils;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.TextView;

import com.amap.api.location.AMapLocation;
import com.amap.api.maps2d.AMap;
import com.amap.api.maps2d.AMapUtils;
import com.amap.api.maps2d.CameraUpdateFactory;
import com.amap.api.maps2d.UiSettings;
import com.amap.api.maps2d.model.BitmapDescriptor;
import com.amap.api.maps2d.model.BitmapDescriptorFactory;
import com.amap.api.maps2d.model.CameraPosition;
import com.amap.api.maps2d.model.CircleOptions;
import com.amap.api.maps2d.model.LatLng;
import com.amap.api.maps2d.model.Marker;
import com.amap.api.maps2d.model.MarkerOptions;
import com.amap.api.maps2d.model.MyLocationStyle;
import com.amap.api.services.core.AMapException;
import com.amap.api.services.core.LatLonPoint;
import com.amap.api.services.core.PoiItem;
import com.amap.api.services.poisearch.PoiResult;
import com.amap.api.services.poisearch.PoiSearch;
import com.blankj.utilcode.utils.ConvertUtils;
import com.blankj.utilcode.utils.ToastUtils;
import com.xdja.eoa.commonutil.ArraysUtils;
import com.xdja.eoa.commonwidget.widget.CommonToolBar;
import com.xdja.eoa.commonwidget.widget.OnToolBarButtonClickListener;
import com.xdja.eoa.commonwidget.widget.ToolBarButtonType;
import com.xdja.eoa.commonwidget.widget.ToolbarButton;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * @author wanjing
 * @date 2019/6/13
 * <p>
 * 位置微调（进入地图，选择附近指定范围内的位置进行微调） 注：地图上会标注两个地点，一个是定位地点，一个是用户选择地点
 */
public class ChooseLocationActivity extends AppCompatActivity implements
        AMap.OnMyLocationChangeListener, PoiSearch.OnPoiSearchListener, AMap.OnCameraChangeListener,
        OnToolBarButtonClickListener, View.OnClickListener {

    public static final String EXTRA_RADIUS = "extra_radius";
    public static final String EXTRA_LATITUDE = "extra_latitude";
    public static final String EXTRA_LONGITUDE = "extra_longitude";
    public static final String EXTRA_POI_NAME = "extra_poi_name";
    public static final String EXTRA_ADDRESS = "extra_address";
    public static final String EXTRA_ERROR_CODE = "extra_error_code";

    private CommonToolBar mToolBar;
    /**
     * 地图控件引用
     */
    private EoaMapView mMapView;
    private ListView mListView;
    private ImageView mZoomIn;
    private ImageView mZoomOut;
    private View mLoadingView;
    private ImageView mProgress;
    private View mErrorView;
    private TextView mErrorTip;

    /**
     * 地图的控制器类，用来操作地图
     */
    private AMap mAMap;

    private float mZoomLevel = 17f;
    /**
     * 用于设置位置微调的半径，中心点为定位地点
     */
    private int mRadius;
    private int mMaxDistance = 8000;
    /**
     * 用户选择地点的经纬度
     */
    private double mLatitude;
    private double mLongitude;
    /**
     * 用于标注用户选择的地点
     */
    private Marker locationMarker;

    private PoiSearchAdapter mAdapter;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.location_activity_choose_location);

        getData();
        initView();
        initMap(savedInstanceState);
        initAdapter();
    }

    private void getData() {
        Bundle bundle = getIntent().getExtras();
        if (bundle != null) {
            mRadius = bundle.getInt(EXTRA_RADIUS);
            mMaxDistance = mRadius > 0 ? mRadius : mMaxDistance;
            mLatitude = bundle.getDouble(EXTRA_LATITUDE);
            mLongitude = bundle.getDouble(EXTRA_LONGITUDE);
        }
    }

    private void initView() {
        mToolBar = (CommonToolBar) findViewById(R.id.location_toolbar);
        mToolBar.setBarButtonClickListener(this);
        mMapView = (EoaMapView) findViewById(R.id.location_mapview);
        mListView = (ListView) findViewById(R.id.location_listview);

        ImageView locate = (ImageView) findViewById(R.id.location_locate);
        mZoomIn = (ImageView) findViewById(R.id.location_zoom_in);
        mZoomOut = (ImageView) findViewById(R.id.location_zoom_out);
        locate.setOnClickListener(this);
        mZoomIn.setOnClickListener(this);
        mZoomOut.setOnClickListener(this);

        mLoadingView = findViewById(R.id.location_loading);
        mProgress = (ImageView) findViewById(R.id.location_loading_progress);
        mErrorView = findViewById(R.id.location_error);
        mErrorTip = (TextView) findViewById(R.id.location_error_tip);

        disableToolbarBtn();
        showLoadingView();
    }

    private void initMap(Bundle savedInstanceState) {
        //在activity执行onCreate时执行mMapView.onCreate(savedInstanceState). 创建地图
        mMapView.onCreate(savedInstanceState);
        if (mAMap == null) {
            mAMap = mMapView.getMap();
        }
        UiSettings uiSettings = mAMap.getUiSettings();
        //隐藏缩放按钮
        uiSettings.setZoomControlsEnabled(false);

        MyLocationStyle myLocationStyle = new MyLocationStyle();
        //定位一次，且将视角移动到地图中心点
        myLocationStyle.myLocationType(MyLocationStyle.LOCATION_TYPE_LOCATE);
        BitmapDescriptor locationIcon = BitmapDescriptorFactory.fromResource(R.drawable.location_ic_detail_marker);
        myLocationStyle.myLocationIcon(locationIcon);
        myLocationStyle.radiusFillColor(ContextCompat.getColor(this, android.R.color.transparent));
        myLocationStyle.strokeColor(ContextCompat.getColor(this, android.R.color.transparent));
        //设置定位蓝点的style
        mAMap.setMyLocationStyle(myLocationStyle);
        //设置为true表示启动显示定位蓝点
        mAMap.setMyLocationEnabled(true);
        //设置地图缩放级别为17f，缩放范围一般是3f-19f
        mAMap.animateCamera(CameraUpdateFactory.zoomTo(mZoomLevel));
        mAMap.setOnMyLocationChangeListener(this);
        mAMap.setOnCameraChangeListener(this);
    }

    private void initAdapter() {
        mAdapter = new PoiSearchAdapter(this);
        mListView.setAdapter(mAdapter);
        mListView.setOnItemClickListener(new AdapterView.OnItemClickListener() {
            @Override
            public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
                LocationActivity.selectedItem = (PoiItem) mAdapter.getItem(position);
                mAdapter.notifyDataSetChanged();

                //在地图上标注用户选择的地点
                addOrUpdateMarker();
            }
        });
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        //在activity执行onDestroy时执行mMapView.onDestroy(). 销毁地图
        mMapView.onDestroy();
    }

    @Override
    protected void onResume() {
        super.onResume();
        //在activity执行onResume时执行mMapView.onResume(). 重新绘制加载地图
        mMapView.onResume();
    }

    @Override
    protected void onPause() {
        super.onPause();
        //在activity执行onPause时执行mMapView.onPause(). 暂停地图的绘制
        mMapView.onPause();
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        //在activity执行onSaveInstanceState时执行mMapView.onSaveInstanceState(outState). 保存地图当前的状态
        mMapView.onSaveInstanceState(outState);
    }

    @Override
    public void onMyLocationChange(Location location) {
        int errorCode = location.getExtras().getInt("errorCode");
        if (errorCode == AMapLocation.LOCATION_SUCCESS) {
            //定位成功
            if (mRadius > 0) {
                //构造一个圆形，在地图上标注位置微调的范围
                mAMap.addCircle(new CircleOptions()
                        .center(new LatLng(location.getLatitude(), location.getLongitude()))
                        .radius(mRadius)
                        .fillColor(ContextCompat.getColor(this, R.color.location_circle_fill_color))
                        .strokeColor(ContextCompat.getColor(this, R.color.location_circle_stroke_color))
                        .strokeWidth(ConvertUtils.dp2px(1)));
            }
            //周边检索POI
            String cityCode = location.getExtras().getString("citycode");
            doPoiSearch(cityCode, location.getLatitude(), location.getLongitude());
        } else {
            //定位失败
            hideLoadingView();
            showErrorView(R.string.location_failure);
        }
    }

    /**
     * 周边检索POI, 中心点为定位地点
     */
    private void doPoiSearch(String cityCode, double latitude, double longitude) {
        //第一个参数表示搜索字符串，第二个参数表示poi搜索类型，第三个参数表示poi搜索区域（空字符串代表全国）
        String ctgr = "政府机构及社会团体|公司企业|商务住宅|科教文化服务|金融保险服务|公共设施|生活服务";
        PoiSearch.Query query = new PoiSearch.Query("", ctgr, cityCode);
        query.setPageSize(1000);
        query.setPageNum(0);
        PoiSearch poiSearch = new PoiSearch(this, query);
        //设置周边搜索的中心点以及半径
        poiSearch.setBound(new PoiSearch.SearchBound(new LatLonPoint(latitude, longitude), mMaxDistance));
        poiSearch.setOnPoiSearchListener(this);
        poiSearch.searchPOIAsyn();
    }

    @Override
    public void onPoiSearched(PoiResult poiResult, int rCode) {
        hideLoadingView();
        if (rCode == AMapException.CODE_AMAP_SUCCESS) {
            //检索成功，解析result获取POI信息
            if (poiResult == null) {
                return;
            }
            List<PoiItem> poiItems = poiResult.getPois();
            if (ArraysUtils.isListEmpty(poiItems)) {
                return;
            }
            if (mLatitude > 0 && mLongitude > 0) {
                findPoi(new LatLng(mLatitude, mLongitude), poiItems);
            }
            mAdapter.setData(poiItems);
            mAdapter.notifyDataSetChanged();

            enableToolbarBtn();
        } else {
            showErrorView(R.string.location_poi_error);
        }
    }

    @Override
    public void onPoiItemSearched(PoiItem poiItem, int i) {

    }

    /**
     * 遍历查找距离用户选择地点最近的 POI
     */
    private void findPoi(final LatLng selectedLatLng, List<PoiItem> poiItems) {
        PoiItem poiItem = Collections.min(poiItems, new Comparator<PoiItem>() {
            @Override
            public int compare(PoiItem o1, PoiItem o2) {
                LatLng latLng1 = new LatLng(o1.getLatLonPoint().getLatitude(), o1.getLatLonPoint().getLongitude());
                LatLng latLng2 = new LatLng(o2.getLatLonPoint().getLatitude(), o2.getLatLonPoint().getLongitude());
                float distance1 = AMapUtils.calculateLineDistance(selectedLatLng, latLng1);
                float distance2 = AMapUtils.calculateLineDistance(selectedLatLng, latLng2);
                return Float.compare(distance1, distance2);
            }
        });
        //判断用户选择的地点是否在范围内
        LatLng latLng = new LatLng(poiItem.getLatLonPoint().getLatitude(), poiItem.getLatLonPoint().getLongitude());
        float distance = AMapUtils.calculateLineDistance(selectedLatLng, latLng);
        if (distance <= mMaxDistance) {
            LocationActivity.selectedItem = poiItem;
            addOrUpdateMarker();
        }
    }

    /**
     * 在地图上标注用户选择的地点
     */
    private void addOrUpdateMarker() {
        LatLonPoint point = LocationActivity.selectedItem.getLatLonPoint();
        LatLng latLng = new LatLng(point.getLatitude(), point.getLongitude());
        if (locationMarker == null) {
            locationMarker = mAMap.addMarker(new MarkerOptions()
                    .position(latLng)
                    .draggable(true)
                    .icon(BitmapDescriptorFactory.fromResource(R.drawable.location_icon_local)));
            return;
        }
        locationMarker.setPosition(latLng);
    }

    private void enableToolbarBtn() {
        ToolbarButton toolbarButton = mToolBar.getButtonIdByType(ToolBarButtonType.RIGHT_FIRST_BUTTON);
        toolbarButton.setEnabled(true);
        toolbarButton.setColor(ContextCompat.getColor(this, R.color.location_title_right));
    }

    private void disableToolbarBtn() {
        ToolbarButton toolbarButton = mToolBar.getButtonIdByType(ToolBarButtonType.RIGHT_FIRST_BUTTON);
        toolbarButton.setEnabled(false);
        toolbarButton.setColor(ContextCompat.getColor(this, R.color.location_title_right_disabled));
    }

    private void showLoadingView() {
        mLoadingView.setVisibility(View.VISIBLE);
        AnimationDrawable animationDrawable = (AnimationDrawable) mProgress.getDrawable();
        animationDrawable.start();
    }

    private void hideLoadingView() {
        mLoadingView.setVisibility(View.GONE);
        AnimationDrawable animationDrawable = (AnimationDrawable) mProgress.getDrawable();
        animationDrawable.stop();
    }

    private void showErrorView(@StringRes int errorTip) {
        mListView.setVisibility(View.GONE);
        mErrorView.setVisibility(View.VISIBLE);
        mErrorTip.setText(errorTip);
    }

    @Override
    public void onCameraChange(CameraPosition cameraPosition) {

    }

    @Override
    public void onCameraChangeFinish(CameraPosition cameraPosition) {
        mZoomLevel = cameraPosition.zoom;
        if (mZoomLevel >= mAMap.getMaxZoomLevel()) {
            mZoomIn.setImageResource(R.drawable.location_ic_zoom_in_disable);
        } else if (mZoomLevel <= mAMap.getMinZoomLevel()) {
            mZoomOut.setImageResource(R.drawable.location_ic_zoom_out_disable);
        } else {
            mZoomIn.setImageResource(R.drawable.location_ic_zoom_in);
            mZoomOut.setImageResource(R.drawable.location_ic_zoom_out);
        }
    }

    @Override
    public void onClick(View v, ToolBarButtonType type) {
        if (type == ToolBarButtonType.LEFT_FIRST_BUTTON) {
            finish();
        } else if (type == ToolBarButtonType.RIGHT_FIRST_BUTTON) {
            if (LocationActivity.selectedItem == null) {
                ToastUtils.showShortToast("请选择位置");
                return;
            }
            PoiItem poiItem = LocationActivity.selectedItem;
            LocationActivity.selectedItem = null;

            Intent intent = new Intent();
            intent.putExtra(EXTRA_LATITUDE, poiItem.getLatLonPoint().getLatitude());
            intent.putExtra(EXTRA_LONGITUDE, poiItem.getLatLonPoint().getLongitude());
            String title = poiItem.getTitle();
            String snippet = poiItem.getSnippet();
            intent.putExtra(EXTRA_POI_NAME, title);
            intent.putExtra(EXTRA_ADDRESS, TextUtils.isEmpty(snippet) ? title : snippet);
            setResult(RESULT_OK, intent);
            finish();
        }
    }

    @Override
    public void onClick(View v) {
        int id = v.getId();
        if (id == R.id.location_zoom_in) {
            //放大
            mAMap.animateCamera(CameraUpdateFactory.zoomTo(++mZoomLevel));
        } else if (id == R.id.location_zoom_out) {
            //缩小
            mAMap.animateCamera(CameraUpdateFactory.zoomTo(--mZoomLevel));
        } else if (id == R.id.location_locate) {
            //定位
        }
    }
}
