package com.xdja.location;

import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Rect;
import android.graphics.drawable.AnimationDrawable;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Bundle;
import android.os.Environment;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AppCompatActivity;
import android.text.TextUtils;
import android.util.Log;
import android.view.DragEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.inputmethod.InputMethodManager;
import android.widget.AdapterView;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.RelativeLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.amap.api.location.AMapLocation;
import com.amap.api.location.AMapLocationClient;
import com.amap.api.location.AMapLocationClientOption;
import com.amap.api.location.AMapLocationListener;
import com.amap.api.maps2d.AMap;
import com.amap.api.maps2d.AMapOptions;
import com.amap.api.maps2d.CameraUpdateFactory;
import com.amap.api.maps2d.LocationSource;
import com.amap.api.maps2d.MapView;
import com.amap.api.maps2d.UiSettings;
import com.amap.api.maps2d.model.BitmapDescriptor;
import com.amap.api.maps2d.model.BitmapDescriptorFactory;
import com.amap.api.maps2d.model.CameraPosition;
import com.amap.api.maps2d.model.LatLng;
import com.amap.api.maps2d.model.MarkerOptions;
import com.amap.api.maps2d.model.MyLocationStyle;
import com.amap.api.services.core.LatLonPoint;
import com.amap.api.services.core.PoiItem;
import com.amap.api.services.geocoder.GeocodeResult;
import com.amap.api.services.geocoder.GeocodeSearch;
import com.amap.api.services.geocoder.RegeocodeQuery;
import com.amap.api.services.geocoder.RegeocodeResult;
import com.amap.api.services.poisearch.PoiResult;
import com.amap.api.services.poisearch.PoiSearch;
import com.blankj.utilcode.utils.LogUtils;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * created on 2017/8/28.
 * author:wangkezhi
 * email:45436660@qq.com
 * summary:选择位置
 */

public class LocationActivity extends AppCompatActivity implements AMap.OnMapScreenShotListener,
        LocationSource, GeocodeSearch.OnGeocodeSearchListener, PoiSearch.OnPoiSearchListener, AMapLocationListener, AMap.OnCameraChangeListener {
    private AMap aMap;
    private Context mContext;

    private PoiSearchAdapter adapter;
    private List<PoiItem> items = new ArrayList<>();
    private PoiResult poiResult;
    public static PoiItem selectedItem;
    private AMapLocation currentLocation;//当前位置

    //声明AMapLocationClient类对象
    public AMapLocationClient mLocationClient = null;
    //声明AMapLocationClientOption对象
    public AMapLocationClientOption mLocationOption = null;
    //声明mListener对象，定位监听器
    private LocationSource.OnLocationChangedListener mListener = null;

    private int currentPage = 0;// 当前页面，从0开始计数
    private PoiSearch.Query query;// Poi查询条件类
    private PoiSearch poiSearch;// POI搜索
    private GeocodeSearch geocoderSearch;//地理编码
    private boolean isFromTopicRelease;//是否从发布话题界面进入

    private MapView mapView;
    private ListView listView;
    private ImageView myLocationBtn;
    private RelativeLayout progressLayout;
    private RelativeLayout bottomLayout;
    private RelativeLayout mapLayout;
    private RelativeLayout errorLayout;
    private ImageView dialog_progress;

    public static final String MAP = "map";
    public static final String ADDRESS = "address";
    public static final String LONGITUDE = "longitude";
    public static final String LATITUDE = "latitude";
    public static final String CITYCODE = "citycode";
    public static final String ADDRESS_DETAIL = "addressdetail";
    public static final String FROM_TOPIC_RELEASE = "from_topic_release";
    public static final String IS_CAN_CHANGE_LOCATION = "is_can_change_location";

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.location_activity_location);
        mContext = this;
        initview(savedInstanceState);
        initLogice();
    }

    private void initview(Bundle savedInstanceState) {
        mapView = (MapView) findViewById(R.id.map);
        mapView.onCreate(savedInstanceState);// 此方法必须重写

        listView = (ListView) findViewById(R.id.listView);
        adapter = new PoiSearchAdapter(this);
        listView.setAdapter(adapter);
        listView.setOnItemClickListener(new AdapterView.OnItemClickListener() {
            @Override
            public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
                if (isCanChangeLocation()) {
                    listItemClick(position);
                }
            }
        });

        myLocationBtn = (ImageView) findViewById(R.id.myLocationBtn);
        myLocationBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                startLocation();
            }
        });

        mapView.setOnDragListener(new View.OnDragListener() {
            @Override
            public boolean onDrag(View v, DragEvent event) {
                ((InputMethodManager) getSystemService(INPUT_METHOD_SERVICE))
                        .hideSoftInputFromWindow(v.getWindowToken(),
                                InputMethodManager.HIDE_NOT_ALWAYS);
                return false;
            }
        });

        progressLayout = (RelativeLayout) findViewById(R.id.progress);
        bottomLayout = (RelativeLayout) findViewById(R.id.bottomLayout);
        mapLayout = (RelativeLayout) findViewById(R.id.mapLayout);
        errorLayout = (RelativeLayout) findViewById(R.id.error_layout);
        dialog_progress = (ImageView) findViewById(R.id.dialog_progress);

        errorLayout.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                errorLayoutClick();
            }
        });

        if (!isNetworkAvailable()) {
            Toast.makeText(mContext, getString(R.string.location_network_error_hint), Toast.LENGTH_SHORT).show();
            showErrorLayout();
        } else {
            errorLayout.setVisibility(View.GONE);
//            mapLayout.setVisibility(View.VISIBLE);
            bottomLayout.setVisibility(View.VISIBLE);
        }
    }

    private void initLogice() {
        getData(getIntent());
        //启动定位
        initMap();
        initLocate();
    }


    /**
     * 设置是否可修改定位点
     */
    public boolean isCanChangeLocation() {
        if (getIntent() == null) {
            return true;
        }
        return getIntent().getBooleanExtra(IS_CAN_CHANGE_LOCATION, true);
    }

    private void initLocate() {
        if (isNetworkAvailable()) {
            if (isFromTopicRelease && selectedItem != null) {
                //设置缩放级别
                aMap.moveCamera(CameraUpdateFactory.zoomTo(17));
                //将地图移动到定位点
                aMap.moveCamera(CameraUpdateFactory.changeLatLng(new LatLng(selectedItem.getLatLonPoint().getLatitude(),
                        selectedItem.getLatLonPoint().getLongitude())));

                aMap.addMarker(new MarkerOptions()
                        .position(new LatLng(selectedItem.getLatLonPoint().getLatitude(), selectedItem.getLatLonPoint().getLongitude()))
                        .icon(BitmapDescriptorFactory.fromBitmap(
                                BitmapFactory.decodeResource(getResources(), R.drawable.location_icon_local)))
                        .draggable(false));

            } else {
                showProgressLayout(true);
            }
            mLocationClient.startLocation();
        }
    }

    public void listItemClick(int position) {
        selectedItem = (PoiItem) adapter.getItem(position);
        getMapScreenShot(null);
        //TODO sunyunlei 如果需要选择确定话,地标需要移动要做以下处理
//        adapter.notifyDataSetChanged();
//        aMap.clear();
//        //设置缩放级别
//        aMap.moveCamera(CameraUpdateFactory.zoomTo(17));
//        //将地图移动到定位点
//        aMap.moveCamera(CameraUpdateFactory.changeLatLng(new LatLng(selectedItem.getLatLonPoint().getLatitude(),
//                selectedItem.getLatLonPoint().getLongitude())));
//
//        aMap.addMarker(new MarkerOptions()
//                .position(new LatLng(selectedItem.getLatLonPoint().getLatitude(), selectedItem.getLatLonPoint().getLongitude()))
//                .icon(BitmapDescriptorFactory.fromBitmap(
//                        BitmapFactory.decodeResource(getResources(), R.drawable.location_icon_local)))
//                .draggable(false));

    }

    /**
     * 对地图进行截屏
     */
    public void getMapScreenShot(View v) {
        aMap.getMapScreenShot(this);
        aMap.invalidate();// 刷新地图
    }


    @Override
    protected void onResume() {
        super.onResume();
        mapView.onResume();
    }

    @Override
    protected void onPause() {
        super.onPause();
        mapView.onPause();
        mLocationClient.stopLocation();
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        mapView.onSaveInstanceState(outState);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        mapView.onDestroy();
        mLocationClient.onDestroy();
    }

    @Override
    public void onMapScreenShot(Bitmap bitmap) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
        if (null == bitmap) {
            return;
        }
        String filepath = Environment.getExternalStorageDirectory() + "/map_"
                + sdf.format(new Date()) + ".amap";
        try {
            FileOutputStream fos = new FileOutputStream(filepath);
            bitmap.compress(Bitmap.CompressFormat.PNG, 100, fos);
            try {
                fos.flush();
            } catch (IOException e) {
                e.printStackTrace();
            }
            try {
                fos.close();
            } catch (IOException e) {
                e.printStackTrace();
            }

        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }

        ImageBean imageBean = new ImageBean();
//        Bitmap bmp = BitmapFactory.decodeFile(filepath);
        imageBean.setMiniMap(BitmapUtils.getMiniMap2(bitmap));
        imageBean.setPath(filepath);

        Intent intent = new Intent();
        intent.putExtra(MAP, imageBean);
        intent.putExtra(ADDRESS, selectedItem.getTitle());
        intent.putExtra(ADDRESS_DETAIL, selectedItem.getSnippet());
        intent.putExtra(LONGITUDE, String.valueOf(selectedItem.getLatLonPoint().getLongitude()));
        intent.putExtra(LATITUDE, String.valueOf(selectedItem.getLatLonPoint().getLatitude()));

        setResult(RESULT_OK, intent);
        finish();
        selectedItem = null;
    }

    public void startLocation() {
        mLocationClient.startLocation();
    }


    public void errorLayoutClick() {
        Intent intent = new Intent();
        intent.setClass(this, LocationActivity.class);
        intent.setFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP);

        if (selectedItem != null) {
            intent.putExtra(ADDRESS, selectedItem.getTitle());
            intent.putExtra(ADDRESS_DETAIL, selectedItem.getSnippet());
            intent.putExtra(LONGITUDE, String.valueOf(selectedItem.getLatLonPoint().getLongitude()));
            intent.putExtra(LATITUDE, String.valueOf(selectedItem.getLatLonPoint().getLatitude()));
        }
        intent.putExtra(FROM_TOPIC_RELEASE, isFromTopicRelease);
        startActivity(intent);
    }


    /**
     * 网络是否可用
     *
     * @return
     */
    public boolean isNetworkAvailable() {
        ConnectivityManager mgr = (ConnectivityManager) this
                .getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo[] info = mgr.getAllNetworkInfo();
        if (info != null) {
            for (int i = 0; i < info.length; i++) {
                if (info[i].getState() == NetworkInfo.State.CONNECTED) {
                    return true;
                }
            }
        }
        return false;
    }

    public void showErrorLayout() {
        errorLayout.setVisibility(View.VISIBLE);
//        mapLayout.setVisibility(View.GONE);
        bottomLayout.setVisibility(View.GONE);
    }

    public void showProgressLayout(boolean show) {
        if (show) {
            progressLayout.setVisibility(View.VISIBLE);
            dialog_progress.setImageResource(R.drawable.location_inner_animation);
            AnimationDrawable animationDrawable = (AnimationDrawable) dialog_progress.getDrawable();
            animationDrawable.start();

            listView.setVisibility(View.GONE);
        } else {
            progressLayout.setVisibility(View.GONE);
            listView.setVisibility(View.VISIBLE);
            dialog_progress.setImageResource(R.drawable.location_inner_animation);
            AnimationDrawable animationDrawable = (AnimationDrawable) dialog_progress.getDrawable();
            animationDrawable.stop();
        }
    }


    private void getData(Intent intent) {
        if (null != intent) {
            isFromTopicRelease = intent.getBooleanExtra(LocationActivity.FROM_TOPIC_RELEASE, false);
            if (isFromTopicRelease) {
                String address = intent.getStringExtra(ADDRESS);
                String addressDetail = intent.getStringExtra(ADDRESS_DETAIL);
                double longitude = intent.getDoubleExtra(LONGITUDE, Double.POSITIVE_INFINITY);
                double latidue = intent.getDoubleExtra(LATITUDE, Double.POSITIVE_INFINITY);
                String citycode = intent.getStringExtra(CITYCODE);
                if (Double.compare(longitude, 0.0) != 0
                        && Double.compare(latidue, 0.0) != 0
                        && !TextUtils.isEmpty(address)) {
                    selectedItem = new PoiItem(null, new LatLonPoint(latidue, longitude), address, addressDetail);
                    selectedItem.setCityCode(citycode);
                }
            }
        }
    }

    /**
     * 初始化AMap对象
     */
    private void initMap() {
        if (aMap == null) {
            aMap = mapView.getMap();
        }

        //目标生成一张1*1像素的图片
        TextView mTextView = new TextView(this);
        LinearLayout.LayoutParams lp = new LinearLayout.LayoutParams(1, 1);
        mTextView.setLayoutParams(lp);
        int widthMeasureSpec = View.MeasureSpec.makeMeasureSpec(1, View.MeasureSpec.EXACTLY);
        int heightMeasureSpec = View.MeasureSpec.makeMeasureSpec(1, View.MeasureSpec.EXACTLY);
        mTextView.measure(widthMeasureSpec, heightMeasureSpec);
        mTextView.layout(0, 0, mTextView.getMeasuredWidth(), mTextView.getMeasuredHeight());
        mTextView.buildDrawingCache();
        Bitmap bitmap = mTextView.getDrawingCache();

        MyLocationStyle myLocationStyle = new MyLocationStyle();
        //定位一次，且将视角移动到地图中心点
        myLocationStyle.myLocationType(MyLocationStyle.LOCATION_TYPE_LOCATE);
        //BitmapDescriptor locationIcon = BitmapDescriptorFactory.fromResource(R.drawable.location_ic_detail_marker);//
        BitmapDescriptor locationIcon = BitmapDescriptorFactory.fromBitmap(bitmap);
        myLocationStyle.myLocationIcon(locationIcon);
        myLocationStyle.radiusFillColor(ContextCompat.getColor(this, android.R.color.transparent));
        myLocationStyle.strokeColor(ContextCompat.getColor(this, android.R.color.transparent));
        //设置定位蓝点的style
        aMap.setMyLocationStyle(myLocationStyle);

        aMap.setLocationSource(this);
        aMap.setOnCameraChangeListener(this);

        //初始化定位
        mLocationClient = new AMapLocationClient(getApplicationContext());
        //设置定位回调监听
        mLocationClient.setLocationListener(this);
        //初始化AMapLocationClientOption对象
        mLocationOption = new AMapLocationClientOption();
        //设置定位模式为AMapLocationMode.Hight_Accuracy，高精度模式。
        mLocationOption.setLocationMode(AMapLocationClientOption.AMapLocationMode.Hight_Accuracy);
        //设置是否返回地址信息（默认返回地址信息）
        mLocationOption.setNeedAddress(true);
        mLocationOption.setOnceLocation(true);
        //设置是否强制刷新WIFI，默认为强制刷新
        mLocationOption.setWifiActiveScan(true);
        mLocationOption.setMockEnable(false);
        //给定位客户端对象设置定位参数
        mLocationClient.setLocationOption(mLocationOption);

        UiSettings settings = aMap.getUiSettings();
        settings.setMyLocationButtonEnabled(false);
        settings.setZoomControlsEnabled(true);
        settings.setLogoPosition(AMapOptions.LOGO_POSITION_BOTTOM_LEFT);
        aMap.setMyLocationEnabled(true);

        geocoderSearch = new GeocodeSearch(this);
        geocoderSearch.setOnGeocodeSearchListener(this);
    }

    @Override
    public void activate(LocationSource.OnLocationChangedListener onLocationChangedListener) {
        mListener = onLocationChangedListener;
    }

    @Override
    public void deactivate() {
        mListener = null;
    }

    /**
     * 坐标转地址
     *
     * @param regeocodeResult
     * @param rCode
     */
    @Override
    public void onRegeocodeSearched(RegeocodeResult regeocodeResult, int rCode) {
        LogUtils.d("onRegeocodeSearched");
        if (rCode == 1000) {
            doSearchQuery("", regeocodeResult.getRegeocodeQuery().getPoint().getLatitude(),
                    regeocodeResult.getRegeocodeQuery().getPoint().getLongitude(),
                    regeocodeResult.getRegeocodeAddress().getCityCode());
        } else {
            showErrorLayout();
            //Toast.makeText(this, "查找坐标点附近位置失败", Toast.LENGTH_SHORT).show();
        }
    }

    /**
     * 地址转坐标
     *
     * @param geocodeResult
     * @param i
     */
    @Override
    public void onGeocodeSearched(GeocodeResult geocodeResult, int i) {
        LogUtils.d("onGeocodeSearched");
    }

    /**
     * 开始进行poi搜索
     */
    protected void doSearchQuery(String key, double latitude, double longitude, String cityCode) {
        showProgressLayout(true);
        currentPage = 0;
        // 第一个参数表示搜索字符串，第二个参数表示poi搜索类型，第三个参数表示poi搜索区域（空字符串代表全国）
        query = new PoiSearch.Query(key, "政府机构及社会团体|公司企业|商务住宅|科教文化服务|金融保险服务|公共设施|生活服务", cityCode);
        query.setPageSize(20);// 设置每页最多返回多少条poiitem
        query.setPageNum(currentPage);// 设置查第一页

        poiSearch = new PoiSearch(this, query);
        poiSearch.setOnPoiSearchListener(this);
        if (TextUtils.isEmpty(key)) {//自动搜索的时候 搜索附近的
            poiSearch.setBound(new PoiSearch.SearchBound(new LatLonPoint(latitude,
                    longitude), 8000));//设置周边搜索的中心点以及半径
        }
        poiSearch.searchPOIAsyn();// 异步搜索
    }

    @Override
    public void onPoiSearched(PoiResult result, int rCode) {
        showProgressLayout(false);
        if (rCode == 1000) {
            if (result != null && result.getQuery() != null) {// 搜索poi的结果
                if (result.getQuery().equals(query)) {// 是否是同一条
                    poiResult = result;
                    // 取得搜索到的poiitems有多少页
                    List<PoiItem> poiItems = poiResult.getPois();// 取得第一页的poiitem数据，页数从数字0开始

                    if (poiItems != null && poiItems.size() > 0) {
                        ((InputMethodManager) getSystemService(INPUT_METHOD_SERVICE))
                                .hideSoftInputFromWindow(getWindow().getDecorView().getWindowToken(),
                                        InputMethodManager.HIDE_NOT_ALWAYS);
                        items = poiItems;
                        //手动搜索地名后 将选中的放到第一位
                        if (null != selectedItem) {
                            for (int i = 0; i < items.size(); i++) {
                                if (items.get(i).getTitle().equals(selectedItem.getTitle())) {
                                    items.remove(i);
                                    items.add(0, selectedItem);
                                    break;
                                }
                            }
                        }
                        adapter.setData(items);
                        adapter.notifyDataSetChanged();
                        selectedItem = items.get(0);
                    }
                } else {
                    Log.d("renjw", "POI 搜索无结果1");
                }
            }
        } else {
            Log.d("renjw", "POI 搜索无结果2");
        }
    }

    @Override
    public void onPoiItemSearched(PoiItem poiItem, int i) {

    }

    @Override
    public void onLocationChanged(AMapLocation aMapLocation) {
        if (null != aMapLocation) {
            aMap.clear();
            currentLocation = aMapLocation;
            //设置缩放级别
            aMap.moveCamera(CameraUpdateFactory.zoomTo(17));
            //将地图移动到定位点
            aMap.moveCamera(CameraUpdateFactory.changeLatLng(new LatLng(currentLocation.getLatitude(), currentLocation.getLongitude())));
            //点击定位按钮 能够将地图的中心移动到定位点
            mListener.onLocationChanged(currentLocation);

            aMap.addMarker(new MarkerOptions()
                    .position(new LatLng(currentLocation.getLatitude(), currentLocation.getLongitude()))
                    .icon(BitmapDescriptorFactory.fromBitmap(
                            BitmapFactory.decodeResource(getResources(), R.drawable.location_icon_local)))
                    .draggable(false));
            if (currentLocation != null) {
                RegeocodeQuery query = new RegeocodeQuery(new LatLonPoint(currentLocation.getLatitude(), currentLocation.getLongitude()), 200, GeocodeSearch.AMAP);
                geocoderSearch.getFromLocationAsyn(query);
            }
        }
    }

    @Override
    public void onCameraChange(CameraPosition cameraPosition) {
//        aMap.clear();
//        aMap.addMarker(new MarkerOptions()
//                .position(cameraPosition.target)
//                .icon(BitmapDescriptorFactory.fromBitmap(
//                        BitmapFactory.decodeResource(getResources(), R.drawable.location_icon_local)))
//                .draggable(false));
    }

    @Override
    public void onCameraChangeFinish(CameraPosition cameraPosition) {
        // 第一个参数表示一个Latlng，第二参数表示范围多少米，第三个参数表示是火系坐标系还是GPS原生坐标系
//        RegeocodeQuery query = null;
//        if (currentLocation == null) {
//            query = new RegeocodeQuery(new LatLonPoint(cameraPosition.target.latitude, cameraPosition.target.longitude), 200, GeocodeSearch.AMAP);
//        } else {
//            query = new RegeocodeQuery(new LatLonPoint(currentLocation.getLatitude(), currentLocation.getLongitude()), 200, GeocodeSearch.AMAP);
//        }
//        geocoderSearch.getFromLocationAsyn(query);
    }

    /**
     * 点击搜索框以外的地方  隐藏键盘
     *
     * @param event
     * @return
     */
    @Override
    public boolean dispatchTouchEvent(MotionEvent event) {
        if (event.getAction() == MotionEvent.ACTION_DOWN
                || event.getAction() == MotionEvent.ACTION_MOVE) {
            View v = getCurrentFocus();
            if (v instanceof EditText) {
                Rect outRect = new Rect();
                v.getGlobalVisibleRect(outRect);
                if (!outRect.contains((int) event.getRawX(), (int) event.getRawY())) {
                    v.clearFocus();
                    ((InputMethodManager) getSystemService(INPUT_METHOD_SERVICE))
                            .hideSoftInputFromWindow(v.getWindowToken(),
                                    InputMethodManager.HIDE_NOT_ALWAYS);
                }
            }
        }
        return super.dispatchTouchEvent(event);
    }

    @Override
    public void onPointerCaptureChanged(boolean hasCapture) {

    }
}
