package com.xdja.location;

import android.content.Context;
import android.content.Intent;
import android.graphics.BitmapFactory;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v7.app.AppCompatActivity;
import android.util.Log;
import android.widget.TextView;

import com.amap.api.location.AMapLocationClient;
import com.amap.api.location.AMapLocationClientOption;
import com.amap.api.maps2d.AMap;
import com.amap.api.maps2d.CameraUpdateFactory;
import com.amap.api.maps2d.UiSettings;
import com.amap.api.maps2d.model.BitmapDescriptorFactory;
import com.amap.api.maps2d.model.LatLng;
import com.amap.api.maps2d.model.MarkerOptions;
import com.amap.api.services.core.LatLonPoint;
import com.amap.api.services.geocoder.GeocodeAddress;
import com.amap.api.services.geocoder.GeocodeQuery;
import com.amap.api.services.geocoder.GeocodeResult;
import com.amap.api.services.geocoder.GeocodeSearch;
import com.amap.api.services.geocoder.RegeocodeQuery;
import com.amap.api.services.geocoder.RegeocodeResult;
import com.xdja.eoa.commonutil.ArraysUtils;
import com.xdja.eoa.commonutil.StatusBarUtil;
import com.xdja.eoa.commonwidget.widget.CommonToolBar;
import com.xdja.location.params.LocationDetailParams;

public class LocationDetailActivity extends AppCompatActivity implements GeocodeSearch.OnGeocodeSearchListener {

    public static final String PARAM_LOCATION_DETAIL = "params_location_detail";
    private LocationDetailParams detailParams;

    private CommonToolBar toolbarLocationDetail;
    private EoaMapView mapView;
    private TextView tvLocationTitle;
    private TextView tvLocationDesc;

    private AMap aMap;
    public AMapLocationClient mLocationClient = null;
    public AMapLocationClientOption mLocationOption = null;
    private GeocodeSearch geocodeSearch;//地理编码

    public static void start(Context context, LocationDetailParams params) {
        Intent starter = new Intent(context, LocationDetailActivity.class);
        starter.putExtra(PARAM_LOCATION_DETAIL, params);
        context.startActivity(starter);
    }

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.location_activity_location_detail);
        if (getIntent() == null || !getIntent().hasExtra(PARAM_LOCATION_DETAIL)) {
            finish();
            return;
        }
        initView(savedInstanceState);
        initMap();
        searchLocationData();
    }

    private void initView(Bundle savedInstanceState) {
        toolbarLocationDetail = (CommonToolBar) findViewById(R.id.toolbar_location_detail);
        mapView = (EoaMapView) findViewById(R.id.view_map);
        mapView.onCreate(savedInstanceState);
        tvLocationTitle = (TextView) findViewById(R.id.tv_location_title);
        tvLocationDesc = (TextView) findViewById(R.id.tv_location_desc);

        StatusBarUtil.immersive(this);
        StatusBarUtil.setPaddingSmart(this, toolbarLocationDetail);
        StatusBarUtil.darkMode(this);
    }

    private void initMap() {
        if (aMap == null) {
            aMap = mapView.getMap();
        }
        //初始化定位
        mLocationClient = new AMapLocationClient(getApplicationContext());
        //初始化AMapLocationClientOption对象
        mLocationOption = new AMapLocationClientOption();
        //设置定位模式为AMapLocationMode.Hight_Accuracy，高精度模式。
        mLocationOption.setLocationMode(AMapLocationClientOption.AMapLocationMode.Hight_Accuracy);
        //设置是否返回地址信息（默认返回地址信息）
        mLocationOption.setNeedAddress(true);
        mLocationOption.setOnceLocation(true);
        //设置是否强制刷新WIFI，默认为强制刷新
        mLocationOption.setWifiActiveScan(true);
        mLocationOption.setMockEnable(false);
        //给定位客户端对象设置定位参数
        mLocationClient.setLocationOption(mLocationOption);

        UiSettings settings = aMap.getUiSettings();
        settings.setMyLocationButtonEnabled(false);
        settings.setZoomControlsEnabled(false);
        aMap.setMyLocationEnabled(true);
    }

    private void searchLocationData() {
        detailParams = getIntent().getParcelableExtra(PARAM_LOCATION_DETAIL);
        geocodeSearch = new GeocodeSearch(this);
        geocodeSearch.setOnGeocodeSearchListener(this);
        if (detailParams.latitude <= 0 || detailParams.longitude <= 0) {
            Log.w("location", "latitude : " + detailParams.latitude + "  longitude : " + detailParams.longitude);
            searchByName(detailParams.desc);
        } else {
            LatLonPoint latLonPoint = new LatLonPoint(detailParams.latitude, detailParams.longitude);
            searchByLatLon(latLonPoint);
            initMapLocation(detailParams.latitude, detailParams.longitude);
        }
        tvLocationTitle.setText(detailParams.getDesc());
        tvLocationDesc.setText(detailParams.getDesc());
    }

    private void searchByLatLon(LatLonPoint latLonPoint) {
        RegeocodeQuery query = new RegeocodeQuery(latLonPoint, 200, GeocodeSearch.AMAP);
        geocodeSearch.getFromLocationAsyn(query);
    }

    private void searchByName(String name) {
        GeocodeQuery geocodeQuery = new GeocodeQuery(name.trim(), "");
        geocodeSearch.getFromLocationNameAsyn(geocodeQuery);
    }

    private void initMapLocation(double lat, double lng) {
        //设置缩放级别
        aMap.moveCamera(CameraUpdateFactory.zoomTo(18));
        //将地图移动到定位点
        aMap.moveCamera(CameraUpdateFactory.changeLatLng(new LatLng(lat,
                lng)));
        aMap.addMarker(new MarkerOptions()
                .position(new LatLng(lat, lng))
                .icon(BitmapDescriptorFactory.fromBitmap(
                        BitmapFactory.decodeResource(getResources(), R.drawable.location_ic_detail_marker)))
                .draggable(false));
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        mapView.onSaveInstanceState(outState);
    }

    @Override
    protected void onResume() {
        super.onResume();
        mapView.onResume();
    }

    @Override
    protected void onPause() {
        super.onPause();
        mapView.onPause();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        mapView.onDestroy();
    }

    @Override
    public void onRegeocodeSearched(RegeocodeResult regeocodeResult, int rCode) {
        if (rCode != 1000) {
            tvLocationDesc.setText(detailParams.getDesc());
            return;
        }
        tvLocationDesc.setText(regeocodeResult.getRegeocodeAddress().getFormatAddress());
    }

    @Override
    public void onGeocodeSearched(GeocodeResult geocodeResult, int rCode) {
        if (rCode != 1000) {
            tvLocationDesc.setText(detailParams.getDesc());
            return;
        }

        if (geocodeResult == null || ArraysUtils.isListEmpty(geocodeResult.getGeocodeAddressList())) {
            tvLocationDesc.setText(detailParams.getDesc());
            return;
        }
        GeocodeAddress geocodeAddress = geocodeResult.getGeocodeAddressList().get(0);
        tvLocationDesc.setText(geocodeAddress.getFormatAddress());
        initMapLocation(geocodeAddress.getLatLonPoint().getLatitude(), geocodeAddress.getLatLonPoint().getLongitude());
    }
}
