package com.xdja.framework.commons.utils.http;

import com.alibaba.fastjson.JSON;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.EntityBuilder;
import org.apache.http.client.methods.*;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContextBuilder;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.DefaultProxyRoutePlanner;
import org.apache.http.message.BasicNameValuePair;

import javax.net.ssl.SSLContext;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * HttpClient请求封装工具类
 * 依赖
 * <pre>
 *     <dependency>
 *       <groupId>org.apache.httpcomponents</groupId>
 *       <artifactId>httpmime</artifactId>
 *       <version>4.3.3</version>
 *     </dependency>
 *
 *     <dependency>
 *       <groupId>com.alibaba</groupId>
 *       <artifactId>fastjson</artifactId>
 *       <version>1.2.13</version>
 *     </dependency>
 * </pre>
 * 使用示例
 * <pre>
 *     HttpUtils.post("http://www.xdja.com/q")
 *              .addJson(query)
 *              .addHeader("Authorization", "8d92cfe6b0b411e8acec80e650218f00")
 *              .execute()
 *              .getString();
 * </pre>
 *
 * @author hsun
 * @version 1.0
 * @since 2018/8/14 下午4:23
 */
public class HttpUtils {

    /**
     * 是否HTTPS
     */
    private boolean isHttps;

    /**
     * 请求类型
     * 1-POST 2-GET 3-PUT 4-DELETE
     */
    private int type;

    /**
     * 请求对象
     */
    private HttpRequestBase request;

    /**
     * httpClient实例
     */
    private CloseableHttpClient httpClient;

    /**
     * post, put请求的参数
     */
    private EntityBuilder builder;
    /**
     * get, delete请求的参数
     */
    private URIBuilder uriBuilder;
    /**
     * 连接工厂
     */
    private LayeredConnectionSocketFactory socketFactory;
    /**
     * 构建httpclient
     */
    private HttpClientBuilder clientBuilder;
    /**
     * 请求的相关配置
     */
    private RequestConfig.Builder config;

    /**
     * 私有构造
     * @param request
     */
    private HttpUtils(HttpRequestBase request) {
        this.isHttps = request.getURI().getScheme().equalsIgnoreCase("https");
        this.request = request;
        this.clientBuilder = HttpClientBuilder.create();
        this.config = RequestConfig.custom().setCookieSpec(CookieSpecs.BROWSER_COMPATIBILITY);

        if (request instanceof HttpPost) {
            this.type = 1;
            builder = EntityBuilder.create().setParameters(new ArrayList<NameValuePair>());
        }

        if (request instanceof HttpGet) {
            this.type = 2;
            uriBuilder = new URIBuilder();
        }

        if (request instanceof HttpPut) {
            this.type = 3;
            builder = EntityBuilder.create().setParameters(new ArrayList<NameValuePair>());
        }

        if (request instanceof HttpDelete) {
            this.type = 4;
            uriBuilder = new URIBuilder();
        }

    }

    /**
     * 创建HttpUtils
     * @param request
     * @return
     */
    private static HttpUtils create(HttpRequestBase request) {
        return new HttpUtils(request);
    }

    /**
     * 构建POST请求
     * @param url
     * @return
     */
    public static HttpUtils post(String url) {
        return create(new HttpPost(url));
    }

    /**
     * 构建POST请求
     * @param uri
     * @return
     */
    public static HttpUtils post(URI uri) {
        return create(new HttpPost(uri));
    }

    /**
     * 构建GET请求
     * @param url
     * @return
     */
    public static HttpUtils get(String url) {
        return create(new HttpGet(url));
    }

    /**
     * 构建GET请求
     * @param uri
     * @return
     */
    public static HttpUtils get(URI uri) {
        return create(new HttpGet(uri));
    }

    /**
     * 构建PUT请求
     * @param url
     * @return
     */
    public static HttpUtils put(String url) {
        return create(new HttpPut(url));
    }

    /**
     * 构建PUT请求
     * @param uri
     * @return
     */
    public static HttpUtils put(URI uri) {
        return create(new HttpPut(uri));
    }

    /**
     * 构建DELETE请求
     * @param url
     * @return
     */
    public static HttpUtils delete(String url) {
        return create(new HttpDelete(url));
    }

    /**
     * 构建DELETE请求
     * @param uri
     * @return
     */
    public static HttpUtils delete(URI uri) {
        return create(new HttpDelete(uri));
    }

    /**
     * 添加请求参数
     * @param name 键
     * @param value 值
     * @return
     */
    public HttpUtils addParameter(final String name, final String value) {
        if (builder != null) {
            builder.getParameters().add(new BasicNameValuePair(name, value));
        } else {
            uriBuilder.addParameter(name, value);
        }
        return this;
    }

    /**
     * 添加请求参数
     * @param parameters 参数map集合
     * @return
     */
    public HttpUtils addParameters(Map<String, String> parameters) {
        List<NameValuePair> values = new ArrayList<>(parameters.size());

        for (Map.Entry<String, String> parameter : parameters.entrySet()) {
            values.add(new BasicNameValuePair(parameter.getKey(), parameter.getValue()));
        }

        if(builder != null) {
            builder.getParameters().addAll(values);
        } else {
            uriBuilder.addParameters(values);
        }
        return this;
    }

     /**
     * 添加请求参数
     * @param parameters
     * @return
     */
    public HttpUtils addParameters(final NameValuePair ...parameters) {
        if (builder != null) {
            builder.getParameters().addAll(Arrays.asList(parameters));
        } else {
            uriBuilder.addParameters(Arrays.asList(parameters));
        }
        return this;
    }

    /**
     * 添加请求体
     * @param bytes
     * @return
     */
    public HttpUtils addBytes(byte[] bytes) {
        builder.setContentType(ContentType.APPLICATION_OCTET_STREAM);
        builder.setBinary(bytes);
        return this;
    }

    /**
     * 设置请求参数
     * 会清除之前的参数
     * @param parameters Map集合
     * @return
     */
    public HttpUtils setParameters(final Map<String, String> parameters) {
        NameValuePair [] values = new NameValuePair[parameters.size()];

        int i = 0;
        for (Map.Entry<String, String> parameter : parameters.entrySet()) {
            values[i++] = new BasicNameValuePair(parameter.getKey(), parameter.getValue());
        }
        setParameters(values);
        return this;
    }

    /**
     * 设置请求参数
     * 会清除之前的参数
     * @param parameters
     * @return
     */
    public HttpUtils setParameters(NameValuePair ... parameters) {
        if (builder != null) {
            builder.setParameters(parameters);
        } else {
            uriBuilder.setParameters(parameters);
        }
        return this;
    }

    /**
     * 设置字节组
     * 会清除之前的参数
     * @param binary
     * @return
     */
    public HttpUtils setBytes(final byte[] binary) {
        if(builder != null) {
            builder.setBinary(binary);
        } else {
            throw new UnsupportedOperationException();
        }
        return this;
    }

    /**
     * 设置文件
     * 会清除之前的参数
     * @param file 文件对象
     * @return
     */
    public HttpUtils setFile(final File file) {
        if(builder != null) {
            builder.setFile(file);
        } else {
            throw new UnsupportedOperationException();
        }
        return this;
    }

    /**
     * 设置输入流
     * 会清除之前的参数
     * @param is 输入流
     * @return
     */
    public HttpUtils setStream(final InputStream is) {
        if(builder != null) {
            builder.setStream(is);
        } else {
            throw new UnsupportedOperationException();
        }
        return this;
    }

    /**
     * 添加JSON参数
     * 会清除之前的参数
     * @param object 对象
     * @return
     */
    public HttpUtils addJson(Object object) {
        if (builder != null) {
            builder.setContentType(ContentType.APPLICATION_JSON);
            builder.setBinary(JSON.toJSONString(object).getBytes());
        } else {
            throw new UnsupportedOperationException();
        }
        return this;
    }

    /**
     * 添加请求头信息
     * @param name
     * @param value
     * @return
     */
    public HttpUtils addHeader(String name, String value) {
        request.addHeader(name, value);
        return this;
    }

    /**
     * 添加请求头信息
     * @param headers
     * @return
     */
    public HttpUtils addHeader(Map<String, String> headers) {
        for (Map.Entry<String, String>  header : headers.entrySet()) {
            request.addHeader(header.getKey(), header.getValue());
        }
        return this;
    }

    /**
     * 设置网络代理
     * @param hostname
     * @param port
     * @return
     */
    public HttpUtils setProxy(String hostname, int port) {
        HttpHost httpHost = new HttpHost(hostname, port);
        return setProxy(httpHost);
    }

    /**
     * 设置网络代理
     * @param host
     * @return
     */
    private HttpUtils setProxy(HttpHost host) {
        DefaultProxyRoutePlanner routePlanner = new DefaultProxyRoutePlanner(host);
        clientBuilder.setRoutePlanner(routePlanner);
        return this;
    }

    /**
     * 设置双向认证的JKS
     * @param jksFilePath
     * @param password
     * @return
     */
    public HttpUtils setJKS(String jksFilePath, String password) {
        return setJKS(new File(jksFilePath), password);
    }

    /**
     * 设置双向认证的JKS
     * @param jksFile
     * @param password
     * @return
     */
    public HttpUtils setJKS(File jksFile, String password) {
        try (InputStream instream = new FileInputStream(jksFile)) {
            return setJKS(instream, password);
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 设置双向认证的JKS
     * @param is
     * @param password
     * @return
     */
    public HttpUtils setJKS(InputStream is, String password) {
        try {
            KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            keyStore.load(is, password.toCharArray());
            return setJKS(keyStore, password);
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 设置双向认证的JKS
     * @param keyStore
     * @param password
     * @return
     */
    public HttpUtils setJKS(KeyStore keyStore, String password) {
        try {
            SSLContext sslContext = SSLContexts.custom().useTLS().loadKeyMaterial(keyStore, password.toCharArray())
                    .loadTrustMaterial(keyStore).build();
            socketFactory = new SSLConnectionSocketFactory(sslContext, SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage(), e);
        }

        return this;
    }

    /**
     * 设置内容编码
     * @param encoding
     * @return
     */
    public HttpUtils setContentEncoding(final String encoding) {
        if(builder != null) {
            builder.setContentEncoding(encoding);
        }
        return this;
    }

    /**
     * 设置ContentType
     * @param contentType
     * @return
     */
    public HttpUtils setContentType(ContentType contentType) {
        if(builder != null) {
            builder.setContentType(contentType);
        }
        return this;
    }

    /**
     * 设置ContentType
     * @param mimeType
     * @param charset 内容编码
     * @return
     */
    public HttpUtils setContentType(final String mimeType, final Charset charset) {
        if(builder != null) {
            builder.setContentType(ContentType.create(mimeType, charset));
        }
        return this;
    }

    /**
     * 设置Socket超时时间,单位:ms
     * @param socketTimeout
     * @return
     */
    public HttpUtils setSocketTimeout(int socketTimeout){
        config.setSocketTimeout(socketTimeout);
        return this;
    }

    /**
     * 设置连接超时时间,单位:ms
     * @param connectTimeout
     * @return
     */
    public HttpUtils setConnectTimeout(int connectTimeout) {
        config.setConnectTimeout(connectTimeout);
        return this;
    }

    /**
     * 设置请求超时时间,单位:ms
     * @param connectionRequestTimeout
     * @return
     */
    public HttpUtils setConnectionRequestTimeout(int connectionRequestTimeout) {
        config.setConnectionRequestTimeout(connectionRequestTimeout);
        return this;
    }

    /**
     * 发送请求
     * @return
     */
    public ResponseWrap execute() {
        return execute(null);
    }

    /**
     * 发送请求
     * @param multiBuilder
     * @return
     */
    public ResponseWrap execute(MultipartEntityBuilder multiBuilder) {
        settingRequest(multiBuilder);
        if(null == httpClient ) {
            httpClient = clientBuilder.build();
        }

        try {
            HttpClientContext context = HttpClientContext.create();
            CloseableHttpResponse response = httpClient.execute(request, context);
            return new ResponseWrap(response);
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 设置请求体
     * @param multiBuilder
     */
    private void settingRequest(MultipartEntityBuilder multiBuilder) {
        URI uri = null;
        if(uriBuilder != null && uriBuilder.getQueryParams().size() != 0) {
            try {
                uri = uriBuilder.setPath(request.getURI().toString()).build();
            } catch (URISyntaxException e) {
                throw new RuntimeException(e.getMessage(), e);
            }
        }

        HttpEntity httpEntity = null;

        switch (type) {
            case 1:
                httpEntity = getEntityBuilder(multiBuilder);
                if(httpEntity.getContentLength() > 0) ((HttpPost)request).setEntity(httpEntity);
                break;

            case 2:
                HttpGet get = ((HttpGet)request);
                if (uri != null)  get.setURI(uri);
                break;

            case 3:
                httpEntity = getEntityBuilder(multiBuilder);
                if(httpEntity.getContentLength() > 0) ((HttpPut)request).setEntity(httpEntity);
                break;

            case 4:
                HttpDelete delete = ((HttpDelete)request);
                if (uri != null) delete.setURI(uri);
                break;
        }

        // HTTPS 请求
        if (isHttps && null != socketFactory) {
             clientBuilder.setSSLSocketFactory(socketFactory);
        } else if (isHttps)  {
            clientBuilder.setSSLSocketFactory(getSSLSocketFactory());
        }

        request.setConfig(config.build());
    }

    /**
     * 获取请求实体
     * @param multiBuilder
     * @return
     */
    private HttpEntity getEntityBuilder(MultipartEntityBuilder multiBuilder) {
        HttpEntity httpEntity;
        if(multiBuilder != null) {
            httpEntity = multiBuilder.build();
        } else {
            httpEntity = builder.build();
        }
        return httpEntity;
    }

    /**
     * 获取连接工厂 使用SSL单向认证
     * @return
     */
    private LayeredConnectionSocketFactory getSSLSocketFactory() {
        try {
            SSLContext sslContext = new SSLContextBuilder().loadTrustMaterial(null, new TrustStrategy() {
                // 信任所有
                public boolean isTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                    return true;
                }
            }).build();

            SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext, SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
            return sslsf;
        } catch (KeyManagementException | NoSuchAlgorithmException | KeyStoreException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 创建一个表单模拟浏览器提交
     * @return
     */
    public FormEntity newForm() {
        return new FormEntity(this);
    }

    /**
     * 表单对象
     */
    public class FormEntity {
        private MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        private HttpUtils httpUtils;

        public FormEntity(HttpUtils httpUtils) {
            this.httpUtils = httpUtils;
            //模拟一个浏览器
            builder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
            builder.setCharset(Charset.forName("UTF-8"));
        }

        /**
         * 添加参数
         * @param name
         * @param value
         * @return
         */
        public FormEntity addParamter(String name, String value) {
            builder.addTextBody(name, value, ContentType.TEXT_PLAIN.withCharset(Charset.forName("UTF-8")));
            return this;
        }

        /**
         * 添加参数
         * @param params
         * @return
         */
        public FormEntity addParamters(Map<String, String> params) {
            if (!params.isEmpty()) {
                for (Map.Entry<String, String> parameter: params.entrySet()) {
                    this.addParamter(parameter.getKey(), parameter.getValue());
                }
            }
            return this;
        }

        /**
         * 添加参数
         * @param name
         * @param bytes
         * @return
         */
        public FormEntity addParamter(String name, byte[] bytes) {
            builder.addBinaryBody(name, bytes);
            return this;
        }

        /**
         *
         * @param name  参数名称
         * @param bytes 参数值
         * @param contentType   参数类型
         * @param filename  文件名
         * @return
         */
        public FormEntity addParamter(String name, byte[] bytes, ContentType contentType, String filename) {
            builder.addBinaryBody(name, bytes, contentType, filename);
            return this;
        }

        public FormEntity setBoundary(String boundary) {
            builder.setBoundary(boundary);
            return this;
        }

        /**
         * 执行请求
         * @return
         */
        public ResponseWrap execute() {
            return httpUtils.execute(builder);
        }
    }
}