package com.xdja.framework.commons.utils.id;

import com.xdja.framework.commons.utils.DigestUtil;
import com.xdja.framework.commons.utils.RandomUtil;
import com.xdja.framework.commons.utils.UUIDUtil;

/**
 * ID生成器
 *
 * 支持UUID:IDGenerator.UUID.generate()
 * 随机串:IDGenerator.RANDOM.generate()
 * MD5(uuid()+random()):IDGenerator.MD5.generate()
 * twitter雪花算法生产（分布式自增唯一）:IDGenerator.SNOW_FLAKE.generate()
 * @author hsun
 * @version 1.0
 * @since 2018/9/3 下午2:24
 */
public interface IDGenerator<T> {

    T generate();

    /**
     * UUID
     */
    IDGenerator<String> UUID = new IDGenerator<String>() {
        @Override
        public String generate() {
            return UUIDUtil.random();
        }
    };

    /**
     * 随机串
     */
    IDGenerator<String> RANDOM = new IDGenerator<String>() {
        @Override
        public String generate() {
            return RandomUtil.randomChar();
        }
    };


    /**
     * MD5
     * md5(uuid()+random())
     */
    IDGenerator<String> MD5 = new IDGenerator<String>() {
        @Override
        public String generate() {
            return DigestUtil.md5Digest(UUIDUtil.random().concat(RandomUtil.randomChar()));
        }
    };

    /**
     * 雪花算法
     */
    IDGenerator<Long> SNOW_FLAKE = new IDGenerator<Long>() {
        @Override
        public Long generate() {
            return SnowflakeIdGenerator.getInstance().nextId();
        }
    };

    /**
     * 雪花算法String
     */
    IDGenerator<String> SNOW_FLAKE_STR = new IDGenerator<String>() {
        @Override
        public String generate() {
            return String.valueOf(SnowflakeIdGenerator.getInstance().nextId());
        }
    };
}
