package com.xdja.framework.commons.utils.id;

import java.util.Random;

/**
 * twitter雪花算法ID生成器
 * https://github.com/twitter/snowflake
 * 根据scala修改
 *
 * @author hsun
 * @version 1.0
 * @since 2018/9/3 下午2:35
 */
public class SnowflakeIdGenerator {

    /**
     * 机器ID
     */
    private long workerId;

    /**
     * 数据标识ID
     */
    private long dataCenterId;

    /**
     * 序列
     */
    private long sequence = 0L;

    /**
     * 开始时间戳 2015-09-07 12:00:00
     */
    private long twepoch = 1441598400000L;

    /**
     * 机器ID所占的位数
     */
    private long workerIdBits = 5L;

    /**
     * 数据标识ID所占的位数
     */
    private long datacenterIdBits = 5L;

    /**
     * 序列在ID中的位数
     */
    private long sequenceBits = 12L;

    /**
     * 最大机器ID
     * 31
     */
    private long maxWorkerId = -1L ^ (-1L << workerIdBits);

    /**
     * 最大数据标识ID
     * 31
     */
    private long maxDataCenterId  = -1L ^ (-1L << datacenterIdBits);

    //每一部分向左的位移
    private long workerIdShift = sequenceBits;
    private long datacenterIdShift  = sequenceBits + workerIdBits;
    private long timestampLeftShift = sequenceBits + workerIdBits + datacenterIdBits;

    /**
     * 生成序列的掩码
     */
    private long sequenceMask = -1L ^ (-1L << sequenceBits);

    /**
     * 上次生成ID的时间戳
     */
    private long lastTimestamp = -1L;

    private static final SnowflakeIdGenerator generator;

    static {
        Random random = new Random();
        //尝试从系统变量中获取
        long workerId = Long.getLong("id-worker", random.nextInt(31));
        long dataCenterId = Long.getLong("id-datacenter", random.nextInt(31));
        generator = new SnowflakeIdGenerator(workerId, dataCenterId);
    }

    public static SnowflakeIdGenerator getInstance() {
        return generator;
    }

    private SnowflakeIdGenerator(long workerId, long dataCenterId) {
        if (workerId > maxWorkerId || workerId < 0) {
            throw new IllegalArgumentException(String.format("worker Id can't be greater than %d or less than 0", maxWorkerId));
        }
        if (dataCenterId > maxDataCenterId || dataCenterId < 0) {
            throw new IllegalArgumentException(String.format("datacenter Id can't be greater than %d or less than 0", maxDataCenterId));
        }
        this.workerId = workerId;
        this.dataCenterId = dataCenterId;
    }

    /**
     * 获得下一个ID (该方法是线程安全的)
     * @return
     */
    public synchronized long nextId() {
        long timestamp = timeGen();

        //如果当前时间小于上一次ID生成的时间戳，说明系统时钟回退过
        if (timestamp < lastTimestamp) {
            throw new UnsupportedOperationException(String.format("Clock moved backwards.  Refusing to generate id for %d milliseconds", lastTimestamp - timestamp));
        }

        //如果是同一时间生成的，则进行毫秒内序列
        if (timestamp == lastTimestamp ) {
            sequence = (sequence + 1) & sequenceMask;
            //毫秒内序列溢出
            if (sequence == 0) {
                //阻塞到下一个毫秒,获得新的时间戳
                timestamp = waitNextMillis(timestamp);
            }
        } else {
            //时间戳改变，毫秒内序列重置
            sequence = 0L;
        }

        //上次生成ID的时间截
        lastTimestamp = timestamp;

        //移位并通过或运算拼到一起组成64位的ID
        return ((timestamp - twepoch) << timestampLeftShift) |  //时间戳部分
                (dataCenterId << datacenterIdShift) |           //数据中心部分
                (workerId << workerIdShift) |                   //机器标识部分
                sequence;                                       //序列号部分
    }

    /**
     * 阻塞到下一个毫秒，直到获得新的时间戳
     * @param lastTimestamp
     * @return
     */
    private long waitNextMillis(long lastTimestamp) {
        long timestamp = timeGen();
        while (timestamp <= lastTimestamp) {
            timestamp = timeGen();
        }
        return timestamp;
    }

    /**
     * 返回以毫秒为单位的当前时间
     * @return
     */
    private long timeGen() {
        return System.currentTimeMillis();
    }

}
