package com.xdja.framework.validation;


import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.xdja.framework.validation.metadata.AnnotationMetadata;
import com.xdja.framework.validation.metadata.ClassMetadata;
import com.xdja.framework.validation.metadata.FieldMetadata;
import com.xdja.framework.validation.utils.ClassMetadataHelper;
import com.xdja.framework.validation.utils.FieldUtils;
import com.xdja.framework.validation.utils.ObjectUtils;
import com.xdja.framework.validation.validator.ValidationContext;
import com.xdja.framework.validation.validator.ValidationException;
import com.xdja.framework.validation.validator.ValidationResult;
import com.xdja.framework.validation.validator.Validator;
import com.xdja.framework.validation.validator.message.MessageInterpolatorFactory;

public class ValidatorUtils {
	
	/**
	 * 校验器缓存
	 */
	@SuppressWarnings("rawtypes")
	private static ConcurrentHashMap<Class<? extends Validator>, Validator> CLASS_VALIDATOR_MAP = new ConcurrentHashMap<Class<? extends Validator>, Validator>();
	/**
	 * 类元数据缓存
	 */
	private static ConcurrentHashMap<Class<?>, ClassMetadata> CLASS_METADATA_MAP = new ConcurrentHashMap<Class<?>, ClassMetadata>();
	/**
	 * 默认失败策略
	 */
	private static boolean DEFAULT_FAIL_FAST = false;
	
	/**
	 * 验证指定对象，可指定分组
	 * @param bean 待校验对象
	 * @param failFast 快速失败，短路失败标记
	 * @param groups 校验分组
	 * @return
	 */
	public static ValidationResult validate(Object bean, String... groups) {
		return validate(bean, DEFAULT_FAIL_FAST, groups);
	}
	
	/**
	 * <p>验证指定对象，可指定分组。校验对象需满足以下情况：</p>
	 * <li>JavaBean</li>
	 * <li>实现Collection接口的集合对象</li>
	 * <li>数组对象</li>
	 * 
	 * 若校验对象是基本类型，或Map，校验结果为true。
	 * 
	 * @param bean 待校验对象
	 * @param failFast 快速失败，短路失败标记
	 * @param groups 校验分组
	 * @return
	 */
	public static ValidationResult validate(Object bean, boolean failFast, String... groups) {
		if (bean == null) {
			throw new IllegalArgumentException("验证对象不可为空");
		}
		cacheClassMetadata(bean);
		//已经做过校验的值，仅针对引用的字段类型，目的是为了解决对象重复引用。若存在重复引用，则直执行一次校验。
		List<Object> validatedValues = new ArrayList<Object>();
		ValidationResult result = new ValidationResult();
		validateBean(bean, failFast, validatedValues, result, groups);
		validatedValues = null;
		return result;
	}
	
	/**
	 * 缓存类元数据，若bean为集合或数组，则获取元素的类元数据。
	 * @param bean
	 */
	private static void cacheClassMetadata(Object bean) {
		ClassMetadata classMetadata = CLASS_METADATA_MAP.get(bean.getClass());
		if (classMetadata != null) {
			return;
		}
		classMetadata = FieldUtils.filterFields(bean);
		CLASS_METADATA_MAP.putIfAbsent(classMetadata.getCls(), classMetadata);
		List<FieldMetadata> fieldMetadatas = classMetadata.getFieldMetadatas();
		if (fieldMetadatas != null && !fieldMetadatas.isEmpty()) {
			for (FieldMetadata fieldMetadata : fieldMetadatas) {
				ClassMetadata reference = fieldMetadata.getReference();
				if (reference != null) {
					CLASS_METADATA_MAP.putIfAbsent(reference.getCls(), reference);
				}
			}
		}
	}
	
	/**
	 * 获取类元数据
	 * @param bean
	 * @return
	 */
	private static ClassMetadata getClassMetadata(Object bean) {
		//获取对象元素的类
		Class<?> realClass = ObjectUtils.getElementClass(bean);
		return CLASS_METADATA_MAP.get(realClass);
	}

	@SuppressWarnings({ "rawtypes" })
	private static boolean validateBean(Object bean, boolean failFast, List<Object> validatedValues, ValidationResult result, String... groups) {
		// 检测field是否存在
		try {
			if (validatedValues.contains(bean)) {
				//如果该值已经校验过，则直接返回成功
				return true;
			}else{
				//标记已经进行过校验
				validatedValues.add(bean);
			}
			ClassMetadata classMetadata = getClassMetadata(bean);
			
			classMetadata = ClassMetadataHelper.loadByGroups(classMetadata, groups);
			//执行类级别校验
			executeClassValidation(bean, failFast, result, classMetadata, groups);
			
			//以下进行成员属性校验
			List<FieldMetadata> fieldMetadatas = classMetadata.getFieldMetadatas();
			if (fieldMetadatas != null && !fieldMetadatas.isEmpty()) {
				//集合或数组进行循环校验
				Iterator iterator = ObjectUtils.toIter(bean);
				if (iterator == null) {
					executeFieldValidation(bean, failFast, validatedValues, result, classMetadata);
					
				}else{
					while (iterator.hasNext()) {
						Object object = iterator.next();
						executeFieldValidation(object, failFast, validatedValues, result, classMetadata);
					}
				}
				
			}
			
		} catch (Exception e) {
			throw new ValidationException("验证对象时出错，验证对象为" + bean, e);
		}
		
		return true;
	}

	/**
	 * 执行类级别校验
	 * @param bean 字段值
	 * @param failFast 快速失败标记
	 * @param result 校验结果
	 * @param classMetadata 类元数据
	 * @param groups 分组名称
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private static void executeClassValidation(Object bean, boolean failFast, ValidationResult result, ClassMetadata classMetadata, String... groups) {
		boolean groupCheck = groups != null && groups.length > 0;
		
		Map<String, Class<? extends Validator>> groupClassValidatorMap = classMetadata.getGroupClassValidatorMap();
		
		//以下执行的类级别的校验
		if (groupCheck && groupClassValidatorMap != null && !groupClassValidatorMap.isEmpty()) {
			for (String group : groups) {
				
				Class<? extends Validator> validatorClass = groupClassValidatorMap.get(group);
				ValidationContext context = new ValidationContext();
				context.setCls(classMetadata.getCls());
				context.setResult(result);
				
				Validator validator = getValidator(validatorClass);
				
				//集合或数组进行循环校验
				Iterator iterator = ObjectUtils.toIter(bean);
				if (iterator == null) {
					if(validator.accept(context, bean)){
						//同一字段有多个验证约束，若有一个失败，则终止
						if(!validator.validate(context, bean)){
							interpolateMessage(result, context);
							if (failFast) {
								break;
							}
						}
					}
				}else{
					while (iterator.hasNext()) {
						Object object = iterator.next();
						if(validator.accept(context, object)){
							//同一字段有多个验证约束，若有一个失败，则终止
							if(!validator.validate(context, object)){
								interpolateMessage(result, context);
								if (failFast) {
									break;
								}
							}
						}
					}
				}
				
				
			}
			
		}
	}

	/**
	 * 执行字段校验
	 * @param bean 字段值
	 * @param failFast 快速失败标记
	 * @param validatedValues 已验证过的值
	 * @param result 校验结果
	 * @param classMetadata 类元数据
	 * @throws IllegalAccessException
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private static void executeFieldValidation(Object bean, boolean failFast, List<Object> validatedValues, 
			ValidationResult result, ClassMetadata classMetadata) throws IllegalAccessException {
		List<FieldMetadata> fieldMetadatas = classMetadata.getFieldMetadatas();
		for (FieldMetadata fieldMetadata : fieldMetadatas) {
			//标记此字段验证是否失败
			boolean fieldFaild = false;
			ValidationContext context = new ValidationContext();
			context.setCls(classMetadata.getCls());
			context.setField(fieldMetadata.getField());
			context.setResult(result);
			// 获取字段值
			Object value = fieldMetadata.getField().get(bean);
			
			List<AnnotationMetadata> annotations = fieldMetadata.getAnnotations();
			if (annotations != null && !annotations.isEmpty()) {
				for (AnnotationMetadata annotationMetadata : annotations) {
					
					//注入绑定的值
					List<FieldMetadata> bindingFields = annotationMetadata.getBindingFields();
					for (FieldMetadata bindingField : bindingFields) {
						Object injectVal = bindingField.getField().get(bean);
						context.getBindingValues().put(bindingField.getField().getName(), injectVal);
					}
					
					Class<? extends Validator> validatorClass = annotationMetadata.getValidatorClass();
					Validator validator = getValidator(validatorClass);
					if(validator.accept(context, value)){
						//同一字段有多个验证约束，若有一个失败，则终止
						if(!validator.validate(context, value)){
							fieldFaild = true;
							interpolateMessage(result, context);
							break;
						}
					}
					
				}
			}
			
			//级联校验
			ClassMetadata reference = fieldMetadata.getReference();
			if (reference != null) {
				if (ObjectUtils.isEmptyCollectionOrArray(value)) {
					continue;
				}
				if (!validateObj(reference, failFast, validatedValues, value, result)) {
					interpolateMessage(result, context);
					fieldFaild = true;
				}
			}
			
			if (fieldFaild && failFast) {
				//快速失败
				break;
			}
			
		}
	}
	
	@SuppressWarnings("rawtypes")
	private static boolean validateObj(ClassMetadata classMetadata, boolean failFast, List<Object> validatedValues, Object value, ValidationResult result) {
		if (value == null) {
			//特别地，对于自我引用情况时，其内部的字段有可能为null
			return true;
		}
		
		Iterator iterator = ObjectUtils.toIter(value);
		if (iterator == null) {
			return validateBean(value, failFast, validatedValues, result);
		}else {
			while (iterator.hasNext()) {
				Object object = iterator.next();
				if (!validateObj(classMetadata, failFast, validatedValues, object, result)) {
					return false;
				}
			}
		}
		return true;
	}
	
	
	/**
	 * 获取指定校验器类的实例对象。优先从缓存中获取，实例对象放入缓存中。
	 * @param validatorClass
	 * @return
	 */
	@SuppressWarnings("rawtypes")
	private static Validator getValidator(Class<? extends Validator> validatorClass) {
		Validator validator = CLASS_VALIDATOR_MAP.get(validatorClass);
		if (validator != null) {
			return validator;
		}
		try {
			validator = validatorClass.newInstance();
		} catch (Exception e) {
			throw new ValidationException("构造校验器失败，校验器为" + validatorClass.getName());
		}
		CLASS_VALIDATOR_MAP.putIfAbsent(validatorClass, validator);
		return validator;
	}
	
	/**
	 * 更新错误模板中的值
	 * @param result
	 * @param context
	 */
	private static void interpolateMessage(ValidationResult result, ValidationContext context){
		int last = result.getErrors().size() - 1;
		String errorMsg = result.getErrors().get(last).getErrorMsg();
		errorMsg = MessageInterpolatorFactory.getInterpolator().interpolate(errorMsg, context);
		result.getErrors().get(last).setErrorMsg(errorMsg);
	}

}
