package com.xdja.framework.validation.handler;

import java.io.IOException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.handler.AbstractHandlerExceptionResolver;

import com.xdja.framework.validation.interceptor.SpringValidationException;

/**
 * Spring验证异常处理。<br/>
 * 对{@link com.xdja.framework.validation.interceptor.SpringValidationInterceptor}抛出的异常{@link SpringValidationException}进行格式化输出处理。<br/>
 * 需要指定{@link ValidationFormat}格式化处理类，和指定输出编码格式。<br/>
 * 默认情况下使用{@link SimpleValidationFormat}格式化，使用UTF-8编码。
 * 
 * @author lvhao
 *
 */
public class SpringValidationExceptionResolver extends AbstractHandlerExceptionResolver{
	
	private static final String DEFAULT_CHARSET = "UTF-8";
	private static final ValidationFormat DEFAULT_FORMAT = new SimpleValidationFormat();
	
	private ValidationFormat format;
	private String charset;
	
	/**
	 * 查找SpringValidationException，在一些情况下，异常会被再次封装，所以往下查找cause，直到查找到为止。
	 * @param t
	 * @return
	 */
	private SpringValidationException findException(Throwable t){
		if (t instanceof SpringValidationException) {
			return (SpringValidationException)t;
		}
		Throwable cause = t.getCause();
		if (cause != null) {
			return findException(cause);
		}
		return null;
	}
	
	@Override
	protected ModelAndView doResolveException(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) {
		
		SpringValidationException sve = findException(ex);
		if (sve != null) {
			if (format == null) {
				format = DEFAULT_FORMAT;
			}
			if (charset == null) {
				charset = DEFAULT_CHARSET;
			}
			
			String out = format.format(sve.getResult());
			try {
				response.setCharacterEncoding(charset);
				response.getWriter().print(out);
			} catch (IOException e) {
				logger.warn("Handling of [" + ex.getClass().getName() + "] resulted in Exception", e);
			}
			return new ModelAndView();
		}
		return null;
	}

	public ValidationFormat getFormat() {
		return format;
	}

	public void setFormat(ValidationFormat format) {
		this.format = format;
	}

	public String getCharset() {
		return charset;
	}

	public void setCharset(String charset) {
		this.charset = charset;
	}
	
 }
