package com.xdja.framework.validation.utils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.xdja.framework.validation.annotation.Binding;
import com.xdja.framework.validation.annotation.FieldType;
import com.xdja.framework.validation.annotation.Good;
import com.xdja.framework.validation.annotation.Group;
import com.xdja.framework.validation.annotation.InjectBinding;
import com.xdja.framework.validation.annotation.ValidatorBy;
import com.xdja.framework.validation.annotation.ValidatorConstraint;
import com.xdja.framework.validation.metadata.AnnotationMetadata;
import com.xdja.framework.validation.metadata.ClassMetadata;
import com.xdja.framework.validation.metadata.FieldMetadata;
import com.xdja.framework.validation.metadata.GroupMetadata;
import com.xdja.framework.validation.validator.ValidationException;
import com.xdja.framework.validation.validator.Validator;

public class FieldUtils {
	
	/**
	 * 加载指定对象待校验的成员属性，并封装成类元数据。<br/>
	 * 
	 * <p>obj需满足以下情况：</p>
	 * <li>JavaBean</li>
	 * <li>实现Collection接口的集合对象</li>
	 * <li>数组对象</li>
	 * 
	 * @param obj
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static ClassMetadata filterFields(Object obj) {
		Object realObj = ObjectUtils.getElement(obj);
		if (realObj != obj) {
			return filterFields(realObj);
		}
		
		ClassMetadata cm = new ClassMetadata();
		Class<?> cls = obj.getClass();
		cm.setCls(cls);

		// 加载分组元数据
		if (hasGroupListAndGroup(cls)) {
			throw new ValidationException("Group.List和Group不可以同时存在");
		} else {
			//List和Group不能同时存在
			GroupWarpper groupWarpper = filterGroupList(cls);
			if (groupWarpper != null) {
				cm.setGroupMetadataMap(groupWarpper.groupMetadataMap);
				cm.setGroupClassValidatorMap(groupWarpper.groupClassValidatorMap);
			} else {
				groupWarpper = filterGroup(cls);
				if (groupWarpper != null) {
					cm.setGroupMetadataMap(groupWarpper.groupMetadataMap);
					cm.setGroupClassValidatorMap(groupWarpper.groupClassValidatorMap);
				}
			}
		}

		try {
			
			//需要绑定的成员属性，目的是将需要绑定的成员属性暂存起来，待全部遍历成员属性后，将InjectBinding的成员属性添加进绑定的成员属性中。
			List<FieldMetadataWarpper> bindingFields = new ArrayList<FieldMetadataWarpper>();
			
			// 获取实体字段集合
			List<Field> fields = ReflectionUtils.getAllFieldsOfClass(cls);
			SortList fieldMetadataSortList = new SortList(fields.size());
			for (Field field : fields) {
				// 通过反射获取该属性对应的值
				field.setAccessible(true);

				FieldMetadata metadata = null;
				FieldMetadataWarpper warpper = filterAnnotation(field);
				//filterAnnotation返回null，该字段有可能是有Good注解，也就是级联校验的字段
				if (warpper == null) {
					metadata = new FieldMetadata();
					metadata.setField(field);
				}
				
				if(warpper != null){
					if (warpper.metadata == null) {
						metadata = new FieldMetadata();
						metadata.setField(field);
						warpper.metadata = metadata;
					} else {
						metadata = warpper.metadata;
					}
					//需要绑定成员属性
					if (warpper.bindingNames != null && warpper.bindingNames.length >0) {
						bindingFields.add(warpper);
					}
				}
				
				

				// 级联
				if (field.isAnnotationPresent(Good.class)) {
					Object value = field.get(obj);
					if (value != null) {
						
						if (ObjectUtils.isEmptyCollectionOrArray(value)) {
							System.out.println("AAA" + value);
							continue;
						}
						//判断下value是否为Collection或Array，若是集合或数组，取第一个元素
						Object element = ObjectUtils.getElement(value);
						// 慎重 对象的循环引用
						// 检查是否为循环引用
						ClassMetadata reference = findSelfReference(cm, element.getClass());
						if (reference == null) {
							reference = filterFields(element);
						}
						metadata.setReference(reference);
						
					}else{
						//skip 当value为空不做处理。所以级联对象校验时，需要保证value不能为null。
						//可以在级联对象字段上添加EmptyCheck注解声明NotNull
					}
				}

				if ((metadata.getAnnotations() == null || metadata.getAnnotations().isEmpty()) && metadata.getReference() == null) {
					// skip 没有验证注解的字段全部忽略，要求所有待验证的字段必须有验证注解
				} else {
					fieldMetadataSortList.insert(metadata, metadata.getOrder());
				}
			}
			
			
			List<FieldMetadata> fieldMetadatas = fieldMetadataSortList.getList();
			if (!bindingFields.isEmpty()) {
				//注入binding字段
				injectBinding(bindingFields, fieldMetadatas);
			}
			
			cm.setFieldMetadatas(fieldMetadatas);
		} catch (Exception e) {
			throw new ValidationException("获取类[" + cls + "]元数据时出错", e);
		}
		
		return cm;
	}
	
	/**
	 * 查找指定类的元数据是否已经存在。目的是为了排除循环引用。<br/>
	 * 若找到类的元数据，则返回元数据的引用。
	 * @param cm
	 * @param cls
	 * @return
	 */
	private static ClassMetadata findSelfReference(ClassMetadata cm, Class<?> cls) {
		if (cm.getCls() == cls) {
			return cm;
		}
		List<FieldMetadata> fieldMetadatas = cm.getFieldMetadatas();
		if (fieldMetadatas != null && !fieldMetadatas.isEmpty()) {
			for (FieldMetadata fieldMetadata : fieldMetadatas) {
				if (fieldMetadata.getReference() != null) {
					ClassMetadata selfReference = findSelfReference(cm, cls);
					if (selfReference != null) {
						return selfReference;
					}
				}
			}
		}
		return null;
	}

	/**
	 * 将待注入绑定的成员属性，添加到指定的Binding中
	 * @param bindingFields
	 * @param fieldMetadatas
	 */
	private static void injectBinding(List<FieldMetadataWarpper> bindingFields, List<FieldMetadata> fieldMetadatas) {
		for (FieldMetadataWarpper warpper : bindingFields) {
			String fieldName = warpper.metadata.getField().getName();
			for (String bindingName : warpper.bindingNames) {
				if (bindingName.length() == 0) {
					throw new ValidationException("字段[" + fieldName + "]指定的绑定名不可为空");
				}

				for (FieldMetadata metadata : fieldMetadatas) {
					boolean find = false;
					List<AnnotationMetadata> annotations = metadata.getAnnotations();
					if (annotations != null && !annotations.isEmpty()) {
						for (AnnotationMetadata annotationMetadata : annotations) {
							//查找绑定名称一致
							if (bindingName.equals(annotationMetadata.getBindingName())) {
								
								if(warpper.metadata == metadata){
									throw new ValidationException("字段[" + fieldName + "]指定的绑定名[" + bindingName + "]不能自我注入");
								}
								find = true;
								// 添加需要注入的字段
								annotationMetadata.getBindingFields().add(warpper.metadata);
								break;
							}
						}

					}

					if (!find) {
						throw new ValidationException("字段[" + fieldName + "]指定的绑定名[" + bindingName + "]不存在");
					}
				}

			}
		}
	}

	/**
	 * 检查指定类中同时存在Group和List
	 * @param clz
	 * @return
	 */
	private static boolean hasGroupListAndGroup(Class<?> clz) {
		Group.List list = clz.getAnnotation(Group.List.class);
		Group group = clz.getAnnotation(Group.class);
		if (list != null && group != null) {
			return true;
		}
		return false;
	}

	/**
	 * 过滤分组List
	 * @param clz
	 * @return
	 */
	@SuppressWarnings("rawtypes")
	private static GroupWarpper filterGroupList(Class<?> clz) {
		GroupWarpper warpper = null;
		Group.List list = clz.getAnnotation(Group.List.class);
		if (list != null) {
			Group[] groups = list.value();
			Map<String, GroupMetadata> groupMetadataMap = new HashMap<String, GroupMetadata>(groups.length);
			Map<String, Class<? extends Validator>> groupClassValidator = new HashMap<String, Class<? extends Validator>>(groups.length);
			if (groups != null && groups.length > 0) {
				for (Group group : groups) {
					if (groupMetadataMap.containsKey(group.name())) {
						throw new ValidationException("分组[" + group.name() + "]重复");
					}
					checkGroup(group);

					if (group.validators().length != 0) {
						groupClassValidator.put(group.name(), group.validators()[0]);
					}

					if (group.fields().length != 0) {
						GroupMetadata groupMetadata = convertGroupToGroupMetadata(group);
						groupMetadataMap.put(group.name(), groupMetadata);
					}
				}
			}
			warpper = new GroupWarpper();
			warpper.groupMetadataMap = groupMetadataMap;
			warpper.groupClassValidatorMap = groupClassValidator;
		}
		
		return warpper;
	}
	
	private static class GroupWarpper{
		/**
		 * 分组与分组元数据
		 */
		Map<String, GroupMetadata> groupMetadataMap;
		/**
		 * 分组与类校验器
		 */
		@SuppressWarnings("rawtypes")
		Map<String, Class<? extends Validator>> groupClassValidatorMap; 
	}

	/**
	 * 过滤分组
	 * @param clz
	 * @return
	 */
	@SuppressWarnings("rawtypes")
	private static GroupWarpper filterGroup(Class<?> clz) {
		GroupWarpper warpper = null;
		Group group = clz.getAnnotation(Group.class);
		if (group != null) {
			warpper = new GroupWarpper();
			
			checkGroup(group);
			
			//分组指定的类级别校验器
			if (group.validators().length != 0) {
				Map<String, Class<? extends Validator>> groupClassValidator = new HashMap<String, Class<? extends Validator>>(1);
				//此处，只取第一个校验器。
				groupClassValidator.put(group.name(), group.validators()[0]);
				warpper.groupClassValidatorMap = groupClassValidator;
			}

			//分组中的成员属性
			if (group.fields().length != 0) {
				Map<String, GroupMetadata> groupMetadataMap = new HashMap<String, GroupMetadata>(1);
				GroupMetadata groupMetadata = convertGroupToGroupMetadata(group);
				groupMetadataMap.put(group.name(), groupMetadata);
				warpper.groupMetadataMap = groupMetadataMap;
			}
			
		}
		return warpper;
	}
	
	@SuppressWarnings("rawtypes")
	private static void checkGroup(Group group) {
		Class<? extends Validator>[] validators = group.validators();
		com.xdja.framework.validation.annotation.Field[] fields = group.fields();
		if (validators.length == 0 && fields.length == 0) {
			throw new ValidationException("分组中的validators和fields不可同时为空");
		}
		if (validators.length != 0 && fields.length != 0) {
			throw new ValidationException("分组中的validators和fields不可同时有值");
		}
		if (validators.length > 1) {
			throw new ValidationException("分组中的validators只可有一个");
		}
	}

	/**
	 * 转成分组为分组元数据
	 * @param group
	 * @return
	 */
	@SuppressWarnings("rawtypes")
	private static GroupMetadata convertGroupToGroupMetadata(Group group) {
		String name = group.name();

		GroupMetadata groupMetadata = new GroupMetadata();
		groupMetadata.setName(name);

		com.xdja.framework.validation.annotation.Field[] fields = group.fields();
		//用于检查分组内的字段是否重复
		List<String> fieldList = new ArrayList<String>(fields.length);
		
		Map<String, Class<? extends Annotation>[]> fieldAnnotationsMap = new HashMap<String, Class<? extends Annotation>[]>(fields.length);
		Map<String, Class<? extends Validator>[]> fieldValidatorsMap = new HashMap<String, Class<? extends Validator>[]>(fields.length);
		Map<String, String[]> fieldGroupsMap = new HashMap<String, String[]>(fields.length);
		
		for (com.xdja.framework.validation.annotation.Field field : fields) {
			String fieldName = field.name();
			if (fieldName.trim().length() == 0) {
				throw new ValidationException("分组中的field名称不可为空");
			}
			if (fieldList.contains(fieldName)) {
				throw new ValidationException("分组中的field名称不可重复");
			}
			fieldList.add(fieldName);

			if (field.annotations().length != 0 && field.validators().length != 0) {
				throw new ValidationException("分组中的annotations和validators不可以同时存在");
			}
			fieldAnnotationsMap.put(fieldName, field.annotations());
			fieldValidatorsMap.put(fieldName, field.validators());
			fieldGroupsMap.put(fieldName, field.groups());
		}
		groupMetadata.setFields(fieldList);
		groupMetadata.setFieldAnnotationsMap(fieldAnnotationsMap);
		groupMetadata.setFieldValidatorsMap(fieldValidatorsMap);
		groupMetadata.setFieldGroupsMap(fieldGroupsMap);
		return groupMetadata;
	}

	/**
	 * 过滤字段的注解，目的是获取注解上标记的校验器，注解包括: Binding,InjectBinding,ValidatorBy注解。<br/>
	 * 但是忽略Good注解。<br/>
	 * 仅当字段上有校验器注解时，才返回FieldMetadataWarpper。否则返回null。
	 * 
	 * @param field
	 * @return
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private static FieldMetadataWarpper filterAnnotation(Field field) {
		String filedName = field.getName();
		Annotation[] annotations = field.getAnnotations();

		// 过滤后的注解，并且添加时排序
		SortList list = new SortList(annotations.length);
		
		String[] bindingNames = null;

		for (Annotation annotation : annotations) {

			// 递归对关联对象进行校验
			if (annotation instanceof Good) {
				continue;
			}

			// 校验器类
			Class<? extends Validator> value = null;
			// 绑定名称
			String bindingName = null;
			
			if (annotation instanceof ValidatorBy) {
				// 自定义的校验器
				value = ((ValidatorBy) annotation).value();
			} else if (annotation instanceof Binding) {
				Binding binding = ((Binding) annotation);
				// 绑定校验
				value = binding.value();
				bindingName = binding.name();
				if (bindingName.length() == 0) {
					throw new ValidationException("字段[" + filedName + "]的绑定名称不可为空");
				}
			} else if (annotation instanceof InjectBinding) {
				// 绑定校验
				//需要遍历查找到对应的binding，并添加进binding中
				bindingNames = ((InjectBinding) annotation).bindingName();
				if (bindingNames.length == 0) {
					throw new ValidationException("字段[" + filedName + "]指定的绑定名不可为空");
				}
				continue;
			} else {
				// 只有校验器的注解才有ValidatorConstraint
				ValidatorConstraint constraint = annotation.annotationType().getAnnotation(ValidatorConstraint.class);

				// 除校验器外，其他注解
				if (constraint == null) {
					continue;
				} else {
					value = constraint.value();
				}

			}

			// order定义属性校验的顺序
			int order = 1;
			try {
				// 这里要求，自定义写的校验注解类，必须有ValidatorConstraint作为Type注解，而且具有order方法
				// 强烈建议添加order方法，如果真没有，就默认order为1
				Method method = annotation.annotationType().getMethod("order");
				if (method != null) {
					order = (Integer) method.invoke(annotation);
				}
			} catch (Exception e) {
				throw new ValidationException("注解[" + annotation + "]获取order失败", e);
			}

			// 防止负数和0
			if (order < 1) {
				throw new ValidationException("注解[" + annotation + "]指定的order必须大于1");
			}

			AnnotationMetadata annotationMetadata = new AnnotationMetadata();
			annotationMetadata.setAnnotation(annotation);
			annotationMetadata.setValidatorClass(value);
			annotationMetadata.setBindingName(bindingName);
			list.insert(annotationMetadata, order);
		}

		if (list.isEmpty() && bindingNames == null) {
			return null;
		}
		
		FieldType fieldType = field.getAnnotation(FieldType.class);
		int filedOrder = 1;
		if (fieldType != null) {
			filedOrder = fieldType.order();
			// 防止负数和0
			if (filedOrder < 1) {
				throw new ValidationException("字段[" + filedName + "]指定的order必须大于1");
			}
		}

		FieldMetadata fa = new FieldMetadata();
		fa.setField(field);
		fa.setOrder(filedOrder);
		fa.setAnnotations(list.getList());
		
		FieldMetadataWarpper warpper = new FieldMetadataWarpper();
		warpper.metadata = fa;
		warpper.bindingNames = bindingNames;
		return warpper;
	}

	private static class FieldMetadataWarpper {
		/**
		 * 成员元数据
		 */
		FieldMetadata metadata;
		/**
		 * 绑定名称，该成员属性需要绑定到哪些校验器上
		 */
		String[] bindingNames;
	}

	/**
	 * 格式化类校验器，仅内部使用
	 * @param cm
	 * @return
	 */
	public static String format(ClassMetadata cm) {
		StringBuilder out = new StringBuilder();
		out.append("Class: " + cm.getCls().getSimpleName());
		out.append("\n\n");
		List<FieldMetadata> fieldMetadatas = cm.getFieldMetadatas();
		if (fieldMetadatas != null) {
			Iterator<FieldMetadata> fmIterator = fieldMetadatas.iterator();
			while (fmIterator.hasNext()) {
				FieldMetadata fieldMetadata = fmIterator.next();
				out.append("Field: " + fieldMetadata.getField().getName());
				out.append("\n");
				List<AnnotationMetadata> annotations = fieldMetadata.getAnnotations();
				if (annotations != null) {
					Iterator<AnnotationMetadata> amIterator = annotations.iterator();
					while (amIterator.hasNext()) {
						AnnotationMetadata am = amIterator.next();
						out.append("Annotation: " + am.getAnnotation().annotationType() + "\tValidator: " + am.getValidatorClass());
						out.append("\n");
					}
				}
				
				ClassMetadata reference = fieldMetadata.getReference();
				if (reference != null) {
					out.append("Reference: " + format(reference));
				}
				out.append("\n");
			}
		}
		return out.toString();
	}

	/**
	 * 打印类元素数据，仅内部使用
	 * @param cm
	 */
	public static void print(ClassMetadata cm) {
		System.out.println(format(cm));
	}

}
