package certkey;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.Security;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;

import com.xdja.cssp.sm2cipher.sm2.cipher.Base64Util;

/**
 * 证书工具类
 * 
 * @author wyf
 *
 */

public class CertUtil {

	public static final String CERT_HEAD = "-----BEGIN CERTIFICATE-----";
	public static final String CERT_TAIL = "-----END CERTIFICATE-----";

	static {
		if (Security.getProvider("BC") == null) {
			Security.addProvider(new BouncyCastleProvider());
		}
	}

	/**
	 * 从16进制字符串进行转换
	 * 
	 * @param str
	 * @return
	 */
	public static X509Certificate getCertFromStr16(String str) {
		byte[] bs = hex2byte(str);
		CertificateFactory cf;
		X509Certificate x509Cert;
		try {
			cf = CertificateFactory.getInstance("X.509", "BC");
			InputStream inStream = new ByteArrayInputStream(bs);
			x509Cert = (X509Certificate) cf.generateCertificate(inStream);
			return x509Cert;
		} catch (Exception e) {
			System.err.println("getCertFromFullStr error: " + e.toString());
		}
		return null;
	}

	/**
	 * 16进制字符串转换为二进制
	 * 
	 * @param str
	 *            String 类型参数
	 * @return
	 */
	public static byte[] hex2byte(String str) {
		if (null == str || str.equals("")) {
			return null;
		}
		str = str.trim();
		StringBuffer sb = new StringBuffer(str);
		int len = sb.length();
		if (len == 0 || len % 2 == 1) {
			return null;
		}
		byte[] b = new byte[len / 2];
		try {
			for (int i = 0; i < len; i += 2) {
				b[i / 2] = (byte) Integer.decode("0x" + sb.substring(i, i + 2))
						.intValue();
			}
			return b;
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * 获取证书的公钥
	 * 
	 * @param str
	 *            证书内容
	 * @param caAlg
	 *            算法；1-RSA；2-SM2
	 * @return 证书公钥 base64格式
	 */
	public static String getPubKey(String str, int caAlg) {

		System.out.println("进入CertUtil->getPubKey: str:\n" + str);

		X509Certificate x509Cert = getCertFromStr(str);
		String pubKey;
		if (caAlg == 1) {
			RSAPublicKey rsa = (RSAPublicKey) x509Cert.getPublicKey();
			pubKey = rsa.getModulus().toString(16).toUpperCase();
		} else {

			ECPublicKey sm2 = (ECPublicKey) x509Cert.getPublicKey();

			String x = sm2.getW().getAffineX().toString(16);

			String y = sm2.getW().getAffineY().toString(16);

			System.out.println("x:" + x.length());
			System.out.println("y:" + y.length());

			if (x.length() == 63) {
				x = "0" + x;
			}
			if (x.length() == 62) {
				x = "00" + x;
			}

			if (y.length() == 63) {
				y = "0" + y;
			}
			if (y.length() == 62) {
				y = "00" + y;
			}

			pubKey = x + y;
		}

		byte[] pub = CertUtil.hex2byte(pubKey);
		return new String(Base64Util.encode(pub));
	}

	/**
	 * 获取证书的公钥
	 * 
	 * @param str
	 *            证书内容
	 * @param caAlg
	 *            算法；1-RSA；2-SM2
	 * @return 证书公钥 base64格式
	 */
	public static String getPubKey(X509Certificate x509Cert, int caAlg) {

		System.out.println("进入CertUtil->getPubKey: str:\n");

		String pubKey;
		if (caAlg == 1) {
			RSAPublicKey rsa = (RSAPublicKey) x509Cert.getPublicKey();
			pubKey = rsa.getModulus().toString(16).toUpperCase();
		} else {
			ECPublicKey sm2 = (ECPublicKey) x509Cert.getPublicKey();

			String x = sm2.getW().getAffineX().toString(16);

			String y = sm2.getW().getAffineY().toString(16);

			System.out.println("x:" + x.length());
			System.out.println("y:" + y.length());

			if (x.length() == 63) {
				x = "0" + x;
			}
			if (x.length() == 62) {
				x = "00" + x;
			}

			if (y.length() == 63) {
				y = "0" + y;
			}
			if (y.length() == 62) {
				y = "00" + y;
			}

			pubKey = x + y;
		}

		byte[] pub = CertUtil.hex2byte(pubKey);
		return new String(Base64Util.encode(pub));
	}

	public static X509Certificate getCertFromStr(String str) {
		str = str.replace(CERT_HEAD, "").replace(CERT_TAIL, "");
		str = str.replace("\r", "").replace("\n", "");
		str = str.replace("\\r", "").replace("\\n", "");

		X509Certificate x509Cert;
		x509Cert = getCertFromB64(str);
		if (x509Cert == null) {
			x509Cert = getCertFromFullStr(str);
		}
		if (x509Cert == null) {
			x509Cert = getCertFromStr16(str);
		}
		return x509Cert;
	}

	public static X509Certificate getCertFromFullStr(String str) {
		CertificateFactory cf;
		X509Certificate x509Cert;
		try {
			cf = CertificateFactory.getInstance("X.509", "BC");
			InputStream inStream = new ByteArrayInputStream(str.getBytes());
			x509Cert = (X509Certificate) cf.generateCertificate(inStream);
			return x509Cert;
		} catch (Exception e) {
			System.err.println("getCertFromFullStr error: " + e.toString());
		}
		return null;
	}

	public synchronized static X509Certificate getCertFromB64(String b64) {
		CertificateFactory cf;
		X509Certificate x509Cert;
		try {
			cf = CertificateFactory.getInstance("X.509", "BC");
			byte[] bsCert = Base64.decode(b64);
			InputStream inStream = new ByteArrayInputStream(bsCert);
			x509Cert = (X509Certificate) cf.generateCertificate(inStream);
			return x509Cert;
		} catch (Exception e) {
			System.err.println("getCertFromB64 error: " + e.toString());
		}
		return null;
	}

	public static String getSn(X509Certificate cert) {
		return cert.getSerialNumber().toString(16).toLowerCase();
	}

	public static String certToFullB64(Certificate cert) {
		String certb64 = "";
		try {

			certb64 = bytesToFullB64(cert.getEncoded());

		} catch (Exception e) {
			System.err.println("certToFullB64 error:" + e.toString());
		}
		return certb64;
	}

	public static String bytesToFullB64(byte[] certder) {
		String certb64;
		try {
			certb64 = new String(Base64.encode(certder));
			certb64 = CERT_HEAD + "\n" + certb64 + "\n" + CERT_TAIL + "\n";
			return certb64;
		} catch (Exception e) {
			System.err.println("certDerToFullB64 error:" + e.toString());
		}
		return null;
	}

	public static String warpCert(String certb64) {
		return CERT_HEAD + "\n" + certb64 + "\n" + CERT_TAIL + "\n";

	}

	public static boolean isSignCert(X509Certificate cert) {
		boolean[] result = cert.getKeyUsage();

		if (result[0] && result[1]) {
			return true;
		}
		return false;
	}

	public static void main(String[] args) {
		String cert = "-----BEGIN CERTIFICATE-----MIICVzCCAfugAwIBAgIIYhEAAAA0nfowDAYIKoEcz1UBg3UFADBGMQswCQYDVQQGEwJjbjEfMB0GA1UEChMWWGluRGFKaWVBbiBDb3Jwb3JhdGlvbjEWMBQGA1UEAwwNWGRqYVN1YkNhX3NtMjAeFw0xNzA5MDUwMzQ1MDVaFw0zNTA0MDkwNjA0MzBaMGYxCzAJBgNVBAYTAmNuMR8wHQYDVQQKExZYaW5EYUppZUFuIENvcnBvcmF0aW9uMQswCQYDVQQLEwJ0ZjEpMCcGA1UEAxMgNzg2NDZhNjE3ODY0NmE2MTM0MzU1NTMyMzcwYTVmM2EwWTATBgcqhkjOPQIBBggqgRzPVQGCLQNCAARU8DBBf+1f/THGf8fnzGhiL0xB4zwGD3jopsQ++NI3PwEwPzwp+U0Mc1AXEbkMWAh9U4BqEx4H5LqO31pmIaT2o4GwMIGtMB8GA1UdIwQYMBaAFEWsAKhifGqo81AsvQijnIHRcPtGMB0GA1UdDgQWBBS2eKRYLGDmHqvP3Vj0dp8dogLsODAJBgNVHRMEAjAAMFAGA1UdHwRJMEcwRaBDoEGGP2h0dHA6Ly9jcmwuc2FmZWNlbnRlci5jb20vQ3JsRG93bmxvYWQ/ZHA9VXNlckNBMzQuY3JsJmNhQWxnPTIuMDAOBgNVHQ8BAf8EBAMCAzgwDAYIKoEcz1UBg3UFAANIADBFAiAOBp91TveYi+JpcaBPX8TAe3bZ6Q/xtUtGOpYKVvS45AIhAO38z1lG663x3Y+a6r7qjUytvHdIvewNi+d6TthRE2NC-----END CERTIFICATE-----";

		System.out.println(CertUtil.getPubKey(cert, 2));

	}
}
