/**
 * <h2>信大捷安卡控件的javascript API</h2>
 * 说明:
 * 1. 文件被引入页面后,会自动初始化,且仅初始化一次
 * 2. 初始化时仅检测安全控件是否安装,以及版本号是否最新
 * 3. 该组件仅有一个顶级对象--Card,该对象也仅有一个方法--ready
 * 4. 方法都支持返回值和回调函数的方式,即回调的参数和返回值的类型一致
 * 5. 在回调函数中都会有success属性,当success=false时,对象会有code和message属性
 * 6. 所调用的函数全部都经过异常处理,也就是说当不需要错误消息和错误码时完全可以忽略这些信息
 * 7. 自动对浏览器兼容性判断,当浏览器支持websocket时会优先建立websocket连接,否则以插件的形式调用控件
 * 8. 最优先兼容信大捷安PC容器
 * @author 马德成
 * @since 2015-02-21
 * @module API
 * @main Card 读卡控件的入口类
 */
(function (root){

	/**
	 * 读卡控件的入口类
	 * @class Card
	 * @main Card
	 */
	(!root.Card) && (root.Card = {});
	//常量定义
	var Const = Card.Const || root.Const;

	//只是一个空函数的简写形式
	var noop = function(){};

	/**
	 * 比较版本号大小
	 * @author 马德成
	 * @date 2015-02-21
	 * @param currentVersion 当前版本号
	 * @param version 需要的最低版本号
	 * @return {String} 
	 * 0. 相同
	 * 1. currentVersion小于version
	 * 2. currentVersion大于version
	 */
	function compareVersion(currentVersion, version) {
		if(currentVersion == version) return 0;

		var currentVersionArray = currentVersion.split('.');
		var versionArray = version.split('.');	
		var len1 = currentVersionArray.length;
		var len2 = versionArray.length;

		if (len1 == len2){
			for (var i = 0; i < len1; i++){
				if (currentVersionArray[i] - versionArray[i] > 0){
					return 2;
				}

				if (currentVersionArray[i] - versionArray[i] < 0){
					return 1;
				}
			 }
		}

		if (len1 < len2) return 1;
		if (len1 > len2) return 2;
		return 0;
	}
	
	/**
	 * 从异常对象中获取错误码
	 * 在以IE为内核的浏览器，在异常中捕获ex.number参数。
	 * 以支持NPAPI的浏览器，在异常中捕获ex。但firefox和chrome浏览器捕获的信息有一些不同
	 * 示例：
	 * 假设错误码为-100。
	 * 在IE浏览器中捕获异常ex.number，得到信息为-100;
	 * 在firefox浏览器中捕获异常 ex，得到信息-100
	 * 在chrome浏览器中捕获异常 ex，得到信息 Error:-100，即”Error:”后面跟着错误码。
	 */
	function getExCode(ex){
		//处理IE
		if(typeof ex.number != 'undefined') return ex.number-0;
		//处理chrome
		if(typeof ex.message!= 'undefined') return ex.message-0;
		//处理firefox
		return ex-0;
	}

	/**
	 * 读取卡函数
	 * @author 马德成
	 * @date 2015-02-21
	 * @param cardType 卡类型 1:TF卡,2:USBKey,其他:不区分 -可选(默认:-1)
	 * @param cardUsage 用户类型 --0-管理卡,1-用户卡 -1:所有
	 */
	function readCard(cardType, cardUsage, callback){
		if(typeof callback != 'function') callback = noop;

		var cardNos = plugin.card().GetAllCards(cardType, cardUsage);
		cardNos = cardNos? cardNos.split('#') : [];
		
		var ret = new CertWrap(cardNos);
		callback.call(ret, ret);
		return ret;
	}
	
	/**
	 * 处理pin码错误
	 * @param {Object} fn
	 * @param {Object} context
	 * @param {Object} callback
	 */
	function getPinResult(fn, context, callback){
		if(typeof callback != 'function') callback = noop;
		
		var ret, lockError = {'-10':true, '-16':true, '-4':true, '-5':true};
		try { 
			var result = fn.call(context, plugin.card());
			var time = result.time - 0;
			if(result.pinTry) {
				ret = time>0?{success:true, code:time, isLock:false}:{success:false, code:time, message:Const.errorDesc[time]||'未知错误', isLock:!!lockError[time]};
			} else {
				ret = time?{success:false, code:time, message: Const.errorDesc[time] || 'PIN码错误', isLock:!!lockError[time]} : {success:true, isLock:false};
			}
		} catch(e) {
			var code = getExCode(e);
			ret = {success:false, code:code, message: Const.errorDesc[code] || '未知错误', isLock:!!lockError[code], ex:e};
		}
		
		callback.call(context, ret);
		return ret;
	}
	
	/**
	 * 拔卡函数处理,当卡被拔出时,stop函数将自动执行
	 * @param {String} cardId 卡号
	 * @param {Object} callback 拔卡时的回调函数
	 * @param {Boolean} isHandle 是否是句柄
	 * @return {Object} {stop:function} 返回一个可以停止监听的函数
	 */
	function pullCard(cardId, callback, isHandle){
		if(typeof callback != 'function') callback = noop;
		if(!cardId) throw new Error('cardId为必须参数');
		var handle, ret = {stop:noop}, error = false;
		
		if(isHandle) {
			handle = cardId;
		} else {
			try {
				if ((handle = plugin.card().GetCardHandle(cardId)) == -1) error = true;
			} catch (e) {
				error = true;
			}
			
			if(error){
				callback.call(ret, ret);
				return ret;
			}
		}
		
		var timer, fn = function(){
			try {
				if(!plugin.card().GetCosVer(handle)) error = true; //没有获取到CosVer
			} catch (e) {
				error = true;
			}
			
			if(error){
				clearInterval(timer);
				callback.call(ret, ret);
			}
		};
		
		timer = setInterval(fn, 800);
		fn();
		ret.stop = function(){if(timer) clearInterval(timer);};
		return ret; //停止检测
	}
	
	/**
	 * 一个函数调用的包装器,统一处理fn的异常,并设置fn的上下文
	 * @param {Function} fn 待执行的函数
	 * @param {Object} ctx fn的上下文
	 * @param {Function} [callback] 执行结果的回调函数
	 */
	function fnCaller(fn, ctx, callback){
		if(typeof callback != 'function') callback = noop;
		var ret;
		
		try {
			ret = fn.call(ctx, plugin.card());
		} catch (e) {
			var code = getExCode(e);
			ret = {success:false, code:code, message:Const.errorDesc[code] || '未知错误', ex: e};
		}
		
		callback.call(ctx, ret);
		return ret;
	}
	
	//初始化插件信息
	var plugin = (function(d){
		var cardCertObj;

		if(typeof safekey == "object" && root.safekey) {
			cardCertObj = safekey; //兼容xdja nw
		} else {
			if (!!root.ActiveXObject || 'ActiveXObject' in root) {
				try {
					cardCertObj = new ActiveXObject("CardCert.WriteCard");
				} catch (e) {
					return {error:{success:false, code:0, message:Const.readyConst[0]}, ex:e};
				}
			} else{

				if(!Card.OS.windows) { //不支持的系统
					return {error:{success:false, code:6, message:Const.readyConst[6]}};
				}

				var os = Card.OS;
				var sys = browsers[os.name.replace(' ', '').toLowerCase()]; //校验系统白名单
				if(!sys) {
					return {error:{success:false, code:6, message:Const.readyConst[6]}};
				}

				var cpu = sys[os.x64?'x64' : 'x86']; //校验CPU白名单
				if(!cpu) {
					return {error:{success:false, code:7, message:Const.readyConst[7]}};
				}

				var browser = Card.browser;
				var names = cpu[browser.image.replace(' ', '').toLowerCase()];
				if(!names || !names.length) {
					return {error:{success:false, code:8, message:Const.readyConst[8]}};
				}

				if(!browser.version) {

				}

				var vers = [];
				for(var i=0; i<names.length; i++ ) {

					var vs = names[i].version.split('-');
					if(vs.length == 1 && new RegExp(vs[0], 'gi').test(browser.version)) {
						vers.push(names[i]);
					} else if(vs.length == 2) {
						browser.version = parseFloat((browser.version + '').replace(/[A-Z]/gi, ''));
						vs[0] = parseFloat(vs[0]);
						vs[1] = parseFloat(vs[1]);

						if(browser.version >= vs[0] && browser.version <= vs[1]) {
							vers.push(names[i]);
						}
					}
				}

				if(!vers.length) {
					return {error:{success:false, code:9, message:Const.readyConst[9]}};
				}

				var pluginType, core;
				for(var j=0; j<vers.length; j++) {
					core = vers[j].core;
					for (var i = 0; i < core.length; i++) {
						if (core[i].x64 == browser.x64 && browser.core.name == core[i].type) {
							pluginType = core[i].plugin;
							break;
						}
					}
				}

				if(!pluginType) {
					return {error:{success:false, code:10, message:Const.readyConst[10]}};
				}

				if(pluginType == 'npapi') { //使用npapi插件
					d.write(Const.readyConst.getEmbed());
					cardCertObj = d.getElementById(Const.readyConst.pluginId);

					if(!cardCertObj.Test) {
						return {error:{success:false, code:0, message:Const.readyConst[0]}};
					}
				} else if(pluginType == 'activex') { //使用activex插件
					try {
						cardCertObj = new ActiveXObject("CardCert.WriteCard");
					} catch (e) {
						return {error:{success:false, code:0, message:Const.readyConst[0]}, ex:e};
					}
				} else if(pluginType == 'websocket'){ //使用websocket
					return {ws:Card.__socket};
				}
			}
		}
		
		var ver = cardCertObj.Test();
		if(compareVersion(ver, Const.lowestVersion) == 1) { //小于最低版本号
			return {error:{success:false, code:3, message:Const.readyConst[3]}};
		} 

		var cardWrap = new CardWrap();
		cardWrap.newVer = compareVersion(ver, Const.version) == 1;
		cardWrap.success = true;
		return {cardWrap:cardWrap, card:function(){return cardCertObj;}};
	})(document);
	
	//初始化过
	var inited;
	
	/**
	 * 初始化卡控件,初始化函数,此函数只会被执行一次
	 * 1. 检测是否安装控件；
	 * 2. 卡控件版本号是否一致；
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {Function} [callback] 对初始化结果处理的回调函数
	 *  callback参数内容为:
	 * 1. 执行成功data={success:true}
	 * 2. 执行失败data={success:false, code:0, message:'错误消息'}
	 * @return {CardWrap}
	 * @method ready
	 * @for Card
	 * @static
	 */
	Card.ready = function (callback){
		if(typeof callback != 'function') callback = noop;

		if(!plugin.ws) {
			if(plugin.error) {
				callback.call(plugin.error, plugin.error);
				return plugin.error;
			}

			callback.call(plugin.cardWrap, plugin.cardWrap);
			return plugin.cardWrap;
		}
		
		if(inited) {
			callback.call(inited, inited);
			return inited;
		}

		//连接ws服务
		plugin.ws.connect(function(wrap, evt){ //连接连接成功

			plugin.ws.getVersion(function(data){ //校验版本号
				if(data.error) {
					inited = data.error;
					return callback.call(inited, inited);
				}

				if(compareVersion(data, Const.lowestVersion) == 1) { //小于最低版本号
					inited = {success:false, code:3, message:Const.readyConst[3]};
					return callback.call(inited, inited);
				}

				wrap.newVer = compareVersion(data, Const.version) == 1;
				wrap.success = true;
				inited = wrap;
				return callback.call(inited, inited);

			});

		}, function(evt){ //初始化连接失败
			inited = {code:5, message:Const.readyConst[5], success:false};
			callback.call(inited, inited);
		});

	};
	
	/**
	 * Cos操作的包装器
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {Number} handle 卡句柄
	 * @class CosWrap
	 * @constructor
	 */
	function CosWrap(handle){
		this.handle = handle;
	}

	/**
	 * 获取Cos版本
	 * @author 马德成
	 * @date 2015-11-10
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, version:'Mod-no1..'}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method getCosVer
	 * @for CosWrap
	 */
	CosWrap.prototype.getCosVer = function (callback){
		var fn = function(plugin){
			var version = plugin.GetCosVer(this.handle);
			return {success:true, version:version};
		};
		return fnCaller(fn, this, callback);
	};

    /**
     * 卡生成随机数
     * @param {Number} mlen 生成随机数的字符串个数（字节数的两倍）
     * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:'随机数字符串'}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method genRandom
     * @for CosWrap
     */
    CosWrap.prototype.genRandom = function(mlen, callback){
        var fn = function(plugin){
            var ret = plugin.GenRandom(this.handle, mlen);
            return {success:!!ret, data:ret};
        };
        return fnCaller(fn, this, callback);
    };

    /**
     * 激活卡功能
     * @param {String} param 激活因子
     * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:'随机数字符串'}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method genRandom
     * @for CosWrap
     */
    CosWrap.prototype.activateCard = function(param, callback){
        var fn = function(plugin){
            var ret = plugin.ActivateCard(this.handle, param);
            return ret? {success:false, code:ret, message:Const.errorDesc[ret]} : {success:true};
        };
        return fnCaller(fn, this, callback);
    };

	/**
	 * 根据卡号创建卡相关操作的工具类
	 * @param {String} cardNo 卡号
	 * @class CertKit
	 * @constructor
	 */
	function CertKit(cardNo){
		//获取卡号
		this.getCardId = function(){return cardNo;};
	}
	
	/**
	 * 获取安全卡信息
	 * @author 马德成
	 * @date 20125-12-5
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, type:1, usage:1, container:[0,3,4]}
	 * <pre>
	 * 。type:卡类型 0:TF卡 1:USB Key
	 * 。usage:卡用途 0:管理卡 1:用户卡
	 * 。container:有证书的容器列表
	 * </pre>
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @method getCardInfo
	 * @for CertKit
	 */
	CertKit.prototype.getCardInfo = function(callback){
		var fn = function(plugin){
			var info = plugin.GetCardInfo(this.getCardId()).split('#');
			return {success:true, type:info[0], usage:info[1], container:info.length == 3? info[2].match(/\d{1}/g):[]};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 读用户交换证书
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {Number} containerId 容器编号
	 * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, cert:cert}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method getCert
	 * @for CertKit
	 */
	CertKit.prototype.getCert = function (containerId, callback){
		var fn = function(plugin){
			var cert = plugin.GetCert(this.getCardId(), containerId);
			return {success:true, cert:cert};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 读取用户证书
	 * @param {Number} containerId 类型 int 容器编号 -必选
	 * @param {Number} certType 证书类型
	 * <pre>
	 * 0:交换证书
	 * 1:签名证书
	 * </pre>
	 * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, cert:cert}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method getCertEx
	 * @for CertKit
	 */
	 CertKit.prototype.getCertEx = function(containerId, certType, callback) {
		var fn = function(plugin){
			var cert = plugin.GetCertEx(this.getCardId(), containerId, certType);
			return {success:true, cert:cert};
		};
		return fnCaller(fn, this, callback);
	};

    /**
     * 二代盾签名
     * @param {String} password 卡用户口令
     * @param {Number} containerid 容器ID
     * @param {String} data 待签名说明
     * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:cert}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method doSignCard2
     * @for CertKit
     */
    CertKit.prototype.doSignCard2 = function (password, containerid, data, callback) {
        var fn = function(plugin){
            var data = plugin.DoSignCard2(this.getCardId(), password, containerid, data);
            return {success:true, data:data};
        };
        return fnCaller(fn, this, callback);
    };

    /**
     * 通过用户证书SN获取所在容器ID
     * @param {String} certsn 证书SN
     * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:1}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method getCtnByCertSn
     * @for CertKit
     */
    CertKit.prototype.getCtnByCertSn = function (certsn, callback) {
        var fn = function(plugin){
            var data = plugin.GetCtnByCertSn(this.getCardId(), certsn);
            return {success:true, data:data};
        };
        return fnCaller(fn, this, callback);
    };

    /**
     * 通过用户证书SN获取所在容器ID
     * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:1}
     * <pre>
     *  . 1 是二代盾
     *  . 0 其他
     * </pre>
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method isUsbKey2
     * @for CertKit
     */
    CertKit.prototype.isUsbKey2 = function (callback) {
        var fn = function(plugin){
            var data = plugin.IsUsbKey2(this.getCardId());
            return {success:true, data:data};
        };
        return fnCaller(fn, this, callback);
    };

    /**
     * 二代盾签名扩展接口
     * @param {String} password 卡用户口令
     * @param {Number} containerid 容器ID
     * @param {String} data 待签名说明
     * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:'签名结果'}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method doSignCard2Ex
     * @for CertKit
     */
    CertKit.prototype.doSignCard2Ex = function (password, containerid, data, callback) {
        var fn = function(plugin){
            var data = plugin.DoSignCard2Ex(this.getCardId(), password, containerid, data);
            return {success:true, data:data};
        };
        return fnCaller(fn, this, callback);
    };

    /**
     * 导入pfx
     * @param {String} pin 卡用户口令
     * @param {Number} containerid 容器ID
     * @param {Number} certusage 证书用途
     * <pre>
     *  0 加密证书
     *  1 签名证书
     * </pre>
     * @param {String} pfx pfx(BASE64编码)
     * @param {String} password 证书口令
     * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:'公钥'}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method importPfx
     * @for CertKit
     */
    CertKit.prototype.importPfx = function (pin, containerid, certusage, pfx, password, callback) {
        var fn = function(plugin){
            var data = plugin.ImportPfx(this.getCardId(), pin, containerid, certusage, pfx, password);
            return {success:true, data:data};
        };
        return fnCaller(fn, this, callback);
    };

	/**
	 * 获取该卡的句柄
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {Function} [callback]  完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, handle:00101010}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method getHandle
	 * @for CertKit
	 */
	CertKit.prototype.getHandle = function(callback) {
		if(typeof callback != 'function') callback = noop;

		var ret, handle;
		try {
			handle = plugin.card().GetCardHandle(this.getCardId());
			ret = {success:true, handle:handle};
		} catch (e) {
			var code = getExCode(e);
			ret = {success:false, code:code, message:Const.errorDesc[code] || '未知错误', ex:e};
		}
		
		callback.call(handle? new CosWrap(handle):this, ret);
		return ret;
	};

	/**
	 * 验证用户口令,使用0x11角色
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {String} pin PIN码
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, isLock:false}
	 * 2. 失败:{success:false, code:1, isLock:false, message:'错误信息', ex:异常对象}
	 * <pre>
	 * 。code为PIN码错误返回重试次数（4 3 2 1）
	 * 。isLock判断卡是否锁死
	 * </pre>
	 * @return {Object} 和callback参数一致
	 * @method checkPin
	 * @for CertKit
	 */
	CertKit.prototype.checkPin = function (pin, callback){
		return this.checkPinEx(pin, 0x11, callback);
	};

	/**
	 * 验证用户口令
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {String} pin PIN码
	 * @param {Number}role 用户角色
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, isLock:false}
	 * 2. 失败:{success:false, code:1, isLock:false, message:'错误信息', ex:异常对象}
	 * <pre>
	 * 。code为PIN码错误返回重试次数（4 3 2 1）
	 * 。isLock判断卡是否锁死
	 * </pre>
	 * @return {Object} 和callback参数一致
	 * @method checkPinEx
	 * @for CertKit
	 */
	CertKit.prototype.checkPinEx = function (pin, role, callback){
		if(!pin) throw new Error('pin为必须参数');
		return getPinResult(function(card){return {time:card.SafePin(this.getCardId(), pin, role)};}, this, callback);
	};
	
	/**
	 * 修改pin码,只有当在卡没被锁死时,才能有用,使用0x11角色
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {String} oldPin 旧PIN码
	 * @param {String} newPin 新PIN码
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, isLock:false}
	 * 2. 失败:{success:false, code:1, isLock:false, message:'错误信息', ex:异常对象}
	 * <pre>
	 * 。code为PIN码错误返回重试次数（4 3 2 1）
	 * 。isLock判断卡是否锁死
	 * </pre>
	 * @return {Object} 和callback参数一致
	 * @method safeChangePin
	 * @for CertKit
	 */
	CertKit.prototype.safeChangePin = function (oldPin, newPin, callback){
		return this.safeChangePinEx(oldPin, newPin, 0x11, callback);
	};
	
	/**
	 * 修改pin码,只有当在卡没被锁死时,才能有用
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {String} oldPin 旧PIN码
	 * @param {String} newPin 新PIN码 
	 * @param {Number} role 角色
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, isLock:false}
	 * 2. 失败:{success:false, code:1, isLock:false, message:'错误信息', ex:异常对象}
	 * <pre>
	 * 。code为PIN码错误返回重试次数（4 3 2 1）
	 * 。isLock判断卡是否锁死
	 * </pre>
	 * @return {Object} 和callback参数一致
	 * @method safeChangePinEx
	 * @for CertKit
	 */
	CertKit.prototype.safeChangePinEx = function (oldPin, newPin, role, callback){
		return getPinResult(function(card){return {time:card.SafeChangePin(this.getCardId(), oldPin, newPin, role)};}, this, callback);
	};
	
	/**
	 * 导入容器证书到浏览器,使用角色0x11
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {Number} containerId 容器编号
	 * @param {Number} certType 证书类型:
	 * <pre>
	 * 0: 交换证书
	 * 1: 签名证书
	 * </pre>
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method importCert
	 * @for CertKit
	 */
	CertKit.prototype.importCert = function (containerId, certType, callback){
		return this.importCertEx(containerId, certType, 0x11, callback);
	};
	
	/**
	 * 导入容器证书到浏览器
	 * @author 马德成
	 * @date 2015-02-21
     * @param {String} pin PIN码
	 * @param {Number} containerId 容器编号
	 * @param {Number} certType 证书类型:
	 * <pre>
	 * 0: 交换证书
	 * 1: 签名证书
	 * </pre>
	 * @param {Number} role 角色
	 * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method importCertEx
	 * @for CertKit
	 */
	CertKit.prototype.importCertEx = function (pin, containerId, certType, role, callback){
		var fn = function(plugin){
			var ret = plugin.ImportKeyCertEx(this.getCardId(), pin, role, containerId, certType);
			return {success:!ret};
		};
		return fnCaller(fn, this, callback);
	};

	/**
	 * 设定浏览器记录卡ID,当关闭浏览器后则失效
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method setCardId
	 * @for CertKit
	 */
	CertKit.prototype.setCardId = function (callback){
		var fn = function(plugin){
			plugin.SetCardId(this.getCardId());
			return {success:true};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 获取PIN码剩余次数,使用角色0x11
	 * @author 马德成
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, code:5, isLock:false} code为剩余次数
	 * 2. 失败:{success:false, code:1, isLock:false, message:'错误信息', ex:异常对象}
	 * <pre>
	 * 。code为PIN码错误返回重试次数（4 3 2 1）
	 * 。isLock判断卡是否锁死
	 * </pre>
	 * @return {Object} 和callback参数一致
	 * @method getPinTryCount
	 * @for CertKit
	 */
	CertKit.prototype.getPinTryCount = function(callback) {
		return this.getPinTryCountEx(0x11, callback);
	};
	
	/**
	 * 获取PIN码剩余次数
	 * @author 马德成
	 * @param {Number} role 角色
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, code:5, isLock:false} code为剩余次数
	 * 2. 失败:{success:false, code:1, isLock:false, message:'错误信息', ex:异常对象}
	 * <pre>
	 * 。code为PIN码错误返回重试次数（4 3 2 1）
	 * 。isLock判断卡是否锁死
	 * </pre>
	 * @return {Object} 和callback参数一致
	 * @method getPinTryCountEx
	 * @for CertKit
	 */
	CertKit.prototype.getPinTryCountEx = function(role, callback) {
		return getPinResult(function(card){return {time:card.GetPinTryCount(this.getCardId(), role), pinTry:true};}, this, callback);
	};

	/**
	 * 根据解锁码,解锁卡并修改PIN码,当卡被锁死时可以被解锁
	 * @author 马德成
	 * @param {String} unlockCode 解锁码
	 * @param {String} newPin 新PIN码
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method unlockCodePin
	 * @for CertKit
	 */
	CertKit.prototype.unlockCodePin = function(unlockCode, newPin, callback){
		return this.unlockCodePinEx(unlockCode, newPin, 0x11, callback);
	};
	
	/**
	 * 根据解锁码,解锁卡并修改PIN码,当卡被锁死时可以被解锁
	 * @author 马德成
	 * @param {String} unlockCode 解锁码
	 * @param {String} newPin 新PIN码
	 * @param {Number} role 角色
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method unlockCodePinEx
	 * @for CertKit
	 */
	CertKit.prototype.unlockCodePinEx = function(unlockCode, newPin, role, callback){
		var fn = function(plugin){
			var code = plugin.ReloadPIN(this.getCardId(), unlockCode, role, newPin) - 0;
			return code?{success:false, code: code, message: Const.errorDesc[code]||'未知错误'}:{success:true};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 生成公私钥对
	 * @author 马德成
	 * @param {String} pin 卡用户口令
	 * @param {Number} containerid 容器编号
	 * @param {Number} alg 算法：
	 * <pre>
	 * 。0:RSA
	 * 。1:SM2
	 * </pre>
	 * @param {Number} certusage 算法用途：
	 * <pre>
	 * 。0:交换
	 * 。1:签名
	 * </pre>
	 * @param {Number} keybits bits 位数：
	 * <pre>
	 * 。RSA:一般为1024
	 * 。SM2:一般为256
	 * </pre>
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, publicKey:'公钥：RSA-Base64编码的M，SM2-B64编码的x=...y=..'}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method genKeypair
	 * @for CertKit
	 */
	CertKit.prototype.genKeypair = function(pin, containerid, alg, certusage, keybits, callback){
		var fn = function(plugin){
			var publicKey = plugin.GenKeypair(this.getCardId(), pin, containerid, alg, certusage, keybits);
			return {success:true, publicKey:publicKey};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 生成公私钥对,默认使用0号容器,交换证书,1024位RSA算法,参见{{#crossLink "CertKit/genKeypair"}}{{/crossLink}}
	 * @param {String} pin 卡用户口令
	 * @param {Number} [containerid] 容器编号
	 * @param {Number} [certusage] 算法用途：
	 * <pre>
	 * 。0:交换
	 * 。1:签名
	 * </pre>
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, publicKey:'公钥：RSA-Base64编码的M，SM2-B64编码的x=...y=..'}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method genKeypairRSA
	 * @for CertKit
	 */
	CertKit.prototype.genKeypairRSA = function(pin, containerid, certusage, callback) {
		if(typeof containerid == 'function'){
			callback = containerid;
			containerid = null;
		} else if(typeof certusage == 'function'){
			callback = certusage;
			certusage = null;
		} 
		return this.genKeypair(pin, containerid||0, 0, certusage||0, 1024, callback);
	};
	
	/**
	 * 生成公私钥对,默认使用0号容器,交换证书,256位SM2算法,参见{{#crossLink "CertKit/genKeypair"}}{{/crossLink}}
	 * @author 马德成
	 * @param {String} pin 卡用户口令
	 * @param {Number} [containerid] 容器编号
	 * @param {Number} [certusage] 算法用途：
	 * <pre>
	 * 。0:交换
	 * 。1:签名
	 * </pre>
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, publicKey:'公钥：RSA-Base64编码的M，SM2-B64编码的x=...y=..'}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method genKeypairSM2
	 * @for CertKit
	 */
	CertKit.prototype.genKeypairSM2 = function(pin, containerid, certusage, callback) {
		if(typeof containerid == 'function'){
			callback = containerid;
			containerid = null;
		} else if(typeof certusage == 'function'){
			callback = certusage;
			certusage = null;
		}
		return this.genKeypair(pin, containerid || 0, 1, certusage || 0, 256, callback);
	};
	
	/**
	 * 向卡容器写入证书
	 * @author 马德成
	 * @param {String} pin 卡用户口令
	 * @param {Number} containerid 容器编号
	 * @param {Number} certusage 算法用途：
	 * <pre>
	 * 。0:交换
	 * 。1:签名
	 * </pre>
	 * @param {String} cert 十六进制编码证书
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method writeCert
	 * @for CertKit
	 */
	CertKit.prototype.writeCert = function(pin, containerid, certusage, cert, callback){
		var fn = function(plugin){
			var ret = plugin.WriteCert(this.getCardId(), pin, containerid, certusage, cert);
			return ret? {success:false, code:ret, message:Const.errorDesc[ret]} : {success:true};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 向卡容器写入证书
	 * @author 马德成
	 * @param {String} [pin] 卡用户口令
	 * @param {Number} [containerid] 容器编号
	 * @param {Number} [certusage] 算法用途：
	 * <pre>
	 * 。0:交换
	 * 。1:签名
	 * </pre>
	 * @param {String} cert 十六进制编码证书
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method writeCertEx
	 * @for CertKit
	 */
	CertKit.prototype.writeCertEx = function(pin, containerid, certusage, cert, callback){
		var _cert = cert;
		if(typeof containerid == 'string'){
			_cert = containerid;
			containerid = null;
			if(typeof certusage == 'function'){
				callback = certusage;
				certusage = null;
			}
		} else if(typeof certusage == 'string'){
			_cert = certusage;
			if(typeof cert == 'function') callback = cert;
			certusage = null;
			cert = null;
		}
		return this.writeCert(pin, containerid||0, certusage||0, _cert, callback);
	};
	
	/**
	 * 写网关单证书（公钥）
	 * @author 马德成
	 * @param {String} pin 卡用户口令
	 * @param {String} cert 十六进制编码证书
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method writeGateCert
	 * @for CertKit
	 */
	CertKit.prototype.writeGateCert = function(pin, cert, callback){
		var fn = function(plugin){
			var ret = plugin.WriteGateCert(this.getCardId(), pin, cert);
			return ret? {success:false} : {success:true, code:ret, message:Const.errorDesc[ret]}
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 写网关双证书（公钥）
	 * @author 马德成
	 * @param {String} pin 卡用户口令
	 * @param {String} cert 十六进制编码证书
	 * @param {String} cert2 监管证书：pem编码 十六进制编码
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method writeGateCertEx
	 * @for CertKit
	 */
	CertKit.prototype.writeGateCertEx = function(pin, cert, cert2, callback){
		var fn = function(plugin){
			var ret = plugin.WriteGateCertEx(this.getCardId(), pin, cert, cert2);
			return ret? {success:false, code:ret, message:Const.errorDesc[ret]} : {success:true};
		};
		return fnCaller(fn, this, callback);
	};

	/**
	 * 写Imsi
	 * @param {String} password 卡用户口令
	 * @param {String} imsi IMSI号(十六进制编码)
     * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method writeImsi
	 * @for CertKit
	 */
	CertKit.prototype.writeImsi = function (password, imsi, callback) {
		var fn = function(plugin){
			var ret = plugin.WriteImsi(this.getCardId(), password, imsi);
			return ret? {success:false, code:ret, message:Const.errorDesc[ret]} : {success:true};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 导入私钥
	 * 将加密私钥导入卡内（支持RSA-1024，SM2，目前不支持RSA_2048）
	 * @author 马德成
	 * @param {Number} role 角色
	 * @param {String} pin 卡用户口令
	 * @param {Number} container 容器ID
	 * @param {Number} alg 算法类型:
	 * <pre>
	 * 。0:RSA
	 * 。1:SM2
	 * </pre>
	 * @param {Number} usage 加密类型:
	 * <pre>
	 * 。0:交换
	 * 。1:签名
	 * </pre>
	 * @param {String} privateKey 要导入的私钥Bsae64编码
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, data:'传入证书的公钥加密之后的密文'}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method importAsymKey
	 * @for CertKit
	 */
	CertKit.prototype.importAsymKey = function(role, pin, container, alg, usage, privateKey, callback){
		var fn = function(plugin){
			var str = plugin.ImportAsymKey(this.getCardId(), role, pin, container, alg, usage, privateKey);
			return {success:true, data:str};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 私钥运算
	 * 用卡内私钥解密，并由传入的证书得到公钥加密明文，将加密的密文传出
	 * @author 马德成
	 * @param {Number} role 角色
	 * @param {String} pin Pin码
	 * @param {Number} container 容器ID
	 * @param {Number} alg 算法类型 0-RSA 1-SM2
	 * @param {Number} usage 证书类型 0- 交换证书  1- 签名证书
	 * @param {String} inBase64 老设备Base64形式的待解密数据(加密后kuep)
	 * @param {String} certBase64 新设备Base64形式的证书内容
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true, dekuep:'老设备解密后的kuep'}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method asymDecryption
	 * @for CertKit
	 */
	CertKit.prototype.asymDecryption = function(role, pin, container, alg, usage, inBase64, certBase64, callback) {
		var fn = function(plugin){
			var str = plugin.AsymDecryption(this.getCardId(), role, pin, container, alg, usage, inBase64, certBase64);
			return {success:!!str, dekuep:str};
		};
		return fnCaller(fn, this, callback);
	};

	/**
	 * 使用卡内证书进行签名运算
	 * @author 马德成
	 * @date 2015-11-28
	 * @param role 角色
	 * @param pin 卡密码
	 * @param container 容器ID
	 * @param alg 算法类型 0-RSA 1-SM2
	 * @param usage 证书类型 0-交换证书 1-签名证书
	 * @param inData 待签名的数据
	 * @param dataType 待签名数据的数据类型 0-inData是经过hash运算的值 1-inData没经过hash运算
	 * @param callback 回调函数 形式如{success:true, signature:'签名值,经base64编码后的值'}
	 */
	CertKit.prototype.cardSign = function(role, pin, container, alg, usage, inData, dataType, callback){
		var fn = function(plugin){
			var str = plugin.CardSign(this.getCardId(), role, pin, container, alg, usage, inData, dataType);
			return {success:!!str, signature:str};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 使用卡内证书进行RAS签名运算
	 * @author 马德成
	 * @date 2015-11-28
	 * @param pin 卡密码
	 * @param container 容器ID
	 * @param usage 证书类型 0-交换证书 1-签名证书
	 * @param inData 待签名的数据(原文)
	 * @param callback 回调函数 形式如{success:true, signature:'签名值,经base64编码后的值'}
	 */
	CertKit.prototype.cardRSASign = function(pin, container, usage, inData, callback){
		return this.cardSign(0x11, pin, container, 0, usage, inData, 1, callback);
	};
	
	/**
	 * 使用卡内证书进行SM2签名运算
	 * @author 马德成
	 * @date 2015-11-28
	 * @param pin 卡密码
	 * @param container 容器ID
	 * @param usage 证书类型 0-交换证书 1-签名证书
	 * @param inData 待签名的数据(原文)
	 * @param callback 回调函数 形式如{success:true, signature:'签名值,经base64编码后的值'}
	 */
	CertKit.prototype.cardSM2Sign = function(pin, container, usage, inData, callback){
		return this.cardSign(0x11, pin, container, 1, usage, inData, 1, callback);
	};
	
	/**
	 * 使用卡内证书进行RAS签名运算
	 * @author 马德成
	 * @date 2015-11-28
	 * @param pin 卡密码
	 * @param container 容器ID
	 * @param usage 证书类型 0-交换证书 1-签名证书
	 * @param inData 待签名的数据(经过hash运算)
	 * @param callback 回调函数 形式如{success:true, signature:'签名值,经base64编码后的值'}
	 */
	CertKit.prototype.cardSha1RSASign = function(pin, container, usage, inData, callback){
		return this.cardSign(0x11, pin, container, 0, usage, inData, 0, callback);
	};
	
	/**
	 * 使用卡内证书进行SM2签名运算
	 * @author 马德成
	 * @date 2015-11-28
	 * @param pwd 卡密码
	 * @param container 容器ID
	 * @param usage 证书类型 0-交换证书 1-签名证书
	 * @param inData 待签名的数据(经过hash运算)
	 * @param callback 回调函数 形式如{success:true, signature:'签名值,经base64编码后的值'}
	 */
	CertKit.prototype.cardSha1SM2Sign = function(pwd, container, usage, inData, callback){
		return this.cardSign(0x11, pwd, container, 1, usage, inData, 0, callback);
	};

    /**
     * 获取解锁码 获取8位解锁码
     * @param {String} parameter1 明文
     * @param {String} parameter2 16位密钥
     * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:'长度为8的解锁码'}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method genCode
     * @for CertKit
     */
	CertKit.prototype.genCode = function(parameter1, parameter2, callback){
        var fn = function(plugin){
            var str = plugin.GenCode(this.getCardId(), parameter1, parameter2);
            return {success:!!str, data:str};
        };
        return fnCaller(fn, this, callback);
	};

    /**
     * SM1加解密运算   卡外送入密钥，进行SM1加解密运算
     * @param {String} tempkey 16字节SM1密钥(32个内存字符)
     * @param {String} pDataIn 需要加密的数据(内存字符)
     * @param {Number} flag 指示加密解密与运算模式
     * @param {Number} pIV 输入iv输出iv，在CBC时有效
     * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:'SM1运算结果'}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method sm1key
     * @for CertKit
     */
	CertKit.prototype.sm1key = function(tempkey, pDataIn, flag, pIV, callback){
        var fn = function(plugin){
            var str = plugin.SM1KEY(this.getCardId(), tempkey, pDataIn, pDataIn.length, flag, pIV);
            return {success:!!str, data:str};
        };
        return fnCaller(fn, this, callback);
	};

    /**
     * SM3运算
     * @param {String} pDataIn 需要加密的数据(内存字符)
     * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:'SM3运算结果，长度为32字节（64个内存字符）'}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method sm3
     * @for CertKit
     */
	CertKit.prototype.sm3 = function(pDataIn, callback){
        var fn = function(plugin){
            var str = plugin.SM3(this.getCardId(), pDataIn, pDataIn.length);
            return {success:!!str, data:str};
        };
        return fnCaller(fn, this, callback);
	};

    /**
     * 卡内写文件接口 将字符串写入009a文件
     * @param {String} writestr 需要写入文件的字符串
     * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method writeFileToCard
     * @for CertKit
     */
	CertKit.prototype.writeFileToCard = function(writestr, callback){
        var fn = function(plugin){
            var ret = plugin.WriteFileToCard(this.getCardId(), writestr);
            return ret? {success:false, code:ret, message:Const.errorDesc[ret]} : {success:true};
        };
        return fnCaller(fn, this, callback);
	};

	/**
	 * 卡内写文件接口 将字符串写入指定文件
	 * @param {String} writestr 需要写入文件的字符串
	 * @param {String} fidStr 需要的文件id字符串
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method writeFileToCardEx
	 * @for CertKit
	 */
	CertKit.prototype.writeFileToCardEx = function(fidStr,writestr, callback){
		var fn = function(plugin){
			var ret = plugin.WriteFileToCardEx(this.getCardId(),fidStr, writestr,writestr.length);
			return ret? {success:false, code:ret, message:Const.errorDesc[ret]} : {success:true};
		};
		return fnCaller(fn, this, callback);
	};


	/**
	 * 卡内读文件接口
	 * @param {String} fidStr 需要的文件id字符串
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method readBinaryFile
	 * @for CertKit
	 */
	CertKit.prototype.readBinaryFile = function(fidStr,callback){
		var fn = function(plugin){
			var ret = plugin.ReadBinaryFile(this.getCardId(),fidStr);
			return ret? {success:false, code:ret, message:Const.errorDesc[ret]} : {success:true,data: ret};
		};
		return fnCaller(fn, this, callback);
	};

    /**
     * 获取激活状态
     * @param {String} param 激活因子
     * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:'随机数字符串'}
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method checkActState
     * @for CertKit
     */
    CertKit.prototype.checkActState = function(callback){
        var fn = function(plugin){
            var ret = plugin.CheckActState(this.getCardId());
            return ret? {success:false, code:ret, message:Const.errorDesc[ret]} : {success:true};
        };
        return fnCaller(fn, this, callback);
    };

	/**
	 * 当拔卡时,执行的操作
	 * @author 马德成
	 * @date 2015-02-21
	 * @param callback 类型 function()当拔卡时执行的回调
	 */
	CertKit.prototype.onpullcard = function(callback){
		return pullCard(this.getCardId(), callback);
	};
	
	/**
	 * 根据解锁码,解锁加密盘
	 * @author LB
	 * @param {String} unlockCode 解锁码
	 * @param {String} newPin 新PIN码
	 * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
	 * 1. 成功:{success:true}
	 * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
	 * @return {Object} 和callback参数一致
	 * @method unlockUsbSecuZone
	 * @for CertKit
	 */
	CertKit.prototype.unlockUsbSecuZone = function(unlockCode, newPin, callback){
		var fn = function(plugin){
			/*-2    参数错误
			  -1    找不到卡
			  -19  解密静态解锁码失败
			  -43  解锁加密盘失败
			*/
			var code = plugin.UnlockUsbSecuZone(this.getCardId(), unlockCode, newPin) - 0;
			return (code<0)?{success:false, code: code, message: Const.errorDesc[code]||'未知错误'}:{success:true};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 对卡号枚举的包装器
	 * @author 马德成
	 * @date 2015-02-21
	 * @param {Array} cardNos 卡号列表
	 * @class CertWrap
	 * @constructor
	 */
	function CertWrap(cardNos){
		for(var i = 0; i < cardNos.length; i++ ) {
            this[i] = new CertKit(cardNos[i]);
        }
        this.length = cardNos.length;
	}
	
	/**
	 * 遍历卡号
	 * @author 马德成
	 * @date 2015-02-21
	 * @param callback 类型 function(index, cardId)
	 */
	CertWrap.prototype.each = function (callback){
		if(typeof callback == 'function') {
			for (var i = 0; i < this.length; i++) {
				callback.call(this[i], i, this[i]);
			}
		}
		return this;
	};
	
	/**
	 * 把另一个certWrap放到当前CertWrap里
	 * 例如:
	 * var cards = plugin.readUserUSBKeyCard(); //读取Ukey
	 * var tfs = plugin.readUserTFCard();//读取TF卡
	 * cards.push(tfs); //把读取的TF卡结果push到cards
	 * @param certWrap
	 * @returns {CertWrap}
	 */
	CertWrap.prototype.push = function(certWrap) {
		if(!certWrap || !certWrap.length) return this;
		var len = this.length;
		
		for (var i = 0; i < certWrap.length; i++) {
			this[len++] = certWrap[i];
		}
		this.length = this.length + certWrap.length;
		return this;
	};

	/**
	 * 读取安全卡操作的类
	 * @author 马德成
	 * @date 2015-02-21
	 * @class CardWrap
	 * @constructor
	 */
	function CardWrap(){}
	
	/**
	 * 当拔卡时,执行的操作
	 * @author 马德成
	 * @date 2015-02-21
	 * @param cardId 类型 string 卡号
	 * @param callback 类型 function()当拔卡时执行的回调
	 */
	CardWrap.prototype.onpullcard = function(cardId, callback){
		return pullCard(cardId, callback);
	};
	
	/**
	 * 当拔卡时,执行的操作
	 * @author 马德成
	 * @date 2015-02-21
	 * @param handle 类型 string 卡句柄
	 * @param callback 类型 function()当拔卡时执行的毁掉
	 */
	CardWrap.prototype.onpullhandle = function(handle, callback){
		return pullCard(handle, callback, true);
	};

	/**
	 * 读取所有卡包含用户卡管理卡,TF卡和USB Key
	 * @author 李惠荣
	 * @date 2017-12-18
	 */
	CardWrap.prototype.readAllCard = function (callback){
		return readCard.call(this, -1, -1, callback);
	};
	
	/**
	 * 读取所有用户卡,包括TF卡和USB Key
	 * @author 马德成
	 * @date 2015-02-21
	 */
	CardWrap.prototype.readUserCard = function (callback){
		console.log(this);
		return readCard.call(this, -1, Const.cardUsage.X_CARD_USAGE_USER, callback);
	};
	
	/**
	 * 读取ACE手机的芯片号
	 * @author 马德成
	 * @date 2015-08-14
	 */
	CardWrap.prototype.readUserACE = function (callback){
		return readCard.call(this, 2, Const.cardUsage.X_CARD_USAGE_USER, callback);
	};
	
	/**
	 * 读取用户TF卡
	 * @author 马德成
	 * @date 2015-02-21
	 */
	CardWrap.prototype.readUserTFCard = function (callback){
		return readCard.call(this, 0, Const.cardUsage.X_CARD_USAGE_USER, callback);
	};
	
	/**
	 * 读取用户USB Key
	 * @author 马德成
	 * @date 2015-02-21
	 */
	CardWrap.prototype.readUserUSBKeyCard = function (callback){
		return readCard.call(this, 1, Const.cardUsage.X_CARD_USAGE_USER, callback);
	};
	
	/**
	 * 读取管理员卡,包括TF卡和USB Key
	 * @author 马德成
	 * @date 2015-02-21
	 */
	CardWrap.prototype.readAdminCard = function (callback){
		return readCard.call(this, -1, Const.cardUsage.X_CARD_USAGE_ADMIN, callback);
	};
	
	/**
	 * 读取管理员卡,包括TF卡
	 * @author 马德成
	 * @date 2015-02-21
	 */
	CardWrap.prototype.readAdminTFCard = function (cardType, callback){
		return readCard.call(this, 0, Const.cardUsage.X_CARD_USAGE_ADMIN, callback);
	};
	
	/**
	 * 读取管理员USB Key
	 * @author 马德成
	 * @date 2015-02-21
	 */
	CardWrap.prototype.readAdminUSBKeyCard = function (cardType, callback){
		return readCard.call(this, 1, Const.cardUsage.X_CARD_USAGE_ADMIN, callback);
	};
	
	/**
	 * 在左面右下角弹出消息提示框
	 * @author 马德成
	 * @date 2015-02-21
	 * @param message 消息内容
	 * @param time 显示时长(单位:s)，如果取值0，永久显示
	 * @param  type 1-oa  2-zw
	 * @param callback 类型 function(data)
	 * 成功,data={success:true}
	 * 失败,data={success:false, code:0, message:'错误信息', ex:异常对象}
	 */
	CardWrap.prototype.showMsgTip = function (message, time, type, callback){
		var fn = function(plugin){
			plugin.ShowMsgTip(message, time, type);
			return {success:true};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 获当前浏览器记录的卡ID,当关闭浏览器后则获取不到记录
	 * @author 马德成
	 * @date 2015-02-21
	 * @param callback 类型 function(data)
	 * 成功,data={success:true, cardId:'1df0d0f0'}
	 * 失败,data={success:false, code:0, message:'错误信息', ex:异常对象}
	 */
	CardWrap.prototype.getCard = function(callback){
		var fn = function(plugin){
			return {success:true, cardId:plugin.GetCardId()};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 根据卡号获取CertKit对象
	 * @param cardNo 卡号
	 * @returns {CertKit} 卡操作的工具类
	 */
	CardWrap.prototype.cardKit = function(cardNo){
		return new CertKit(cardNo);
	};
	
	/**
	 * 私钥运算
	 * 用卡内私钥解密，并由传入的证书得到公钥加密明文，将加密的密文传出
	 * @param cardId 老卡的卡号
	 * @param role 角色
	 * @param pin Pin码
	 * @param container 容器ID
	 * @param alg 算法类型 0-RSA 1-SM2
	 * @param usage 证书类型 0- 交换证书  1- 签名证书
	 * @param inBase64 老设备Base64形式的待解密数据(加密后kuep)
	 * @param certBase64 新设备Base64形式的证书内容
	 * @param callback 回调函数 形式如{success:true, dekuep:'老设备解密后的kuep'}
	 */
	CardWrap.prototype.asymDecryption = function(cardId, role, pin, container, alg, usage, inBase64, certBase64, callback) {
		var fn = function(plugin){
			var str = plugin.AsymDecryption(cardId, role, pin, container, alg, usage, inBase64, certBase64);
			return {success:!!str, dekuep:str};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 获取连接手机个数
	 * @return 返回枚举的手机个数
	 */
	CardWrap.prototype.countACE = function(callback){
		var fn = function(plugin){
			return {success:true, count:plugin.EnumACE() || 0};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 连接ACE手机
	 * code -31:未连接手机
	 */
	CardWrap.prototype.connectACE = function(callback){
		var fn = function(plugin){
			var str = plugin.ConnectACE();
			return {success:str == undefined? false : !!!str};
		};
		return fnCaller(fn, this, callback);
	};
	
	/**
	 * 卸载APK
	 */
	CardWrap.prototype.uninstallApk = function(callback){
		var fn = function(plugin){
			plugin.UnInstallApk();
			return {success:true};
		};
		return fnCaller(fn, this, callback);
	};


    /**
     * 通过用户证书SN获取所在容器ID
     * @param {String} certsn 证书SN，卡内任意容器证书SN
     * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
     * 1. 成功:{success:true, data:1}
     * <pre>
     *  . 1 是二代盾
     *  . 0 其他
     * </pre>
     * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
     * @return {Object} 和callback参数一致
     * @method isUsbKey2Ex
     * @for CardWrap
     */
    CardWrap.prototype.isUsbKey2Ex = function (certsn, callback) {
        var fn = function(plugin){
            var data = plugin.IsUsbKey2Ex(certsn);
            return {success:true, data:data};
        };
        return fnCaller(fn, this, callback);
    };

	/**
	 * 颜色的16进制形式,转换为rgb形式
	 * @param hex 16进制颜色值
	 * @returns {Array} rgb颜色数组
	 */
	function hexToRgb(hex) {
		if (!hex || !{7:true, 4:true}[hex.length] || !/^#/.test(hex)){
			return ;
		}

		if (hex.length == 4) hex = '#' + hex[1] + hex[1] + hex[2] + hex[2] + hex[3] + hex[3];
		var colors = [];

		for(var i=1, j=0; i<7; i+=2) colors[j++] = ('0x' + hex.slice(i, i+2)) - 0;
		return colors;
	}

	/**
	 * 渲染一个安全输入框
	 * @param options 配置项
	 * 如果options是一个string类型(输入框id),则直接返回输入框对象
	 * 否则options应该是一个对象,其中必须参数为
	 * @class CardWrap
	 */
	CardWrap.prototype.safeInput = function(options){
		var addMethod = function (obj) {
			//获取文本
			obj.val = function(){return obj.getText();};
			//启用控件
			obj.enable = function(){obj.EnableEdit(1);};
			//禁用控件
			obj.disable = function(){obj.EnableEdit(0);};
			//清空内容
			obj.empty = function(){obj.ClearEdit();};
			obj.color = [0, 0, 0];
			obj.bgcolor = [255, 255, 255];

			//设置背景色
			obj.setBgColor = function(hexColor){
				var rgb = hexToRgb(hexColor);
				if(!rgb) return ;
				var nw = obj.color.concat(rgb);
				obj.SetColor(nw[0], nw[1], nw[2], nw[3], nw[4], nw[5]);
				obj.bgcolor = rgb;
			};

			//设置字体色
			obj.setColor = function(hexColor){
				var rgb = hexToRgb(hexColor);
				if(!rgb) return ;
				var nw = rgb.concat(obj.bgcolor);
				obj.SetColor(nw[0], nw[1], nw[2], nw[3], nw[4], nw[5]);
				obj.color = rgb;
			};

			return obj;
		};

		if(typeof options == 'string') {
			var objSrc = document.getElementById(options);
			if(objSrc) return addMethod(objSrc);
			return objSrc;
		}

		if(!options) return null;
		options = options||{};
		if(typeof options.error != 'function') options.error = noop;

		if(plugin.error) {
			options.error.call(options, plugin.error);
			return null;
		}

		if(!plugin.activeX){//不支持ActiveXObject控件
			options.error.call(options, {code:-100, message:Const.errorDesc[-100]});
			return null;
		}

		//设置默认值
		options.width = options.width|| 150;
		options.height = options.height|| 20;
		options.render = options.render || document.body;
		options.id = options.id || ('mdc-object-' + Math.random()).replace('0.', '');
		options['class'] = options.className; //同时支持className属性和class(class为关键字,使用时要加'')

		if(typeof options.render == 'string') {
			options.render = document.getElementById(options.render);
		}
		if(!options.render) return null;

		var exist = true;
		var obj = document.getElementById(options.id);
		if(!obj) { obj = document.createElement("object");  exist = false; }
		obj.setAttribute('classid', 'CLSID:3F6D7581-6684-4113-A9BE-CD0DB641EC05');

		//设置内部参数
		var params;
		if(typeof options.params == 'function') {
			params = options.params.call(obj, obj);

		} else if(typeof options.params == 'object') {
			params = options.params;
		}

		if(params) {
			var param;
			for(var it in params) {
				param = document.createElement('param');
				param.setAttribute('name', it);
				param.setAttribute('value', params[it]);
				obj.appendChild(param);
			}
		}
		// end 设置内部参数

		var attrs = {classid:true, render:true, error:true, params:true};

		//设置属性和事件
		if (document.addEventListener) {
			for(var it in options){
				if(!attrs[it] && options[it]){
					if(/^on\w+/i.test(it)){
						if(typeof options[it] == 'function') obj.addEventListener(it.replace(/^on/i,''), options[it], false);
					} else {
						obj.setAttribute(it, options[it]);
					}
				}
			}
		} else if (document.attachEvent) {
			for(var it in options && options[it]){
				if(!attrs[it]){
					if(/^on\w+/i.test(it)){
						if(typeof options[it] == 'function') obj.attachEvent(it, options[it]);
					} else {
						obj.setAttribute(it, options[it]);
					}
				}
			}
		}
		//end 设置属性和事件

		if(!exist) options.render.appendChild(obj);

		obj.id = options.id;
		return addMethod(obj);
	};

})(window);