package com.xdja.pki.gmssl.sdf.pcie;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.sdf.AbstractSdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.*;
import com.xdja.pcie.sdf.jni.SdfApi;
import com.xdja.pcie.sdf.jni.alg.AlgId;
import com.xdja.pcie.sdf.jni.bean.EccCipher;
import com.xdja.pcie.sdf.jni.bean.EccPrivateKey;
import com.xdja.pcie.sdf.jni.bean.EccPublicKey;
import com.xdja.pcie.sdf.jni.bean.EccSignature;

import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.PublicKey;

public class PcieSdfSDK extends AbstractSdfSDK {

    private SdfApi sdfApi = new SdfApi();

    @Override
    public void init() throws SdfSDKException {
        //打开设备
        int ret = sdfApi.openDevice(dev);
        checkRet("openDevice", ret);
        //打开 session
        ret = sdfApi.openSession(dev[0], ses);
        checkRet("openSession", ret);
    }

    @Override
    public void release() throws SdfSDKException {
        int ret = sdfApi.closeSession(ses[0]);
        checkRet("closeSession", ret);
        ret = sdfApi.closeDevice(dev[0]);
        checkRet("closeDevice", ret);
    }

    @Override
    public byte[] generateRandom(int uiLength) throws SdfSDKException {
        byte[] pucRandom = new byte[uiLength];
        int ret = sdfApi.generateRandom(ses[0], uiLength, pucRandom);
        checkRet("generateRandom", ret);
        return pucRandom;
    }

    @Override
    public SdfECCKeyPair generateKeyPairEcc() throws SdfSDKException {
        //ECC签名公钥  ECC加密公钥
        EccPublicKey pucPublicKey = new EccPublicKey();
        EccPrivateKey pucPrivateKey = new EccPrivateKey();
        int ret = sdfApi.generateKeyPairEcc(ses[0], AlgId.SGD_SM2, GMSSLX509Utils.ECC_KEY_BITS, pucPublicKey, pucPrivateKey);
        checkRet("generateKeyPairEcc", ret);
        byte[] x = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucPublicKey.getX());
        byte[] y = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucPublicKey.getY());
        byte[] k = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucPrivateKey.getK());
        return new SdfECCKeyPair(
                new SdfECCPublicKey(x, y),
                new SdfECCPrivateKey(k)
        );
    }

    @Override
    public void getPrivateKeyAccessRight(int index, byte[] password) throws SdfSDKException {
        int ret = sdfApi.getPrivateKeyAccessRight(ses[0], index, password, password.length);
        checkRet("getPrivateKeyAccessRight", ret);
    }

    @Override
    public void releasePrivateKeyAccessRight(int index) throws SdfSDKException {
        int ret = sdfApi.releasePrivateKeyAccessRight(ses[0], index);
        checkRet("releasePrivateKeyAccessRight", ret);
    }

    public PublicKey importSM2SignKey(KeyStore keyStore, char[] pw, int index) throws SdfSDKException {
        return importSM2Key(keyStore, pw, index, true);
    }

    public PublicKey importSM2EnvKey(KeyStore keyStore, char[] pw, int index) throws SdfSDKException {
        return importSM2Key(keyStore, pw, index, false);
    }

    private PublicKey importSM2Key(KeyStore keyStore, char[] pw, int index, boolean isSign) throws SdfSDKException {
        try {
            String alias = keyStore.aliases().nextElement();
            PrivateKey privateKey = (PrivateKey) keyStore.getKey(alias, pw);
            PublicKey publicKey = keyStore.getCertificate(alias).getPublicKey();
            importSM2PrivateKey(privateKey, index, isSign);
            importSM2PublicKey(publicKey, index, isSign);
            return publicKey;
        } catch (Exception e) {
            throw new SdfSDKException(e);
        }
    }

    public void importSM2PublicKey(PublicKey key, int index, boolean isSign) throws SdfSDKException {
        String method = isSign ? "sm2ImportSignpub" : "sm2ImportEncpub";
        EccPublicKey pucPublicKey = PcieSdfSDKUtils.generateEccPublicKey(logger, key);

        int ret;
        if (isSign) {
            ret = sdfApi.sm2ImportSignpub(ses[0], index, pucPublicKey);
        } else {
            ret = sdfApi.sm2ImportEncpub(ses[0], index, pucPublicKey);
        }
        checkRet(method, ret);

        EccPublicKey ecPublicKey_exp = new EccPublicKey();
        if (isSign) {
            ret = sdfApi.exportSignPublicKeyEcc(ses[0], index, ecPublicKey_exp);
        } else {
            ret = sdfApi.exportEncPublicKeyEcc(ses[0], index, ecPublicKey_exp);
        }
        checkRet("exportSignPublicKeyEcc", ret);
    }

    public void importSM2PrivateKey(PrivateKey key, int index, boolean isSign) throws SdfSDKException {
        String method = isSign ? "sm2ImportSignpri" : "sm2ImportEncpri";
        EccPrivateKey pucPrivateKey = PcieSdfSDKUtils.generateEccPrivateKey(logger, key);
        int ret;
        if (isSign) {
            ret = sdfApi.sm2ImportSignpri(ses[0], index, pucPrivateKey, 0);
        } else {
            ret = sdfApi.sm2ImportEncpri(ses[0], index, pucPrivateKey, 0);
        }
        checkRet(method, ret);
    }

    @Override
    public SdfECCSignature internalSignECC(int index, byte[] data) throws SdfSDKException {
        EccSignature pucSignature = new EccSignature();
        int ret = sdfApi.internalSignEcc(ses[0], index, data, data.length, pucSignature);
        checkRet("internalSignECC", ret);
        byte[] r = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucSignature.getR());
        byte[] s = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucSignature.getS());
        return new SdfECCSignature(r, s);
    }

    @Override
    public SdfECCPublicKey exportSignPublicKeyEcc(int index) throws SdfSDKException {
        EccPublicKey pucPublicKey = new EccPublicKey();
        int ret = sdfApi.exportSignPublicKeyEcc(ses[0], index, pucPublicKey);
        checkRet("externalVerifyECC", ret);
        byte[] x = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucPublicKey.getX());
        byte[] y = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucPublicKey.getY());
        return new SdfECCPublicKey(x, y);
    }

    @Override
    public void externalVerifyECC(SdfECCPublicKey publicKey, byte[] data, SdfECCSignature sdfECCSignature) throws SdfSDKException {
        byte[] r = GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(sdfECCSignature.getR());
        byte[] s = GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(sdfECCSignature.getS());
        EccSignature pucSignature = new EccSignature(r, s);
        byte[] x = GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(publicKey.getX());
        byte[] y = GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(publicKey.getY());
        EccPublicKey pucPublicKey = new EccPublicKey(GMSSLX509Utils.ECC_KEY_BITS, x, y);
        int ret = sdfApi.externalVerifyEcc(ses[0], AlgId.SGD_SM2, pucPublicKey, data, data.length, pucSignature);
        checkRet("externalVerifyECC", ret);
    }

    @Override
    public SdfECCPublicKey exportEncPublicKeyEcc(int index) throws SdfSDKException {
        EccPublicKey pucPublicKey = new EccPublicKey();
        int ret = sdfApi.exportEncPublicKeyEcc(ses[0], index, pucPublicKey);
        checkRet("exportEncPublicKeyEcc", ret);
        byte[] x = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucPublicKey.getX());
        byte[] y = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucPublicKey.getY());
        return new SdfECCPublicKey(x, y);
    }

    @Override
    public SdfECCCipher externalEncryptECC(SdfECCPublicKey publicKey, byte[] data) throws SdfSDKException {
        EccCipher pucEncData = new EccCipher();
        EccPublicKey pucPublicKey = PcieSdfSDKUtils.generateEccPublicKey(logger, publicKey.getX(), publicKey.getY());
        int ret = sdfApi.externalEncryptEcc(ses[0], AlgId.SGD_SM2, pucPublicKey, data, data.length, pucEncData);
        checkRet("externalEncryptECC", ret);

        GMSSLByteArrayUtils.printHexBinary(logger, "enc data x", pucEncData.getX());
        GMSSLByteArrayUtils.printHexBinary(logger, "enc data y", pucEncData.getY());
        byte[] x = GMSSLByteArrayUtils.filterByteArrayZeroInHead(pucEncData.getX());
        byte[] y = GMSSLByteArrayUtils.filterByteArrayZeroInHead(pucEncData.getY());
        return new SdfECCCipher(x, y, pucEncData.getM(), pucEncData.getL(), pucEncData.getC());
    }

    @Override
    public byte[] internalDecryptECC(int index, int len, SdfECCCipher sdfECCCipher) throws SdfSDKException {
        byte[] x = GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(sdfECCCipher.getX());
        byte[] y = GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(sdfECCCipher.getY());
        EccCipher pucEncData = new EccCipher(x, y, sdfECCCipher.getM(), sdfECCCipher.getL(), sdfECCCipher.getC());
        byte[] pucData = new byte[len];
        int[] puiDataLength = {len};
        int ret = sdfApi.internalDecryptEcc(ses[0], index, AlgId.SGD_SM2, pucEncData, pucData, puiDataLength);
        checkRet("internalDecryptECC", ret);
        return pucData;
    }

    @Override
    public long[] importKey(byte[] pucKey) throws SdfSDKException {
        long[] phKeyHandle = {0};
        int ret = sdfApi.importKey(ses[0], pucKey, pucKey.length, phKeyHandle);
        checkRet("importKey", ret);
        return phKeyHandle;
    }

    @Override
    public byte[] encrypt(long[] phKeyHandle, byte[] pucIv, byte[] pucData) throws SdfSDKException {
        byte[] pucEncData = new byte[pucData.length];
        int[] puiEncDataLength = {128};
        int ret = sdfApi.encrypt(ses[0], phKeyHandle[0], AlgId.SGD_SM4_CBC, pucIv, pucData, pucData.length, pucEncData, puiEncDataLength);
        checkRet("encrypt", ret);
        return pucEncData;
    }

    @Override
    public byte[] decrypt(long[] phKeyHandle, byte[] pucIv, byte[] pucEncData) throws SdfSDKException {
        byte[] pucData = new byte[pucEncData.length];
        int[] puiDataLength = {128};
        int ret = sdfApi.decrypt(ses[0], phKeyHandle[0], AlgId.SGD_SM4_CBC, pucIv, pucEncData, pucEncData.length, pucData, puiDataLength);
        checkRet("decrypt", ret);
        return pucData;
    }

    @Override
    public void destroyKey(long[] phKeyHandle) throws SdfSDKException {
        int ret = sdfApi.destroyKey(ses[0], phKeyHandle[0]);
        checkRet("destroyKey", ret);
    }

    @Override
    public void sm3HashInit() throws SdfSDKException {
        int ret = sdfApi.hashInit(ses[0], AlgId.SGD_SM3, null, null, 0);
        checkRet("sm3HashInit", ret);
    }

    @Override
    public void sm3HashInit(byte[] pucID, SdfECCPublicKey publicKey) throws SdfSDKException {
        EccPublicKey pucPublicKey = PcieSdfSDKUtils.generateEccPublicKey(logger, publicKey.getX(), publicKey.getY());
        int ret = sdfApi.hashInit(ses[0], AlgId.SGD_SM3, pucPublicKey, pucID, pucID.length);
        checkRet("sm3HashInit", ret);
    }

    @Override
    public void sm3HashUpdate(byte[] data) throws SdfSDKException {
        int ret = sdfApi.hashUpdate(ses[0], data, data.length);
        checkRet("sm3HashUpdate", ret);
    }

    @Override
    public byte[] sm3HashFinal() throws SdfSDKException {
        byte[] pucHash = new byte[32];
        int[] pucHashLen = {32};
        int ret = sdfApi.hashFinal(ses[0], pucHash, pucHashLen);
        checkRet("sm3HashFinal", ret);
        return pucHash;
    }

    @Override
    public void calculateMac(int uiKeyIndex) throws SdfSDKException {
        int ret;
        int uiKeyBits_SES = 128;

        long[] phKeyHandle = {0};
        EccCipher pucKey = new EccCipher();

        //对称算法运算类函数缓冲区指针定义
        byte[] pucIV = new byte[1024];
        byte[] pucEncDate = new byte[1024];
        int[] pucEncDateLen = {1024};

        //要加密的数据 和数据长度
        String pucDate_t = "12345678123456781234567812345678";
        byte[] pucDate = pucDate_t.getBytes();

        ret = sdfApi.generateKeyWithIpkEcc(ses[0], uiKeyIndex, uiKeyBits_SES, pucKey, phKeyHandle);
        checkRet("generateKeyWithIpkEcc", ret);
        ret = sdfApi.calculateMac(ses[0], phKeyHandle[0], AlgId.SGD_SM4_MAC, pucIV, pucDate, pucDate.length, pucEncDate, pucEncDateLen);
        checkRet("calculateMac", ret);
    }

    @Override
    public byte[] sm3Hmac(byte[] dataIn, byte[] key) throws SdfSDKException {
        int ret;
        byte[] mac = new byte[32];
        ret = sdfApi.sm3Hmac(ses[0], dataIn, dataIn.length, key, key.length, mac);
        checkRet("sm3HmacDemo", ret);
        return mac;
    }

}
