package com.xdja.ra.sdk;

import com.alibaba.fastjson.JSON;
import com.xdja.pki.apache.client.core.ClientErrorEnum;
import com.xdja.pki.apache.client.core.ClientKeyStoreConfig;
import com.xdja.pki.apache.client.result.AdaptClientResult;
import com.xdja.pki.apache.client.result.RAClientResult;
import com.xdja.pki.apache.client.utils.ApacheClientHttpUtils;
import com.xdja.ra.bean.*;
import com.xdja.ra.constant.SdkConstants;
import com.xdja.ra.ennum.ReqMethodEnum;
import com.xdja.ra.ennum.SignatureAlgorithmEnum;
import com.xdja.ra.error.ErrorEnum;
import com.xdja.ra.utils.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;

import java.io.UnsupportedEncodingException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.URLEncoder;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;

/**
 * RA-SDK 接口
 *
 * @author ggp
 * @Date 2019/11/1 13:54
 */
public class SDKService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final Object LOCK_TRANS_ID = "transId";

    private static SDKService sdkService;

    public static SDKService getInstance() {
        return sdkService;
    }

    public static Config config;

    public SDKService(Config config) {
        SDKService.config = config;
    }

    /**
     * 初始化类 init
     * @param serverIp
     * @param serverPort
     * @param prikeyPath
     * @param prikeyPwd
     * @param trustCertPath
     * @param systemFlag
     */
    public SDKService(String serverIp, String serverPort, String prikeyPath, String prikeyPwd, String trustCertPath, String systemFlag) {
        try {
            config = new Config();
            config.setServerIp(serverIp);
            config.setServerPort(serverPort);
            config.setPrikeyPath(prikeyPath);
            config.setPriKeyPwd(prikeyPwd);
            config.setTrustCertPath(trustCertPath);
            config.setSystemFlag(systemFlag);
            this.init();   // 构造初始化数据
            SignUtils.sign(SDKService.config.getSignName(), SDKService.config.getPrivateKey(), "init");
            JSON.toJSONString(new UserInfo());
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            throw new RuntimeException("初始化SDK失败", e);
        }
    }

    /**
     * 初始化类 init + https
     * @param serverIp
     * @param serverPort
     * @param isHttps
     * @param prikeyPath
     * @param prikeyPwd
     * @param trustCertPath
     * @param systemFlag
     */
    public SDKService(String serverIp, String serverPort, boolean isHttps, String prikeyPath, String prikeyPwd, String trustCertPath, String systemFlag) {
        try {
            new SDKService(serverIp,serverPort,prikeyPath,prikeyPwd,trustCertPath,systemFlag);
            config.setHttps(isHttps);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            throw new RuntimeException("初始化SDK失败", e);
        }
    }

    /**
     * 初始化类 init + https + keyStoreConfig
     *
     * @param serverIp
     * @param serverPort
     * @param isHttps
     * @param systemFlag
     * @param keyStoreConfig
     */
    public SDKService(String serverIp, String serverPort, boolean isHttps, String systemFlag, KeyStoreConfig keyStoreConfig) {
        try {
            new SDKService(serverIp, serverPort, keyStoreConfig.getClientKeyStorePath(), keyStoreConfig.getClientKeyStorePwd(), null, systemFlag);
            config.setHttps(isHttps);
            config.setKeyStoreConfig(keyStoreConfig);
            //todo 从P12文件中获取CA信任证书链 待验证
            X509Certificate[] trustCerts = KeyStoreUtils.getTrustCertFromP12(keyStoreConfig.getTrustKeyStorePwd(), keyStoreConfig.getClientKeyStorePath());
            config.setTrustCert(trustCerts);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            throw new RuntimeException("初始化SDK失败", e);
        }
    }

    /**
     * 初始化类 init + https + keyStoreConfig
     * @param serverIp
     * @param serverPort
     * @param isHttps
     * @param prikeyPath
     * @param prikeyPwd
     * @param trustCertPath
     * @param systemFlag
     * @param keyStoreConfig
     */
    public SDKService(String serverIp, String serverPort, boolean isHttps, String prikeyPath, String prikeyPwd, String trustCertPath, String systemFlag, KeyStoreConfig keyStoreConfig) {
        try {
            new SDKService(serverIp,serverPort,prikeyPath,prikeyPwd,trustCertPath,systemFlag);
            config.setHttps(isHttps);
            config.setKeyStoreConfig(keyStoreConfig);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            throw new RuntimeException("初始化SDK失败", e);
        }
    }

    /**
     * 初始化类
     * @param serverIp
     * @param serverPort
     * @param isHttps
     * @param prikeyPath
     * @param prikeyPwd
     * @param trustCertPath
     * @param systemFlag
     * @param keyStoreConfig
     * @param isUseHsm
     */
    public SDKService(String serverIp, String serverPort, boolean isHttps, String prikeyPath, String prikeyPwd, String trustCertPath, String systemFlag,KeyStoreConfig keyStoreConfig,boolean isUseHsm) {
        new SDKService(serverIp,serverPort,isHttps,prikeyPath,prikeyPwd,trustCertPath,systemFlag,keyStoreConfig);
        config.setUseHsm(isUseHsm);
    }

    private void init() {
        try {
            String baseUrl = config.getServerIp() + ":" + config.getServerPort() + "/ra-openapi";
            config.setRaBaseUrl(baseUrl);
            X509Certificate cert = (X509Certificate) KeyStoreUtils.getPublicKeyFromP12(null, config.getPrikeyPath(), config.getPriKeyPwd());
            config.setSystemCert(cert);
            config.setUserCertSn(cert.getSerialNumber().toString(16));
            config.setCertSignAlgOid(cert.getSigAlgOID());
            if (null == cert.getSigAlgName()) {
                config.setSignName(SignatureAlgorithmEnum.getName(config.getCertSignAlgOid()));
            } else {
                config.setSignName(cert.getSigAlgName());
            }
            if(null != config.getTrustCertPath()) {
                X509Certificate[] caCerts;
                List<X509Certificate> certs = null;
                try {
                    String trustCertP7b = FileUtils.read(config.getTrustCertPath());
                    byte[] base64Bytes = Base64.decode(trustCertP7b);
                    certs = CertUtils.getCertListFromP7b(base64Bytes);
                } catch (Exception e) {
                    byte[] trustCertP7b = FileUtils.readByBinary(config.getTrustCertPath());
                    certs = SignedDataUtils.resolveCertChain(trustCertP7b);
                }
                caCerts = new X509Certificate[certs.size()];
                for (int i = 0; i < certs.size(); i++) {
                    caCerts[i] = certs.get(i);
                }
                config.setTrustCert(caCerts);
            }
            PrivateKey privateKey = SignUtils.getPrivateKeyFromP12(config.getPrikeyPath(), config.getPriKeyPwd().toCharArray());
            config.setPrivateKey(privateKey);
            sdkService = new SDKService(config);
        } catch (Exception e) {
            logger.error("初始化SDKService异常", e);
        }
    }

    /**
     * 用户注册
     *
     * @param userType
     * @param userInfo
     * @return
     */
    public Result registUser(Integer userType, UserInfo userInfo) {
        Result result;
        try {
            String reqUrl = "/v1/ra-openapi/user/" + userType;
            String source = reqUrl + JSON.toJSONString(userInfo);
            result = buildData(reqUrl, source, ReqMethodEnum.POST.name, JSON.toJSONString(userInfo));
            result.setInfo(null);
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 用户信息查询
     *
     * @param userType
     * @param identType
     * @param identNumber
     * @return
     */
    public Result getUserInfo(Integer userType, Integer identType, String identNumber) {
        try {
            String reqUrl = "/v1/ra-openapi/user/" + userType + "/" + identType + "/" + identNumber;
            return getUser(reqUrl);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 根据签名证书sn获取用户信息
     *
     * @param sn
     * @return
     */
    public Result getUserInfo(String sn) {
        try {
            String reqUrl = "/v1/ra-openapi/user/" + sn;
            return getUser(reqUrl);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    private Result getUser(String reqUrl) throws Exception {
        String source = reqUrl;
        Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
        if (result.isSuccess() && null != result.getInfo()) {
            UserInfo userInfo = JSON.parseObject((new String((byte[])result.getInfo())), UserInfo.class);
            result.setInfo(userInfo);
        } else {
            result.setInfo(null);
        }
        return result;
    }

    /**
     * 用户信息更新
     *
     * @param userType
     * @param identType
     * @param identNumber
     * @return
     */
    public Result updateUserInfo(Integer userType, Integer identType, String identNumber, UserInfo userInfo) {
        try {
            String reqUrl = "/v1/ra-openapi/user/" + userType + "/" + identType + "/" + identNumber;
            String source = reqUrl + JSON.toJSONString(userInfo);
            Result result = buildData(reqUrl, source, ReqMethodEnum.PUT.name, JSON.toJSONString(userInfo));
            result.setInfo(null);
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 修改用户状态
     *
     * @param userType
     * @param identType
     * @param identNumber
     * @param userStatus
     * @return
     */
    public Result updateUserStatus(Integer userType, Integer identType, String identNumber, Integer userStatus) {
        try {
            String reqUrl = "/v1/ra-openapi/user/" + userType + "/" + identType + "/" + identNumber + "/" + userStatus;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.PUT.name, null);
            result.setInfo(null);
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 获取RA系统BaseDn
     *
     * @return
     */
    public Result getRASystemBaseDn() {
        try {
            String reqUrl = "/v1/ra-openapi/apply/baseDn";
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess() && null != result.getInfo()) {
                result.setInfo(new String((byte[])result.getInfo()));
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 获取用户模板信息
     *
     * @param systemFlag
     * @return
     */
    public Result getTemplateInfoList(String systemFlag) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/template/" + systemFlag;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess() && null != result.getInfo()) {
                List<TemplateInfo> templateInfoList = JSON.parseArray(new String((byte[])result.getInfo()), TemplateInfo.class);
                result.setInfo(templateInfoList);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }

    }

    /**
     * 签发申请-发起
     *
     * @param issueApply
     * @return
     */
    public Result startIssueUserCertApply(IssueApply issueApply) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/issue";
            return getApplyResult(JSON.toJSONString(issueApply), reqUrl);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 更新申请-发起
     *
     * @param updateApply
     * @return
     */
    public Result startUpdateUserCertApply(UpdateApply updateApply) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/update";
            return getApplyResult(JSON.toJSONString(updateApply), reqUrl);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 撤销申请-发起
     *
     * @param revokeApply
     * @returne
     */
    public Result startRevokeUserCertApply(RevokeApply revokeApply) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/revoke";
            return getApplyResult(JSON.toJSONString(revokeApply), reqUrl);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 恢复申请-发起
     *
     * @param recoveryApply 发起恢复申请时使用的类对象
     * @return ApplyRep类型的签发申请返回对象
     */
    public Result startRecoveryUserCertApply(RecoveryApply recoveryApply) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/recovery";
            return getApplyResult(JSON.toJSONString(recoveryApply), reqUrl);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 冻结/解冻申请-发起 (第六迭代)
     *
     * @param freezeApply 发起冻结/解冻申请时使用的类对象
     * @return ApplyRep类型的签发申请返回对象
     */
    public Result startFreezeOrUnFreezeUserCertApply(FreezeApply freezeApply) {
        try {
            //true-冻结  false-解冻
            String reqUrl = "/v1/ra-openapi/apply/freeze";
            return getApplyResult(JSON.toJSONString(freezeApply), reqUrl);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 申请审核
     *
     * @param applyType  申请类型	1-签发申请 2-更新申请 3-撤销申请 4-恢复申请
     * @param applyNo    申请编号	发起申请时，返回的申请编号
     * @param checkApply 审核申请对象	审核各种申请审核类对象
     * @return DoubleCode类型的两码
     */
    public Result checkUserCertApply(Integer applyType, String applyNo, CheckApply checkApply) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/check/" + applyType + "/" + applyNo;
            String source = reqUrl + JSON.toJSONString(checkApply);
            Result result = buildData(reqUrl, source, ReqMethodEnum.PUT.name, JSON.toJSONString(checkApply));
            if (result.isSuccess() && null != result.getInfo()) {
                DoubleCode doubleCode = JSON.parseObject(new String((byte[])result.getInfo()), DoubleCode.class);
                result.setInfo(doubleCode);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 申请制证执行 兼容无 keyFormat 参数
     *
     * @param applyType
     * @param applyNo
     * @param userCertReq
     * @return
     */
    public Result generateUserCert(Integer applyType, String applyNo, UserCertReq userCertReq){
        return generateUserCert(applyType, applyNo, userCertReq, null);
    }
    /**
     * 申请制证执行
     *
     * @param
     * @param applyType   申请类型 1-签发申请 2-更新申请 4-恢复申请
     * @param applyNo     申请编号 发起申请时，返回的申请编号
     * @param userCertReq 用户证书生成对象 包含生成证书必须的两码和申请书信息
     * @return UserCertRep类型的用户证书生成响应对象
     */
    public Result generateUserCert(Integer applyType, String applyNo, UserCertReq userCertReq, Integer keyFormat) {
        Result result = new Result();
        if (applyType == null || userCertReq == null || StringUtils.isAnyBlank(applyNo, userCertReq.getAuthCode(), userCertReq.getRefCode())) {
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }
        if (applyType != SdkConstants.CERT_APPLY_TYPE_ISSUE_1 && applyType != SdkConstants.CERT_APPLY_TYPE_UPDATE_2
                && applyType != SdkConstants.CERT_APPLY_TYPE_RECOVERY_4) {
            return Result.failure(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
        }

        if(keyFormat == null){
            keyFormat = SdkConstants.KEY_FORMAT_0010_1;
        }else {
            if (!keyFormat.equals(SdkConstants.KEY_FORMAT_0010_1) && !keyFormat.equals(SdkConstants.KEY_FORMAT_0016_2) && !keyFormat.equals(SdkConstants.KEY_FORMAT_0016_3)) {
                return Result.failure(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
            }
            if (!applyType.equals(SdkConstants.CERT_APPLY_TYPE_ISSUE_1) && !applyType.equals(SdkConstants.CERT_APPLY_TYPE_UPDATE_2)) {
                return Result.failure(ErrorEnum.APPLY_TYPE_CANNOT_SUPPORT_KEY_FORMAT);
            }
        }

        //事务控制ID  modify by wangtf nanotime+随机数，解决高并发下事务ID重复问题
        String transId;
        synchronized (LOCK_TRANS_ID){
            transId = String.valueOf(System.nanoTime())+ (ThreadLocalRandom.current().nextLong(0,9999999) + 90000000);
        }
//      String transId = UUID.randomUUID().toString().replace("-", "");
        RaCmpApi raCmpApi = new RaCmpApi();
        Result sdkResult;
        try {
            KeyStoreConfig keyStoreConfig = config.getKeyStoreConfig();
            ClientKeyStoreConfig clientKeyStoreConfig = null;
            if (keyStoreConfig!=null){
                clientKeyStoreConfig = new ClientKeyStoreConfig();
                BeanUtils.copyProperties(keyStoreConfig, clientKeyStoreConfig);
            }
            sdkResult = raCmpApi.sendCertReqMessage(applyType, transId, config.getSystemFlag(), config.getCertSignAlgOid(), applyNo, userCertReq, keyFormat,clientKeyStoreConfig);
        } catch (Exception e) {
            logger.error("SDK内部异常", e);
            return Result.failure(ErrorEnum.SDK_INNER_EXCEPTION);
        }
        if(logger.isDebugEnabled()) {
            logger.debug("SDKService.generateUserCert.result:{}", SdkJsonUtils.object2Json(sdkResult.getInfo()));
        }
        return getCmpResult(result, sdkResult);
    }

    /**
     * 证书签发确认消息
     *
     * @param applyNo
     * @return
     */
    public Result certApplyConfirmMsg(String applyNo) {
        Result result = new Result();
        RaCmpApi raCmpApi = new RaCmpApi();
        Result sdkResult;
        KeyStoreConfig keyStoreConfig = config.getKeyStoreConfig();
        ClientKeyStoreConfig clientKeyStoreConfig = null;
        if (keyStoreConfig!=null){
            clientKeyStoreConfig = new ClientKeyStoreConfig();
            BeanUtils.copyProperties(keyStoreConfig,clientKeyStoreConfig);
        }
        try {
            sdkResult = raCmpApi.sendConfirmMessage(config.getCertSignAlgOid(), config.getSystemFlag(), applyNo,clientKeyStoreConfig);
        } catch (Exception e) {
            return Result.failure(ErrorEnum.SDK_INNER_EXCEPTION);
        }
        return getCmpResult(result, sdkResult);
    }

    /**
     * 申请制证 确认消息接口融合
     *
     * @param applyType
     * @param applyNo
     * @param userCertReq
     * @return
     */
    public Result generateUserCertAndConfirm(Integer applyType, String applyNo, UserCertReq userCertReq, Integer keyFormat) {

        Result result = generateUserCert(applyType, applyNo, userCertReq, keyFormat);
        if (result.isSuccess()) {
            Result confirmResult = certApplyConfirmMsg(applyNo);
            if (!confirmResult.isSuccess()) {
                return confirmResult;
            }
        }
        return result;

    }
    /**
     * 证书签发错误消息
     *
     * @param applyNo
     * @param errorMsg
     * @return
     */
    public Result certApplyErrorMsg(String applyNo, ErrorMsg errorMsg) {
        Result result = new Result();
        RaCmpApi raCmpApi = new RaCmpApi();
        KeyStoreConfig keyStoreConfig = config.getKeyStoreConfig();
        ClientKeyStoreConfig clientKeyStoreConfig = null;
        if (keyStoreConfig!=null){
            clientKeyStoreConfig = new ClientKeyStoreConfig();
            BeanUtils.copyProperties(keyStoreConfig,clientKeyStoreConfig);
        }
        Result sdkResult;
        try {
            sdkResult = raCmpApi.sendErrorMessage(applyNo, config.getCertSignAlgOid(), config.getSystemFlag(), errorMsg.getErrorMsg(), errorMsg.getErrorCode(), clientKeyStoreConfig);
        } catch (Exception e) {
            return Result.failure(ErrorEnum.SDK_INNER_EXCEPTION);
        }
        return getCmpResult(result, sdkResult);
    }


    /**
     * 查询申请
     *
     * @param applyType 申请类型	1-签发申请 2-更新申请 3-撤销申请 4-恢复申请
     * @param applyNo   申请编号	发起申请时，返回的申请编号
     * @return CertApplyInfo类型的证书申请信息 根据申请类型的不同 返回该结构体中不同的类属性内容
     */
    public Result getUserCertApplyInfo(Integer applyType, String applyNo) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/" + applyType + "/" + applyNo;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess() && null != result.getInfo()) {
                CertApplyInfo certApplyInfo = JSON.parseObject(new String((byte[])result.getInfo()), CertApplyInfo.class);
                result.setInfo(certApplyInfo);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 查询申请状态
     *
     * @param applyNo
     * @return
     */
    public Result baseCertApplyQuery(String applyNo) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/" + applyNo;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess() && null != result.getInfo()) {
                BaseCertApply baseCertApply = JSON.parseObject(new String((byte[])result.getInfo()), BaseCertApply.class);
                result.setInfo(baseCertApply);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 更新证书申请信息
     *
     * @param applyType
     * @param applyNo
     * @param editCertApplyInfo
     * @return
     */
    public Result editUserCertApplyInfo(Integer applyType, String applyNo, EditCertApplyInfo editCertApplyInfo) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/" + applyType + "/" + applyNo;
            String source = reqUrl + JSON.toJSONString(editCertApplyInfo);
            return buildData(reqUrl, source, ReqMethodEnum.PUT.name, JSON.toJSONString(editCertApplyInfo));
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 查询用户证书
     *
     * @param userType
     * @param identType
     * @param identNumber
     * @return
     */
    public Result queryUserCertInfoList(Integer userType, Integer identType, String identNumber) {
        try {
            String reqUrl = "/v1/ra-openapi/cert/" + userType + "/" + identType + "/" + identNumber;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess() && null != result.getInfo()) {
                List<UserCertInfo> userCertInfoList = JSON.parseArray(new String((byte[])result.getInfo()), UserCertInfo.class);
                result.setInfo(userCertInfoList);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 查询证书详情
     *
     * @param certSn
     * @return
     */
    public Result getUserCertBaseInfo(String certSn) {
        try {
            String reqUrl = "/v1/ra-openapi/cert/" + certSn;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess() && null != result.getInfo()) {
                CertBaseInfo certBaseInfo = JSON.parseObject((new String((byte[])result.getInfo())), CertBaseInfo.class);
                result.setInfo(certBaseInfo);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 证书下载
     *
     * @param certSn
     * @return
     */
    public Result downloadUserCertFile(String certSn) {
        try {
            String reqUrl = "/v1/ra-openapi/cert/file/" + certSn;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (!result.isSuccess()) {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }

    }

    /**
     * 根据签名证书sn获取base64格式证书
     *
     * @param certSn
     * @return
     */
    public Result getUserCert(String certSn) {
        try {
            String reqUrl = "/v1/ra-openapi/cert/base64/" + certSn;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess() && null != result.getInfo()) {
                UserCertStr userCertStr = JSON.parseObject(new String((byte[])result.getInfo()), UserCertStr.class);
                result.setInfo(userCertStr);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }

    }

    /**
     * 测试端口连通性
     */
    private boolean isHostConnectivity(String host, int port) {
        Socket socket = new Socket();
        try {
            socket.connect(new InetSocketAddress(host, port));
        } catch (Exception e) {
            logger.debug("isHostConnectivity socket connect {}:{} error", host, port, e);
            return false;
        } finally {
            try {
                socket.close();
            } catch (Exception e) {
                logger.debug("isHostConnectivity socket close {}:{} error", host, port, e);
            }
        }
        return true;
    }

    /**
     * 组装数据  发送请求
     *
     * @param reqUrl
     * @param source
     * @param methodName
     * @return
     * @throws Exception
     */
    private Result buildData(String reqUrl, String source, String methodName, String reqBody) throws UnsupportedEncodingException {
        String[] temp = reqUrl.split("/");
        StringBuilder urlBuilder = new StringBuilder();
        for (String str : temp) {
            urlBuilder.append(URLEncoder.encode(str, "UTF-8")).append("/");
        }
        String url = config.getRaBaseUrl() + urlBuilder.toString();
        String signData = SignUtils.sign(config.getSignName(), config.getPrivateKey(), source);

        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("systemFlag", config.getSystemFlag());
        headerMap.put("signAlg", config.getSignName());
        headerMap.put("signValue", signData);
        headerMap.put("signSn", config.getUserCertSn());

        KeyStoreConfig keyStoreConfig = config.getKeyStoreConfig();
        ClientKeyStoreConfig clientKeyStoreConfig = null;
        if (keyStoreConfig!=null){
            clientKeyStoreConfig = new ClientKeyStoreConfig();
            BeanUtils.copyProperties(keyStoreConfig, clientKeyStoreConfig);
        }
        if (null == reqBody) {
            reqBody = "";
        }
        Result result = new Result();
        try {
            CloseableHttpResponse closeableHttpResponse = ApacheClientHttpUtils.sendApacheClientRequest(reqBody.getBytes(), null, headerMap, url.substring(0, url.length() - 1), "application/json", config.getSignName(), config.isHttps(), methodName,false, clientKeyStoreConfig);
            RAClientResult resultFromClientResponse = AdaptClientResult.getClientResponse(closeableHttpResponse);
            if (!resultFromClientResponse.isSuccess()) {
                result.setErrorMsg(new ErrorMsg(resultFromClientResponse.getErrorMsg().getErrorCode(), resultFromClientResponse.getErrorMsg().getErrorMsg()));
            }
            result.setInfo(resultFromClientResponse.getInfo());
            return result;
        } catch (Exception e) {
            logger.error("请求返回数据解析异常",e);
            result.setErrorMsg(new ErrorMsg(ClientErrorEnum.RESOLVE_CLIENT_RESULT_EXCEPTION.code,ClientErrorEnum.RESOLVE_CLIENT_RESULT_EXCEPTION.desc));
            return result;
        }
    }

    /**
     * 发起申请数据组装
     *
     * @param applyReq
     * @param reqUrl
     * @return
     * @throws Exception
     */
    private Result getApplyResult(String applyReq, String reqUrl) throws Exception {
        String source = reqUrl + applyReq;
        Result result = buildData(reqUrl, source, ReqMethodEnum.POST.name, applyReq);
        if (result.isSuccess() && null != result.getInfo()) {
            ApplyRep applyRep = JSON.parseObject((new String((byte[])result.getInfo())), ApplyRep.class);
            result.setInfo(applyRep);
        } else {
            result.setInfo(null);
        }
        return result;
    }

    private Result getCmpResult(Result result, Result sdkResult) {
        if (!sdkResult.isSuccess()) {
            if (SdkConstants.ERROR_CODE_400.equals(String.valueOf(sdkResult.getErrorMsg().getErrorCode()).substring(0, 3))) {
                return Result.failure(ErrorEnum.SDK_INNER_EXCEPTION);
            } else {
                result.setErrorMsg(sdkResult.getErrorMsg());
                return result;
            }
        }
        return Result.success(sdkResult.getInfo());
    }
}
