package com.xdja.ra.utils;

import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2SignUtils;
import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.digests.SHA1Digest;
import org.bouncycastle.crypto.digests.SHA256Digest;
import org.bouncycastle.crypto.digests.SM3Digest;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMDecryptorProvider;
import org.bouncycastle.openssl.PEMEncryptedKeyPair;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcePEMDecryptorProviderBuilder;
import org.bouncycastle.util.encoders.Base64;

import javax.crypto.Cipher;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.security.*;
import java.util.Enumeration;

/**
 * @author: ggp
 * @Date: 2019/11/2 15:57
 * @Description:
 */
public class SignUtils {

    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    /**
     * 签名
     *
     * @param algName
     * @param key
     * @param source
     * @return
     * @throws Exception
     */
    public static String sign(String algName, PrivateKey key, String source) {
        try {
            Signature signature = Signature.getInstance(algName, BouncyCastleProvider.PROVIDER_NAME);
            signature.initSign(key);
            signature.update(source.getBytes());
            return new String(Base64.encode(signature.sign()));
        } catch (Exception e) {
            throw new RuntimeException("签名失败",e);
        }
    }

    /**
     * 验签
     *
     * @param algName
     * @param key
     * @param source
     * @param sign
     * @return
     * @throws Exception
     */
    public static boolean verify(String algName, PublicKey key, String source, String sign) throws Exception {
        Signature signature = Signature.getInstance(algName, BouncyCastleProvider.PROVIDER_NAME);
        signature.initVerify(key);
        signature.update(source.getBytes());
        return signature.verify(Base64.decode(sign.getBytes()));
    }

    /**
     * 加密
     *
     * @param algName
     * @param key
     * @param source
     * @return
     * @throws Exception
     */
    public static String encryption(String algName, Key key, String source) throws Exception {
        Cipher cipher = Cipher.getInstance(algName, BouncyCastleProvider.PROVIDER_NAME);
        cipher.init(Cipher.ENCRYPT_MODE, key);
        byte[] result = cipher.doFinal(source.getBytes());
        return new String(Base64.encode(result));
    }

    /**
     * 哈希
     *
     * @param algName 算法id 1 sm3  2 sha-1 3 sha256
     * @param source
     * @return
     * @throws Exception
     */
    public static String hash(int algName, String source) throws Exception {
        Digest digest = null;
        switch (algName) {
            case 1:
                digest = new SM3Digest();
                break;
            case 2:
                digest = new SHA1Digest();
                break;
            case 3:
                digest = new SHA256Digest();
                break;
            default:
                throw new RuntimeException("不支持的hash算法id:" + algName);
        }
        byte[] data = source.getBytes();
        digest.update(data, 0, data.length);
        byte[] result = new byte[digest.getDigestSize()];
        digest.doFinal(result, 0);
        return new String(Base64.encode(result));
    }

    /**
     * 从私钥文件中读取私钥
     *
     * @param path
     * @return
     */
    public static PrivateKey getPrivateKey(String path) throws Exception {
        File file = new File(path);
        if (!file.exists()) {
            throw new FileNotFoundException("文件不存在");
        }
        PEMParser pemParser = new PEMParser(new FileReader(path));
        Object object = pemParser.readObject();
        pemParser.close();

        PEMDecryptorProvider decProv = new JcePEMDecryptorProviderBuilder().build(null);
        JcaPEMKeyConverter converter = new JcaPEMKeyConverter().setProvider("BC");
        KeyPair kp;
        if (object instanceof PEMEncryptedKeyPair) {
            kp = converter.getKeyPair(((PEMEncryptedKeyPair) object).decryptKeyPair(decProv));
        } else {
            kp = converter.getKeyPair((PEMKeyPair) object);
        }
        return kp.getPrivate();
    }

    /**
     * 从pfx文件中解析私钥
     *
     * @param path
     * @param password
     * @return
     */
    public static PrivateKey getPrivateKeyFromP12(String path, char[] password) throws Exception {
        File file = new File(path);
        if (!file.exists()) {
            throw new FileNotFoundException("文件不存在");
        }
        KeyStore keyStore = KeyStore.getInstance("PKCS12",new BouncyCastleProvider());
        keyStore.load(new FileInputStream(path), password);
        Enumeration<String> aliases = keyStore.aliases();
        PrivateKey key = null;
        while (aliases.hasMoreElements()) {
            String name =aliases.nextElement();
            key = (PrivateKey) keyStore.getKey(name, password);
            if(null != key){
                return key;
            }
        }
        throw new RuntimeException("解析pfx文件失败");
    }
}
