package com.xdja.simplestorage;

import android.content.Context;
import android.os.Build;
import android.os.Process;
import android.os.SystemClock;
import android.security.KeyPairGeneratorSpec;
import android.security.keystore.KeyGenParameterSpec;
import android.security.keystore.KeyProperties;
import android.util.Log;

import java.io.IOException;
import java.math.BigInteger;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.UnrecoverableEntryException;
import java.security.cert.CertificateException;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Arrays;
import java.util.Calendar;
import java.util.GregorianCalendar;

import javax.security.auth.x500.X500Principal;

/**
 * @author zhangxiaolong@xdja.com <br/>
 * @date 2019/2/21 <br/>
 */
public class KeyStoreUtils {

    static final String KEYSTORE_PROVIDER_ANDROID_KEYSTORE = "AndroidKeyStore";
    /**
     * 秘钥默认长度
     */
    static final int DEFAULT_KEY_SIZE = 224;

    private final static String mAlias = Process.myUid() + "";

    private static KeyStoreUtils instance;
    private Context context;

    private KeyStore ks;

    private KeyStoreUtils(Context context) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
        try {
            ks = KeyStore.getInstance(KEYSTORE_PROVIDER_ANDROID_KEYSTORE);
            ks.load(null);
        } catch (KeyStoreException e) {
            e.printStackTrace();
        } catch (CertificateException e) {
            e.printStackTrace();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 得到Key
     *
     * @param context
     * @return 返回16位的key。此key在应用未被卸载前是同一个。也就是说，每次应用卸载后再安装，调用此方法得到的key才不相同。
     */
    public synchronized static byte[] getKey(Context context) {
        return KeyStoreUtils.getInstance(context).getKey();
    }

    static KeyStoreUtils getInstance(Context context) {
        if (instance == null) {
            synchronized (KeyStoreUtils.class) {
                if (instance == null) {
                    instance = new KeyStoreUtils(context);
                }
            }
        }
        return instance;
    }

    private byte[] key;

    byte[] getKey() {
        if (key == null) {
            if (!isHaveKeyStore()) {
                genKey();
            }
            key = getKeyFromSystem();
        }
        return key;
    }


    /**
     * 产生随机数
     *
     * @return
     */
    private synchronized void genKey() {
        try {
            long start = SystemClock.elapsedRealtime();
            genKeyPair(context);
            long end = SystemClock.elapsedRealtime();
            Log.e("genKey", "gen time: " + (end - start));
        } catch (InvalidAlgorithmParameterException e) {
            e.printStackTrace();
        } catch (NoSuchProviderException e) {
            e.printStackTrace();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
    }

    private byte[] getKeyFromSystem() {
        try {
            KeyStore.Entry entry = ks.getEntry(mAlias, null);
            if (entry == null) {
                return null;
            }
            if (!(entry instanceof KeyStore.PrivateKeyEntry)) {
                return null;
            }
            PublicKey publicKey = ks.getCertificate(mAlias).getPublicKey();
            byte[] keyBytes = publicKey.getEncoded();
            return Arrays.copyOfRange(keyBytes, keyBytes.length - 16, keyBytes.length);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (KeyStoreException e) {
            e.printStackTrace();
        } catch (UnrecoverableEntryException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 创建一个公共和私人密钥,并将其存储使用Android密钥存储库中,因此,只有
     * 这个应用程序将能够访问键。
     *
     * @param context
     * @throws InvalidAlgorithmParameterException
     * @throws NoSuchProviderException
     * @throws NoSuchAlgorithmException
     */
    private KeyPair genKeyPair(Context context) throws
            InvalidAlgorithmParameterException,
            NoSuchProviderException, NoSuchAlgorithmException {
        AlgorithmParameterSpec spec;
        String algorithm;
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            Calendar start = new GregorianCalendar();
            Calendar end = new GregorianCalendar();
            end.add(Calendar.YEAR, 10);
            algorithm = KeyProperties.KEY_ALGORITHM_RSA;
            spec = new KeyPairGeneratorSpec.Builder(context)
                    .setKeySize(DEFAULT_KEY_SIZE)
                    .setKeyType(KeyProperties.KEY_ALGORITHM_EC)
                    .setAlias(mAlias)
                    .setSubject(new X500Principal("CN=" + mAlias))
                    .setSerialNumber(BigInteger.valueOf(DEFAULT_KEY_SIZE))
                    .setStartDate(start.getTime())
                    .setEndDate(end.getTime())
                    .build();
        } else {
            algorithm = KeyProperties.KEY_ALGORITHM_EC;
            spec = new KeyGenParameterSpec.Builder(mAlias, KeyProperties.PURPOSE_ENCRYPT | KeyProperties.PURPOSE_DECRYPT)
                    .setKeySize(DEFAULT_KEY_SIZE)
                    .setUserAuthenticationRequired(false)
                    .setDigests(KeyProperties.DIGEST_NONE)
                    .build();
        }
        KeyPairGenerator kpGenerator = KeyPairGenerator
                .getInstance(algorithm, KEYSTORE_PROVIDER_ANDROID_KEYSTORE);
        kpGenerator.initialize(spec);
        KeyPair kp = kpGenerator.generateKeyPair();
        return kp;
    }


    /**
     * 判断是否创建过秘钥
     *
     * @return
     * @throws KeyStoreException
     * @throws CertificateException
     * @throws NoSuchAlgorithmException
     * @throws IOException
     * @throws UnrecoverableEntryException
     */
    private boolean isHaveKeyStore() {

        try {
            //从Android加载密钥对密钥存储库中
            try {
                KeyStore.Entry entry = ks.getEntry(mAlias, null);
                if (entry != null) {
                    return true;
                }
            } catch (NullPointerException e) {
                e.printStackTrace();
            }
        } catch (KeyStoreException e) {
            e.printStackTrace();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (UnrecoverableEntryException e) {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * 获得本地AndroidKeyStore中的公钥
     *
     * @return
     */
    private PublicKey getLocalPublicKey() {
        try {
            //从Android加载密钥对密钥存储库中
            KeyStore.Entry entry = ks.getEntry(mAlias, null);
            if (entry == null) {
                return null;
            }
            if (!(entry instanceof KeyStore.PrivateKeyEntry)) {
                return null;
            }
            PublicKey publicKey = ks.getCertificate(mAlias).getPublicKey();
            return publicKey;
        } catch (KeyStoreException e) {
            e.printStackTrace();
            return null;
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
            return null;
        } catch (UnrecoverableEntryException e) {
            e.printStackTrace();
            return null;
        }
    }
}
