package com.xdja.simplestorage;

import android.content.Context;
import android.os.Process;
import android.text.TextUtils;
import android.util.Base64;

import com.tencent.mmkv.MMKV;

/**
 * @author zhangxiaolong@xdja.com <br/>
 * @date 2019/9/25 <br/>
 */
public class SimpleStorage {

    private Context context;
    private final String name;
    private final boolean encryptFlag;
    private final static String DEFAULT_ENCRYPT_NAME = "_default_encrypt_" + Process.myUid();

    private SimpleStorage(Context context, String name, boolean encryptFlag) {
        this.context = context;
        this.name = name;
        this.encryptFlag = encryptFlag;
    }

    public void putString(String key, String value) {
        MMKV mmkv = getMmkv();
        mmkv.putString(key, value);
    }

    public void putBytes(String key, byte[] value) {
        MMKV mmkv = getMmkv();
        mmkv.putBytes(key, value);
    }

    public void putInt(String key, int value) {
        MMKV mmkv = getMmkv();
        mmkv.putInt(key, value);
    }

    public String getString(String key, String defaultValue) {
        MMKV mmkv = getMmkv();
        return mmkv.getString(key, defaultValue);
    }

    public byte[] getBytes(String key) {
        MMKV mmkv = getMmkv();
        return mmkv.getBytes(key, null);
    }

    public int getInt(String key, int defaultValue) {
        MMKV mmkv = getMmkv();
        return mmkv.getInt(key, defaultValue);
    }

    public String[] getAllKey() {
        MMKV mmkv = getMmkv();
        return mmkv.allKeys();
    }

    public void remove(String key) {
        MMKV mmkv = getMmkv();
        mmkv.remove(key);
    }

    public void removeAll() {
        MMKV mmkv = getMmkv();
        mmkv.clearAll();
    }

    private MMKV mmkv;

    private MMKV getMmkv() {
        if (mmkv != null) {
            return mmkv;
        }

        init();
        synchronized (SimpleStorage.class) {
            if (!encryptFlag) {
                byte[] key = KeyStoreUtils.getKey(context);
                String keyStr = Base64.encodeToString(key, Base64.NO_WRAP);
                if (TextUtils.isEmpty(name)) {
                    mmkv = MMKV.mmkvWithID(DEFAULT_ENCRYPT_NAME, 1, keyStr);
                } else {
                    mmkv = MMKV.mmkvWithID(name, 1, keyStr);
                }
            } else {
                if (TextUtils.isEmpty(name)) {
                    mmkv = MMKV.defaultMMKV();
                } else {
                    mmkv = MMKV.mmkvWithID(name);
                }
            }
        }
        return mmkv;
    }

    private static boolean initFlag = false;

    private synchronized void init() {
        if (!initFlag) {
            MMKV.initialize(context);
        }
    }


    public static class Builder {
        private String name = "";
        private boolean encryptFlag = false;

        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        public Builder setEncrypt(boolean flag) {
            this.encryptFlag = flag;
            return this;
        }

        public SimpleStorage build(Context context) {
            return new SimpleStorage(context, name, encryptFlag);
        }
    }
}
