package com.xdja.multichip;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.text.TextUtils;

import com.xdja.multichip.utils.GetValueFromProperties;

import java.util.ArrayList;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/6/7 <br/>
 */

public class AccessControlJni {

    /**
     * 可用
     */
    public static final int OK = 0;
    /**
     * 不可用
     */
    public static final int CAN_NOT_USE = -101;


    private Context context;

    public AccessControlJni(Context context) {
        this.context = context;
    }

    public static final String STR_TRUE = "true";
    public static final String STR_FALSE = "false";

    private static ArrayList<String> METHOD_LIST = new ArrayList<>();

    static {
        METHOD_LIST.add("ChangePIN");
        METHOD_LIST.add("ReloadPIN");
    }

    /**
     * 检查包名，是否有权限使用method
     *
     * @param pkgName 包名
     * @param method  方法名
     * @return 0：有权限；其他：无权限
     */
    public int checkByPkgName(String pkgName, String method) {
        try {
            int ret;
            ret = judgeMethodIsNeedCheck(method);
            if (ret == OK) {
                return ret;
            }

            ret = judgeVerifyFlag(context);
            if (ret == OK) {
                return ret;
            }

            ret = judgeIsSystemApp(context, pkgName);
            if (ret == OK) {
                return ret;
            }

            ret = judgeSignatures(context, pkgName);
            if (ret == OK) {
                return ret;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return CAN_NOT_USE;
    }

    /**
     * 判断配置文件中的标志位
     *
     * @param context
     * @return
     */
    private int judgeVerifyFlag(Context context) {
        String verifyPkgFlag = GetValueFromProperties.getValue(context, "config.properties", "verifyPkgFlag");
        if (TextUtils.isEmpty(verifyPkgFlag) || STR_FALSE.equals(verifyPkgFlag)) {
            return OK;
        }
        return CAN_NOT_USE;
    }

    /**
     * 判断方法是否需要接着判断
     *
     * @param method
     * @return
     */
    private int judgeMethodIsNeedCheck(String method) {
        if (!METHOD_LIST.contains(method)) {
            return OK;
        }
        return CAN_NOT_USE;
    }

    /**
     * 判断是否是系统应用
     *
     * @param context
     * @param pkgName
     * @return
     */
    private int judgeIsSystemApp(Context context, String pkgName) {
        PackageManager pm = context.getPackageManager();
        try {
            ApplicationInfo info = pm.getApplicationInfo(pkgName, 0);
            if ((info.flags & ApplicationInfo.FLAG_SYSTEM) != 0) {
                return OK;
            }
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        return CAN_NOT_USE;
    }

    /**
     * 检查签名是否跟本应用一致
     *
     * @param context
     * @param pkgName
     * @return
     */
    private int judgeSignatures(Context context, String pkgName) {
        String selfPkgName = context.getPackageName();
        if (selfPkgName.equals(pkgName)) {
            return OK;
        }
        PackageManager pm = context.getPackageManager();
        if (pm.checkSignatures(pkgName, selfPkgName) == PackageManager.SIGNATURE_MATCH) {
            return OK;
        }
        return CAN_NOT_USE;
    }
}
