package com.xdja.multichip.process.vhsm;

import android.Manifest;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Environment;
import android.text.TextUtils;
import android.util.Pair;

import com.xdja.multi.unitepin.jar.MultiChipUnitePinManager;
import com.xdja.multichip.jniapi.JarMultiJniApiErrorCode;
import com.xdja.multichip.utils.PermissionUtil;

import java.io.File;

/**
 * @author zhangxiaolong@xdja.com <br/>
 * @date 2019/1/15 <br/>
 */
public class DefaultVhsmBean extends VhsmBean {

    private static DefaultVhsmBean instance;
    private Context context;

    private DefaultVhsmBean(Context context) {
        this.context = context;
    }

    /**
     * 得到
     *
     * @param context
     * @return
     */
    public static DefaultVhsmBean make(Context context) {
        if (instance == null) {
            synchronized (DefaultVhsmBean.class) {
                if (instance == null) {
                    instance = new DefaultVhsmBean(context);
                }
            }
        }
        return instance;
    }


    /**
     * 默认卡号
     */
    private String defaultId;
    private String defaultCardId;
    private File defaultVhsmFile;

    @Override
    public String getId() throws VhsmBeanException {
        if (TextUtils.isEmpty(defaultId)) {
            getDefaultIdAndCardIdInCache();
        }

        if (!TextUtils.isEmpty(defaultId)) {
            return defaultId;
        }

        //1.先获取Android_Id
        String id = VhsmUtils.getAndroidId(context);
        if (TextUtils.isEmpty(id)) {
            //2. 如果为空，再获取imei
            id = VhsmUtils.getImei(context);

            //3. 获取mac地址
            if (TextUtils.isEmpty(id)) {
                id = VhsmUtils.getMacAddress(context);
            }
        }
        if (!TextUtils.isEmpty(id)) {
            defaultId = id;
        }
        return defaultId;
    }

    @Override
    public String getCardId() throws VhsmBeanException {
        if (!TextUtils.isEmpty(defaultCardId)) {
            return defaultCardId;
        }
        String id = getId();
        if (TextUtils.isEmpty(id)) {
            return "";
        }
        defaultCardId = VhsmUtils.calcVhsmCardId(id);
        saveDefaultIdAndCardIdInCache(id, defaultCardId);
        return defaultCardId;
    }

    /**
     * 获取默认虚拟卡文件位置
     *
     * @return
     */
    @Override
    public File getVhsmDatFile() throws VhsmBeanException {
        if (defaultVhsmFile != null) {
            return defaultVhsmFile;
        }
        boolean granted = PermissionUtil.isGranted(context, Manifest.permission.WRITE_EXTERNAL_STORAGE);
        if (!granted){
            throw new VhsmBeanException(VhsmUtils.NO_PERMISSION);
        }
        String filePath = Environment.getExternalStorageDirectory().getPath() + "/xdja/vhsm";
        File vhsmDatafileDir = new File(filePath);
        if (!vhsmDatafileDir.exists()) {
            if (!vhsmDatafileDir.mkdirs()) {
                throw new VhsmBeanException(VhsmUtils.NO_PERMISSION);
            }
        }

        String cardId = getCardId();
        if (TextUtils.isEmpty(cardId)) {
            throw new VhsmBeanException("Get cardId is null!");
        }
        String fileName = cardId + ".dat";
        //虚拟卡文件为卡号
        defaultVhsmFile = new File(vhsmDatafileDir, fileName);
        return defaultVhsmFile;
    }

    /**
     * 创建虚拟卡时的默认PIN码
     *
     * @return
     */
    @Override
    public String getVhsmPwd() {
        try {
            Pair<Integer, String> pinPair = MultiChipUnitePinManager.getInstance().getPin(context, getCardId(), 0x11);
            if (pinPair != null && pinPair.first == 0 && TextUtils.isEmpty(pinPair.second)) {
                return pinPair.second;
            }
            pinPair = MultiChipUnitePinManager.getInstance().getPin(context, getCardId(), 0x01);
            if (pinPair != null && pinPair.first == 0 && TextUtils.isEmpty(pinPair.second)) {
                return pinPair.second;
            }
        } catch (VhsmBeanException e) {
            e.printStackTrace();
        }

        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < 6; i++) {
            sb.append("1");
        }
        return sb.toString();
    }

    @Override
    public int handleVhsmBeanException(VhsmBeanException e) {
        if (e.getMessage().equals(VhsmUtils.NO_PERMISSION)) {
            return JarMultiJniApiErrorCode.RET_NO_PERMISSION;
        }
        return JarMultiJniApiErrorCode.RET_EXCEPTION;
    }

    private static final String CACHE_FILE_NAME = "default_vhsm";
    /**
     * 关键字，默认卡号。
     * 有id计算出来的卡号。
     */
    private static final String KEY_String_defaultCardId = "defaultCardId";
    /**
     * 关键字，默认id。
     * 用于计算卡号的id。
     */
    private static final String KEY_String_defaultId = "defaultId";

    /**
     * 获取使用的id和对应的cardId
     *
     * @param cardId
     */
    private synchronized void saveDefaultIdAndCardIdInCache(String id, String cardId) {
        if (TextUtils.isEmpty(cardId) || TextUtils.isEmpty(id)) {
            return;
        }
        SharedPreferences preferences = context.getSharedPreferences(CACHE_FILE_NAME, Context.MODE_PRIVATE);
        SharedPreferences.Editor edit = preferences.edit();
        edit.putString(KEY_String_defaultCardId, cardId);
        edit.putString(KEY_String_defaultId, id);
        edit.apply();

        defaultCardId = cardId;
        defaultId = id;
    }

    /**
     * 获取默认id和卡号
     *
     * @return first是id，second是cardId
     */
    private Pair<String, String> getDefaultIdAndCardIdInCache() {
        SharedPreferences preferences = context.getSharedPreferences(CACHE_FILE_NAME, Context.MODE_PRIVATE);
        String cardId = preferences.getString(KEY_String_defaultCardId, "");
        String id = preferences.getString(KEY_String_defaultId, "");
        if (TextUtils.isEmpty(id) || TextUtils.isEmpty(cardId)) {
            return Pair.create("", "");
        }
        defaultId = id;
        defaultCardId = cardId;
        return Pair.create(id, cardId);
    }

}
