package com.xdja.multichip.process;

import android.os.Bundle;
import android.os.RemoteException;
import android.util.Log;
import android.util.Pair;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.XDJA_SM2_PRIKEY;
import com.xdja.SafeKey.XDJA_SM2_PUBKEY;
import com.xdja.multichip.jniapi.Arithmetic;
import com.xdja.multichip.param.CertBean;
import com.xdja.multichip.utils.ConvertUtil;
import com.xdja.skfapi.EccCipherBlob;
import com.xdja.skfapi.EccPublicKeyBlob;
import com.xdja.skfapi.EnvelopedKeyBlob;
import com.xdja.skfapi.SkfApiCode;

import java.util.Arrays;

import static com.xdja.skfapi.SkfApiCode.ECC_MAX_XCOORDINATE_BITS_LEN;
import static com.xdja.skfapi.SkfApiCode.ECC_MAX_YCOORDINATE_BITS_LEN;

/**
 * @author: zhangxiaolong@xdja.com <br/>
 * date:   2018/4/19 <br/>
 */

public class SuppertImportCert {

    public Bundle importCert(SupperJniApiBinder binder, Bundle bundle) {
        Bundle result = new Bundle();
        Pair<Integer, String> pair = importCertOther(binder, bundle);
        result.putInt(CertBean.KEY_RET_INT, pair.first);
        result.putString(CertBean.KEY_INFO_STRING, pair.second);
        return result;
    }

    /**
     * 导入证书
     *
     * @param binder 操作卡
     * @param bundle bundle中包含{@link CertBean}中KEY_ 字段
     * @return
     * @throws RemoteException
     */
    public Pair<Integer, String> importCertOther(SupperJniApiBinder binder, Bundle bundle) {
        try {
            int ret;
            //判断是否包含必要信息
            if (!bundle.containsKey(CertBean.KEY_VERSION_INT)
                    || !bundle.containsKey(CertBean.KEY_CERTTYPE_INT)
                    || !bundle.containsKey(CertBean.KEY_CERTALG_INT)
                    || !bundle.containsKey(CertBean.KEY_SIGNCERT_BYTEARRAY)
                    || !bundle.containsKey(CertBean.KEY_DATATYPE_INT)
                    || !bundle.containsKey(CertBean.KEY_PIN_STRING)
                    || !bundle.containsKey(CertBean.KEY_ROLE_INT)
                    || !bundle.containsKey(CertBean.KEY_CONTAINERNO_INT)) {
                return Pair.create(CertBean.PARAM_LACK, "");
            }

            int version = bundle.getInt(CertBean.KEY_VERSION_INT);
            if (version != CertBean.VERSION_CUR) {
                return Pair.create(CertBean.NOT_SUPPORT_VERSION, "Service support version is " + CertBean.VERSION_CUR);
            }
            int certType = bundle.getInt(CertBean.KEY_CERTTYPE_INT);
            int certAlg = bundle.getInt(CertBean.KEY_CERTALG_INT);
            if (certType != CertBean.TYPE_SINGLE_SIGN &&
                    certType != CertBean.TYPE_DOUBLE &&
                    certType != CertBean.TYPE_SINGLE_EXCHANGE) {
                return Pair.create(CertBean.PARAM_ERROR, "ImportCert type error: " + certType);
            }
            if (certAlg != CertBean.ALG_RSA && certAlg != CertBean.ALG_SM2) {
                return Pair.create(CertBean.PARAM_ERROR, "ImportCert alg error: " + certAlg);
            }

            String pin = bundle.getString(CertBean.KEY_PIN_STRING);
            int role = bundle.getInt(CertBean.KEY_ROLE_INT);
            ret = binder.VerifyPIN(binder.mHandle, role, pin.getBytes(), pin.length());
            if (ret != 0) {
                return Pair.create(CertBean.PIN_ERROR, "" + ret);
            }
            int containerNo = bundle.getInt(CertBean.KEY_CONTAINERNO_INT);
            String containerName = "content " + containerNo;
            SupperHandleContainer handleContainer = new SupperHandleContainer();
            ret = handleContainer.createContainer(binder, containerNo, containerName);
            if (ret != 0 && ret != JNIAPI.XKR_FILE_EXIST) {
                return Pair.create(CertBean.CREATE_CONTAINER_ERROR, "" + ret);
            }

            // 双证
            if (certType == CertBean.TYPE_DOUBLE) {
                if (certAlg == CertBean.ALG_SM2) {

                    byte[] exchangePriKey;
                    byte[] exchangePubKey;
                    byte[] exchangeCert;
                    byte[] signCert;

                    int dataType = bundle.getInt(CertBean.KEY_DATATYPE_INT);
                    if (dataType == CertBean.DATA_TYPE_SKF) {
                        byte[] envelopedKeyBytes = bundle.getByteArray(CertBean.KEY_ENVELOPEDKEY_BYTEARRAY);
                        if (envelopedKeyBytes == null) {
                            return Pair.create(CertBean.PARAM_LACK, CertBean.KEY_ENVELOPEDKEY_BYTEARRAY + " lost.");
                        }
                        EnvelopedKeyBlob envelopedKey = getBean(envelopedKeyBytes);
                        if (envelopedKey == null) {
                            return Pair.create(CertBean.ANALYSIS_DATA_ERROR, "Analysis EnvelopedKeyBlob error.");
                        }
                        byte[] signPriFid;
                        signPriFid = Arithmetic.getSignPriKeyFid(containerNo);
                        exchangePriKey = getDecPriKey(binder, envelopedKey, signPriFid);
                        if (exchangePriKey == null) {
                            return Pair.create(CertBean.ANALYSIS_DATA_ERROR, "Analysis exchangePriKey error.");
                        }
                        exchangePubKey = getPubKey(envelopedKey);
                        if (exchangePubKey == null) {
                            return Pair.create(CertBean.ANALYSIS_DATA_ERROR, "Analysis exchangePubKey error.");
                        }
                        exchangeCert = bundle.getByteArray(CertBean.KEY_EXCHANGECERT_BYTEARRAY);

                        signCert = bundle.getByteArray(CertBean.KEY_SIGNCERT_BYTEARRAY);
                    } else if (dataType == CertBean.DATA_TYPE_NORMAL) {
                        exchangePriKey = bundle.getByteArray(CertBean.KEY_EXCHANGEPRIKEY_BYTEARRAY);
                        exchangePubKey = bundle.getByteArray(CertBean.KEY_EXCHANGEPUBKEY_BYTEARRAY);
                        exchangeCert = bundle.getByteArray(CertBean.KEY_EXCHANGECERT_BYTEARRAY);
                        signCert = bundle.getByteArray(CertBean.KEY_SIGNCERT_BYTEARRAY);
                    } else {
                        return Pair.create(CertBean.PARAM_ERROR, "ImportCert is double cert,but dataType is neither SKF nor NORMAL!");
                    }
                    Pair<Integer, String> writeCertPair = writeSM2DoubleCert(binder,
                            containerNo,
                            exchangePubKey,
                            exchangePriKey,
                            exchangeCert,
                            signCert);
                    if (writeCertPair.first != 0) {
                        return writeCertPair;
                    }
                } else {
                    return Pair.create(CertBean.NOT_SUPPORT_ALG, "");
                }
            }//单证，签名证书
            else if (certType == CertBean.TYPE_SINGLE_SIGN) {
                byte[] signCert = bundle.getByteArray(CertBean.KEY_SIGNCERT_BYTEARRAY);
                if (certAlg == CertBean.ALG_SM2 || certAlg == CertBean.ALG_RSA) {
                    byte[] certFid = new byte[2];
                    certFid[0] = 0x00;
                    certFid[1] = (byte) (0x28 + 4 + containerNo * 7);
                    ret = binder.WriteCert(binder.mHandle, certFid, signCert, signCert.length);
                    if (ret != 0) {
                        return Pair.create(CertBean.WRITE_SIGN_CERT_ERROR, "" + ret);
                    }
                } else {
                    return Pair.create(CertBean.PARAM_ERROR, "");
                }
            }//单证，交换证书
            else if (certType == CertBean.TYPE_SINGLE_EXCHANGE) {
                byte[] signCert = bundle.getByteArray(CertBean.KEY_SIGNCERT_BYTEARRAY);
                if (certAlg == CertBean.ALG_SM2 || certAlg == CertBean.ALG_RSA) {
                    byte[] certFid = new byte[2];
                    certFid[0] = 0x00;
                    certFid[1] = (byte) (0x28 + 1 + containerNo * 7);
                    ret = binder.WriteCert(binder.mHandle, certFid, signCert, signCert.length);
                    if (ret != 0) {
                        return Pair.create(CertBean.WRITE_EXCHANGE_CERT_ERROR, "" + ret);
                    }
                } else {
                    return Pair.create(CertBean.PARAM_ERROR, "");
                }
            }


            int alg;
            int type;
            if (certAlg == CertBean.ALG_RSA) {
                alg = 0;
            } else {
                alg = 1;
            }
            if (certType == CertBean.TYPE_SINGLE_SIGN) {
                type = 0;
            } else if (certType == CertBean.TYPE_SINGLE_EXCHANGE) {
                type = 1;
            } else {
                type = 2;
            }

            ret = handleContainer.updateContainer(binder, containerNo, alg, type);
            if (ret != 0) {
                return Pair.create(CertBean.UPDATE_CONTAINER_ERROR, "" + ret);
            }
            return Pair.create(0, "");

        } catch (Exception e) {
            e.printStackTrace();
            return Pair.create(CertBean.EXCEPTION, "");
        }
    }

    /**
     * 写SM2双证信息相关
     *
     * @param binder
     * @param containerNo
     * @param exchangePubKey
     * @param exchangePriKey
     * @param exchangeCert
     * @param signCert
     * @return
     * @throws RemoteException
     */
    private Pair<Integer, String> writeSM2DoubleCert(SupperJniApiBinder binder,
                                                     int containerNo,
                                                     byte[] exchangePubKey,
                                                     byte[] exchangePriKey,
                                                     byte[] exchangeCert,
                                                     byte[] signCert) throws RemoteException {
        int ret;
        if (exchangePubKey == null) {
            return Pair.create(CertBean.PARAM_ERROR, CertBean.KEY_EXCHANGEPUBKEY_BYTEARRAY + " lost.");
        }
        if (exchangePriKey == null) {
            return Pair.create(CertBean.PIN_ERROR, CertBean.KEY_EXCHANGEPRIKEY_BYTEARRAY + " lost.");
        }
        if (exchangeCert == null) {
            return Pair.create(CertBean.PARAM_ERROR, CertBean.KEY_EXCHANGECERT_BYTEARRAY + " lost.");
        }
        if (signCert == null) {
            return Pair.create(CertBean.PARAM_ERROR, CertBean.KEY_SIGNCERT_BYTEARRAY + " lost.");
        }

        //交换公钥
        byte[] exchangePubFid = Arithmetic.getExchangePubKeyFid(containerNo);
        XDJA_SM2_PUBKEY pubKey = new XDJA_SM2_PUBKEY();
        System.arraycopy(exchangePubKey, 0, pubKey.x, 0, pubKey.x.length);
        System.arraycopy(exchangePubKey, 32, pubKey.y, 0, pubKey.y.length);
        //写入交换公钥
        ret = binder.WriteSm2PubKey(binder.mHandle, exchangePubFid, pubKey);
        if (ret != 0) {
            return Pair.create(CertBean.WRITE_EXCHANGE_PUBKEY_ERROR, "" + ret);
        }

        //交换私钥
        byte[] exchangePriFid = Arithmetic.getExchangePriKeyFid(containerNo);
        XDJA_SM2_PRIKEY priKey = new XDJA_SM2_PRIKEY();
        if (exchangePriKey.length == JNIAPI.KEY_LEN_SM2 * 2) {
            System.arraycopy(exchangePriKey, JNIAPI.KEY_LEN_SM2, priKey.d, 0, priKey.d.length);
        } else {
            System.arraycopy(exchangePriKey, 0, priKey.d, 0, priKey.d.length);
        }
        //写入交换私钥
        ret = binder.WriteSm2PriKey(binder.mHandle, exchangePriFid, priKey);
        if (ret != 0) {
            return Pair.create(CertBean.WRITE_EXCHANGE_PRIKEY_ERROR, "" + ret);
        }

        byte[] exchangeCertFid = Arithmetic.getExchangeCertFid(containerNo);
        ret = binder.WriteCert(binder.mHandle, exchangeCertFid, exchangeCert, exchangeCert.length);
        if (ret != 0) {
            return Pair.create(CertBean.WRITE_EXCHANGE_CERT_ERROR, "" + ret);
        }

        byte[] signCertFid = Arithmetic.getSignCertFid(containerNo);
        ret = binder.WriteCert(binder.mHandle, signCertFid, signCert, signCert.length);
        if (ret != 0) {
            return Pair.create(CertBean.WRITE_SIGN_CERT_ERROR, "" + ret);
        }
        return Pair.create(0, "");
    }

    /**
     * 得到EnvelopedKeyBlob结构数据
     *
     * @param bytes
     * @return
     */
    private EnvelopedKeyBlob getBean(byte[] bytes) {
        if (bytes == null) {
            return null;
        }
        EnvelopedKeyBlob envelopedKey = new EnvelopedKeyBlob();

        int start;
        int len;

        start = 0;
        len = 4;
        envelopedKey.version = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, start, start + len));

        start = start + len;
        len = 4;
        envelopedKey.ulSymmAlgID = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, start, start + len));

        start = start + len;
        len = 4;
        envelopedKey.ulBits = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, start, start + len));

        start = start + len;
        len = 64;
        envelopedKey.cbEncryptedPriKey = Arrays.copyOfRange(bytes, start, start + len);

        start = start + len;
        len = 4 + (ECC_MAX_XCOORDINATE_BITS_LEN / 8) + (ECC_MAX_YCOORDINATE_BITS_LEN / 8);
        envelopedKey.pubKey = getEccPublicKeyBlob(Arrays.copyOfRange(bytes, start, start + len));

        int tmpStart = start + len + (ECC_MAX_XCOORDINATE_BITS_LEN / 8) + (ECC_MAX_YCOORDINATE_BITS_LEN / 8) + 32;
        int tmpLen = 4;
        int eccCipherLen = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, tmpStart, tmpStart + tmpLen));

        start = start + len;
        len = (ECC_MAX_XCOORDINATE_BITS_LEN / 8) + (ECC_MAX_YCOORDINATE_BITS_LEN / 8) + 32 + 4 + eccCipherLen;
        envelopedKey.eccCipherBlob = getEccCipherBlob(Arrays.copyOfRange(bytes, start, start + len));

        return envelopedKey;
    }

    private EccPublicKeyBlob getEccPublicKeyBlob(byte[] bytes) {
        if (bytes == null) {
            return null;
        }
        EccPublicKeyBlob eccPublicKeyBlob = new EccPublicKeyBlob();
        int start;
        int len;

        start = 0;
        len = 4;
        eccPublicKeyBlob.bitLen = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, start, start + len));

        start = start + len;
        len = ECC_MAX_XCOORDINATE_BITS_LEN / 8;
        eccPublicKeyBlob.xCoordinate = Arrays.copyOfRange(bytes, start, start + len);

        start = start + len;
        len = ECC_MAX_YCOORDINATE_BITS_LEN / 8;
        eccPublicKeyBlob.yCoordinate = Arrays.copyOfRange(bytes, start, start + len);

        return eccPublicKeyBlob;
    }

    private EccCipherBlob getEccCipherBlob(byte[] bytes) {
        if (bytes == null) {
            return null;
        }
        EccCipherBlob eccCipher = new EccCipherBlob();

        int start;
        int len;

        start = 0;
        len = ECC_MAX_XCOORDINATE_BITS_LEN / 8;
        eccCipher.xXCoordinate = Arrays.copyOfRange(bytes, start, start + len);

        start = start + len;
        len = ECC_MAX_YCOORDINATE_BITS_LEN / 8;
        eccCipher.yCoordinate = Arrays.copyOfRange(bytes, start, start + len);

        start = start + len;
        len = 32;
        eccCipher.hash = Arrays.copyOfRange(bytes, start, start + len);

        start = start + len;
        len = 4;
        eccCipher.cipherLen = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, start, start + len));

        start = start + len;
        len = eccCipher.cipherLen;
        byte[] tmp = Arrays.copyOfRange(bytes, start, start + len);
        eccCipher.cipher = new byte[len];
        System.arraycopy(tmp, 0, eccCipher.cipher, 0, len);

        return eccCipher;
    }

    /**
     * 得到EccCipherBlob的byte数组。
     * 注：不包括里面的cipherLen
     *
     * @param eccCipher
     * @return
     */
    private byte[] getEccCipherBytes(EccCipherBlob eccCipher) {
        if (eccCipher == null) {
            return null;
        }
        int len = eccCipher.xXCoordinate.length / 2 +
                eccCipher.yCoordinate.length / 2 +
                eccCipher.hash.length +
                4 +
                eccCipher.cipherLen;
        byte[] result = new byte[len];

        int start = 0;
        len = eccCipher.xXCoordinate.length / 2;
        System.arraycopy(eccCipher.xXCoordinate, len, result, start, len);

        start = start + len;
        len = eccCipher.yCoordinate.length / 2;
        System.arraycopy(eccCipher.yCoordinate, len, result, start, len);

        start = start + len;
        len = eccCipher.hash.length;
        System.arraycopy(eccCipher.hash, 0, result, start, len);

        start = start + len;
        len = 4;
        System.arraycopy(ConvertUtil.intToBytes(eccCipher.cipherLen), 0, result, start, len);

        start = start + len;
        len = eccCipher.cipher.length;
        System.arraycopy(eccCipher.cipher, 0, result, start, len);

        return result;
    }

    /**
     * 得到私钥
     *
     * @param binder       binder
     * @param envelopedKey 数据结构
     * @param signPriFid   签名私钥fid
     * @return
     * @throws RemoteException
     */
    private byte[] getDecPriKey(SupperJniApiBinder binder, EnvelopedKeyBlob envelopedKey, byte[] signPriFid) throws RemoteException {
        // 加过密的对称秘钥
        byte[] eccCipherBytes = getEccCipherBytes(envelopedKey.eccCipherBlob);
        byte[] out = new byte[eccCipherBytes.length];
        int[] outLen = new int[1];
        int ret;
        // 解密得到 对称秘钥
        ret = binder.SM2DecryptGM(binder.mHandle, signPriFid, eccCipherBytes, eccCipherBytes.length, out, outLen);
        if (ret != 0) {
            return null;
        }

        // 解密过的对称秘钥
        byte[] decPriKey = Arrays.copyOf(out, outLen[0]);
        byte[] priKey = new byte[envelopedKey.cbEncryptedPriKey.length];
        //使用对称算法 解密 交换私钥
        if (envelopedKey.ulSymmAlgID == SkfApiCode.SGD_SM1_ECB) {
            ret = binder.SM1KEY(binder.mHandle, decPriKey, envelopedKey.cbEncryptedPriKey, envelopedKey.cbEncryptedPriKey.length, JNIAPI.ECB_DECRYPT, priKey, null);
            if (ret != 0) {
                Log.w("ImportCertLog", "SM1KEY ret = " + ret);
                return null;
            }
        } else if (envelopedKey.ulSymmAlgID == SkfApiCode.SGD_SM4_ECB) {
            ret = binder.SM4KEY(binder.mHandle, decPriKey, envelopedKey.cbEncryptedPriKey, envelopedKey.cbEncryptedPriKey.length, JNIAPI.ECB_DECRYPT, priKey, null);
            if (ret != 0) {
                Log.w("ImportCertLog", "SM4KEY ret = " + ret);
                return null;
            }
        } else {
            return null;
        }
        return priKey;
    }

    private byte[] getPubKey(EnvelopedKeyBlob envelopedKey) {
        byte[] pubKey = new byte[64];
        System.arraycopy(envelopedKey.pubKey.xCoordinate, 32, pubKey, 0, 32);
        System.arraycopy(envelopedKey.pubKey.yCoordinate, 32, pubKey, 32, 32);
        return pubKey;
    }

}
